package ru.yandex.direct.logicprocessor.processors.moderation.asset;

import java.time.Duration;
import java.util.List;
import java.util.Map;

import com.google.common.collect.Range;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.common.db.PpcPropertyNames;
import ru.yandex.direct.core.entity.banner.model.VideoWithModerationInfo;
import ru.yandex.direct.core.entity.creative.model.Creative;
import ru.yandex.direct.core.entity.creative.model.CreativeType;
import ru.yandex.direct.core.entity.creative.repository.CreativeRepository;
import ru.yandex.direct.core.entity.moderation.ModerationOperationModeProvider;
import ru.yandex.direct.core.entity.moderation.model.asset.BannerVideoAdditionModerationRequest;
import ru.yandex.direct.core.entity.moderation.service.sending.BannerVideoAdditionSender;
import ru.yandex.direct.env.NonProductionEnvironment;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.ess.config.moderation.asset.BannerVideoAdditionModerationConfig;
import ru.yandex.direct.ess.logicobjects.moderation.asset.BannerAssetModerationEventsObject;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.logicprocessor.common.EssLogicProcessor;
import ru.yandex.direct.logicprocessor.common.EssLogicProcessorContext;
import ru.yandex.direct.logicprocessor.processors.moderation.BannerAssetSimpleModerationEventsProcessor;
import ru.yandex.direct.logicprocessor.processors.moderation.ClientIdLoadingModerationEventsService;
import ru.yandex.direct.logicprocessor.processors.moderation.ModerationEventWithClientInfo;
import ru.yandex.direct.logicprocessor.processors.moderation.writers.BannerVideoAdditionModerationWriter;
import ru.yandex.direct.logicprocessor.processors.moderation.writers.ModerationWriterMonitoring;

import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.core.entity.creative.repository.CreativeConstants.CPM_AUDIO_LAYOUT_IDS;
import static ru.yandex.direct.core.entity.creative.repository.CreativeConstants.CPM_OVERLAY_LAYOUT_IDS;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_MODERATION;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_0;

@JugglerCheck(
        ttl = @JugglerCheck.Duration(minutes = 5),
        needCheck = ProductionOnly.class,
        tags = {DIRECT_PRIORITY_0, DIRECT_MODERATION})
@JugglerCheck(ttl = @JugglerCheck.Duration(minutes = 10), needCheck = NonProductionEnvironment.class, tags =
        {DIRECT_PRIORITY_0})
@EssLogicProcessor(BannerVideoAdditionModerationConfig.class)
public class BannerVideoAdditionModerationEventsProcessor extends BannerAssetSimpleModerationEventsProcessor
        <BannerAssetModerationEventsObject, VideoWithModerationInfo, BannerVideoAdditionModerationRequest> {

    private static final List<Range<Long>> NON_VIDEO_LAYOUT_RANGES = List.of(
            CPM_AUDIO_LAYOUT_IDS,
            CPM_OVERLAY_LAYOUT_IDS
    );

    private final ClientIdLoadingModerationEventsService clientIdLoadingModerationEventsService;
    private final PpcProperty<Integer> newTransportEnabled;
    private final CreativeRepository creativeRepository;

    @Autowired
    public BannerVideoAdditionModerationEventsProcessor(EssLogicProcessorContext essLogicProcessorContext,
                                                        BannerVideoAdditionSender bannerVideoAdditionSender,
                                                        PpcPropertiesSupport ppcPropertiesSupport,
                                                        ClientIdLoadingModerationEventsService clientIdLoadingModerationEventsService,
                                                        BannerVideoAdditionModerationWriter moderationWriter,
                                                        ModerationWriterMonitoring moderationWriterMonitoring,
                                                        ModerationOperationModeProvider moderationOperationModeProvider,
                                                        CreativeRepository creativeRepository) {
        super(essLogicProcessorContext, bannerVideoAdditionSender, moderationWriter, moderationWriterMonitoring,
                moderationOperationModeProvider);

        this.clientIdLoadingModerationEventsService = clientIdLoadingModerationEventsService;
        this.newTransportEnabled =
                ppcPropertiesSupport.get(PpcPropertyNames.ENABLE_VIDEO_ADDITION_TRANSPORT_NEW_MODERATION,
                        Duration.ofMinutes(1));
        this.creativeRepository = creativeRepository;
    }

    @Override
    protected List<Long> getObjectIds(List<BannerAssetModerationEventsObject> events) {
        List<Long> bids = events.stream()
                .map(BannerAssetModerationEventsObject::getBannerId)
                .collect(toList());
        Map<Long, Creative> creativesByBannerIds = creativeRepository.getCreativesByBannerIds(getShard(), bids);
        return clientIdLoadingModerationEventsService.mapEvents(getShard(), events)
                .stream()
                .filter(this::isNewTransportEnabledForEvent)
                .map(e -> e.getEventsObject().getBannerId())
                .filter(bid -> isVideoAddition(creativesByBannerIds.get(bid)))
                .collect(toList());
    }

    private boolean isNewTransportEnabledForEvent(
            ModerationEventWithClientInfo<BannerAssetModerationEventsObject> event) {
        return event.getClientId() % 100 < newTransportEnabled.getOrDefault(0);
    }

    private static boolean isVideoAddition(Creative creative) {
        if (creative == null ||
                !creative.getType().equals(CreativeType.VIDEO_ADDITION_CREATIVE)) {
            return false;
        }
        boolean isVideoLayout = StreamEx.of(NON_VIDEO_LAYOUT_RANGES)
                .noneMatch(range -> range.contains(creative.getLayoutId()));
        return isVideoLayout;
    }
}
