package ru.yandex.direct.logicprocessor.processors.moderation.banner;

import java.util.List;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;

import ru.yandex.direct.core.entity.moderation.model.BannerModerationMeta;
import ru.yandex.direct.core.entity.moderation.model.BannerModerationResponse;
import ru.yandex.direct.core.entity.moderation.model.ModerationDecision;
import ru.yandex.direct.core.entity.moderation.model.Verdict;
import ru.yandex.direct.core.entity.moderation.repository.bulk_update.BulkUpdateHolder;
import ru.yandex.direct.core.entity.moderation.service.receiving.operations.banners.BulkBannersOp;
import ru.yandex.direct.core.entity.moderation.service.receiving.operations.banners.ResetStatusBsSyncedOp;
import ru.yandex.direct.core.entity.moderation.service.receiving.operations.banners.UpdateStatusModerateAndPostModerateOp;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.env.NonProductionEnvironment;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.ess.config.moderation.banner.PerformanceBannerMainModerationConfig;
import ru.yandex.direct.ess.logicobjects.moderation.banner.BannerModerationEventsObject;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.logicprocessor.common.BaseLogicProcessor;
import ru.yandex.direct.logicprocessor.common.EssLogicProcessor;
import ru.yandex.direct.logicprocessor.common.EssLogicProcessorContext;

import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_MODERATION;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_1;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Новый родитель для смарт-баннеров автопринимается (модерироваться будут только его ассеты)
 * В будущем нужно будет ещё и отправлять его в Модерацию, чтобы Модерация знала про эти баннеры
 * (им это нужно для админки и для привязки ассетов к баннерам)
 */
@JugglerCheck(ttl = @JugglerCheck.Duration(minutes = 5), needCheck = ProductionOnly.class,
        tags = {DIRECT_MODERATION, DIRECT_PRIORITY_1})
@JugglerCheck(ttl = @JugglerCheck.Duration(minutes = 10), needCheck = NonProductionEnvironment.class,
        tags = {DIRECT_PRIORITY_1})
@EssLogicProcessor(PerformanceBannerMainModerationConfig.class)
public class PerformanceBannerMainModerationEventsProcessor extends BaseLogicProcessor<BannerModerationEventsObject> {
    private final ApplicationContext applicationContext;
    private final DslContextProvider dslContextProvider;

    private static final List<Class<? extends BulkBannersOp>> OPERATION_CLASSES = List.of(
            UpdateStatusModerateAndPostModerateOp.class,
            ResetStatusBsSyncedOp.class
    );

    @Autowired
    protected PerformanceBannerMainModerationEventsProcessor(ApplicationContext applicationContext,
                                                             EssLogicProcessorContext essLogicProcessorContext,
                                                             DslContextProvider dslContextProvider) {
        super(essLogicProcessorContext);
        this.applicationContext = applicationContext;
        this.dslContextProvider = dslContextProvider;
    }

    @Override
    public void process(List<BannerModerationEventsObject> banners) {
        List<Long> bannerIds = mapList(banners, BannerModerationEventsObject::getBannerId);
        if (bannerIds.isEmpty()) {
            return;
        }

        dslContextProvider.ppcTransaction(getShard(), configuration -> {
            BulkUpdateHolder bulkUpdateHolder = new BulkUpdateHolder();

            List<BulkBannersOp> operations = mapList(OPERATION_CLASSES, applicationContext::getBean);
            for (var op : operations) {
                for (var banner : banners) {
                    // Для каждого объекта изготавливаем фейковый результат модерации с вердиктом Yes
                    // и применяем его через стандартные обработчики
                    // Чтобы минимизировать количество кастомного кода (не копипастить логику принятия)
                    op.consume(bulkUpdateHolder, createAcceptedResponse(banner));
                }
            }
            for (var op : operations) {
                op.flush(configuration, bulkUpdateHolder);
            }
            bulkUpdateHolder.execute(configuration);
        });
    }

    /**
     * Заполняем минимальное кол-во полей, которые нужны операциям, описанным в {@link #OPERATION_CLASSES}
     */
    private BannerModerationResponse createAcceptedResponse(BannerModerationEventsObject moderatedObject) {
        BannerModerationResponse moderationResponse = new BannerModerationResponse();

        BannerModerationMeta meta = new BannerModerationMeta();
        meta.setBannerId(moderatedObject.getBannerId());
        meta.setAdGroupId(moderatedObject.getAdGroupId());
        moderationResponse.setMeta(meta);

        Verdict verdict = new Verdict();
        verdict.setVerdict(ModerationDecision.Yes);
        moderationResponse.setResult(verdict);

        return moderationResponse;
    }
}
