package ru.yandex.direct.logicprocessor.processors.moderation.banner.support;

import java.time.Duration;
import java.util.List;
import java.util.stream.Collectors;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.config.ConfigurableBeanFactory;
import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.common.db.PpcPropertyNames;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.banner.model.BannerWithAdImageModerationInfo;
import ru.yandex.direct.core.entity.banner.model.BannerWithModerationInfo;
import ru.yandex.direct.core.entity.creative.model.CreativeType;
import ru.yandex.direct.core.entity.moderation.ModerationOperationModeProvider;
import ru.yandex.direct.core.entity.moderation.model.BannerModerationMeta;
import ru.yandex.direct.core.entity.moderation.model.ad_image.AdImageBannerModerationRequest;
import ru.yandex.direct.core.entity.moderation.model.cpm.canvas.CanvasBannerModerationRequest;
import ru.yandex.direct.core.entity.moderation.model.cpm.in_banner.InBannerModerationRequest;
import ru.yandex.direct.core.entity.moderation.model.html5.Html5BannerModerationRequest;
import ru.yandex.direct.core.entity.moderation.service.sending.AdImageBannerSender;
import ru.yandex.direct.core.entity.moderation.service.sending.CpcCanvasBannerSender;
import ru.yandex.direct.core.entity.moderation.service.sending.CpcHtml5BannerSender;
import ru.yandex.direct.core.entity.moderation.service.sending.CpcInBannerSender;
import ru.yandex.direct.core.entity.moderation.service.sending.McBannerSender;
import ru.yandex.direct.core.entity.moderation.service.sending.MobileAdImageBannerSender;
import ru.yandex.direct.core.entity.moderation.service.sending.MobileCanvasBannerSender;
import ru.yandex.direct.core.entity.moderation.service.sending.MobileHtml5BannerSender;
import ru.yandex.direct.dbschema.ppc.enums.BannersBannerType;
import ru.yandex.direct.logicprocessor.processors.moderation.ClientIdLoadingModerationEventsService;
import ru.yandex.direct.logicprocessor.processors.moderation.ModerationEventWithClientInfo;
import ru.yandex.direct.logicprocessor.processors.moderation.ModerationRequestsConsumerBuilder;
import ru.yandex.direct.logicprocessor.processors.moderation.writers.AdImageModerationWriter;
import ru.yandex.direct.logicprocessor.processors.moderation.writers.CanvasModerationWriter;
import ru.yandex.direct.logicprocessor.processors.moderation.writers.Html5ModerationWriter;
import ru.yandex.direct.logicprocessor.processors.moderation.writers.InBannerModerationWriter;
import ru.yandex.direct.logicprocessor.processors.moderation.writers.ModerationWriterMonitoring;
import ru.yandex.monlib.metrics.labels.Labels;

@Component
@Scope(value = ConfigurableBeanFactory.SCOPE_PROTOTYPE)
public class AdImageBannerModerationEventsHandlerProvider extends BannerModerationEventsHandlerProvider {
    private final AdImageBannerSender adImageBannerSender;
    private final CpcCanvasBannerSender canvasBannerSender;
    private final CpcHtml5BannerSender html5BannerSender;
    private final McBannerSender mcBannerSender;
    private final CpcInBannerSender inBannerSender;
    private final MobileAdImageBannerSender mobileAdImageBannerSender;
    private final MobileCanvasBannerSender mobileCanvasBannerSender;
    private final MobileHtml5BannerSender mobileHtml5BannerSender;

    private final AdImageModerationWriter adImageModerationWriter;
    private final CanvasModerationWriter canvasModerationWriter;
    private final Html5ModerationWriter html5ModerationWriter;
    private final InBannerModerationWriter inBannerModerationWriter;

    private final ModerationWriterMonitoring moderationWriterMonitoring;
    private final ModerationOperationModeProvider moderationOperationModeProvider;
    private final ClientIdLoadingModerationEventsService clientIdLoadingModerationEventsService;

    private final PpcProperty<Integer> adImageTransportEnabled;
    private final PpcProperty<Integer> canvasTransportEnabled;
    private final PpcProperty<Integer> html5TransportEnabled;

    @Autowired
    public AdImageBannerModerationEventsHandlerProvider(AdImageBannerSender adImageBannerSender,
                                                        CpcCanvasBannerSender canvasBannerSender,
                                                        CpcHtml5BannerSender html5BannerSender,
                                                        McBannerSender mcBannerSender,
                                                        CpcInBannerSender inBannerSender,
                                                        MobileAdImageBannerSender mobileAdImageBannerSender,
                                                        MobileCanvasBannerSender mobileCanvasBannerSender,
                                                        MobileHtml5BannerSender mobileHtml5BannerSender,
                                                        AdImageModerationWriter adImageModerationWriter,
                                                        CanvasModerationWriter canvasModerationWriter,
                                                        Html5ModerationWriter html5ModerationWriter,
                                                        InBannerModerationWriter inBannerModerationWriter,
                                                        ModerationWriterMonitoring moderationWriterMonitoring,
                                                        ModerationOperationModeProvider moderationOperationModeProvider,
                                                        ClientIdLoadingModerationEventsService clientIdLoadingModerationEventsService,
                                                        PpcPropertiesSupport ppcPropertiesSupport) {
        super(List.of(adImageModerationWriter, canvasModerationWriter, html5ModerationWriter, inBannerModerationWriter));

        this.adImageBannerSender = adImageBannerSender;
        this.canvasBannerSender = canvasBannerSender;
        this.html5BannerSender = html5BannerSender;
        this.mcBannerSender = mcBannerSender;
        this.inBannerSender = inBannerSender;
        this.mobileAdImageBannerSender = mobileAdImageBannerSender;
        this.mobileCanvasBannerSender = mobileCanvasBannerSender;
        this.mobileHtml5BannerSender = mobileHtml5BannerSender;

        this.adImageModerationWriter = adImageModerationWriter;
        this.canvasModerationWriter = canvasModerationWriter;
        this.html5ModerationWriter = html5ModerationWriter;
        this.inBannerModerationWriter = inBannerModerationWriter;
        this.moderationWriterMonitoring = moderationWriterMonitoring;
        this.moderationOperationModeProvider = moderationOperationModeProvider;
        this.clientIdLoadingModerationEventsService = clientIdLoadingModerationEventsService;

        this.adImageTransportEnabled =
                ppcPropertiesSupport.get(PpcPropertyNames.ENABLE_AD_IMAGE_TRANSPORT_NEW_MODERATION,
                        Duration.ofMinutes(1));

        this.canvasTransportEnabled =
                ppcPropertiesSupport.get(PpcPropertyNames.ENABLE_CANVAS_TRANSPORT_NEW_MODERATION,
                        Duration.ofMinutes(1));

        this.html5TransportEnabled =
                ppcPropertiesSupport.get(PpcPropertyNames.ENABLE_HTML5_TRANSPORT_NEW_MODERATION,
                        Duration.ofMinutes(1));
    }

    @Override
    public BannerModerationEventsHandler build(ModerationRequestsConsumerBuilder requestsConsumerBuilder) {
        return new BannerModerationEventsHandler.Builder()
                .addBannerModerationSupport(
                        new BannerModerationSupport.Builder<AdImageBannerModerationRequest,
                                BannerWithAdImageModerationInfo, BannerModerationMeta>()
                                .setBannerType(BannersBannerType.image_ad)
                                .setAdGroupType(AdGroupType.BASE)
                                .hasImage(true)
                                .setSender(adImageBannerSender)
                                .setRequestConsumer(requestsConsumerBuilder.build(adImageModerationWriter, moderationWriterMonitoring,
                                        Labels.of("type", "ad_image")))
                                .setFilter(this::filterByAdImageProperty)
                                .build()
                )
                .addBannerModerationSupport(
                        new BannerModerationSupport.Builder<AdImageBannerModerationRequest,
                                BannerWithAdImageModerationInfo, BannerModerationMeta>()
                                .setBannerType(BannersBannerType.image_ad)
                                .setAdGroupType(AdGroupType.MOBILE_CONTENT)
                                .hasImage(true)
                                .setSender(mobileAdImageBannerSender)
                                .setRequestConsumer(requestsConsumerBuilder.build(adImageModerationWriter, moderationWriterMonitoring,
                                        Labels.of("type", "ad_image")))
                                .setFilter(this::filterByAdImageProperty)
                                .build()
                )
                .addBannerModerationSupport(
                        new BannerModerationSupport.Builder<AdImageBannerModerationRequest,
                                BannerWithAdImageModerationInfo, BannerModerationMeta>()
                                .setBannerType(BannersBannerType.mcbanner)
                                .setAdGroupType(AdGroupType.MCBANNER)
                                .hasImage(true)
                                .setSender(mcBannerSender)
                                .setRequestConsumer(requestsConsumerBuilder.build(adImageModerationWriter, moderationWriterMonitoring,
                                        Labels.of("type", "ad_image")))
                                .setFilter(this::filterByAdImageProperty)
                                .build()
                )
                .addBannerModerationSupport(new BannerModerationSupport.Builder<InBannerModerationRequest,
                        BannerWithModerationInfo, BannerModerationMeta>()
                        .setBannerType(BannersBannerType.image_ad)
                        .setCreativeSupport(CreativeSupport.builder()
                                .withCreativeType(CreativeType.CANVAS)
                                .withLayoutIdsIn(List.of(21L, 22L, 23L, 24L, 25L, 26L, 27L))
                                .build())
                        .setSender(inBannerSender)
                        .setRequestConsumer(requestsConsumerBuilder.build(this.inBannerModerationWriter,
                                moderationWriterMonitoring,
                                Labels.of("type", "in_banner")))
                        .build())
                .addBannerModerationSupport(
                        new BannerModerationSupport.Builder<CanvasBannerModerationRequest, BannerWithModerationInfo,
                                BannerModerationMeta>()
                                .setBannerType(BannersBannerType.image_ad)
                                .setCreativeSupport(CreativeSupport.builder()
                                        .withLayoutIdsNotIn(List.of(21L, 22L, 23L, 24L, 25L, 26L, 27L))
                                        .withCreativeType(CreativeType.CANVAS)
                                        .build())
                                .setSender(canvasBannerSender)
                                .setRequestConsumer(requestsConsumerBuilder.build(canvasModerationWriter, moderationWriterMonitoring,
                                        Labels.of("type", "canvas")))
                                .setFilter(this::filterByCanvasProperty)
                                .build()
                )
                .addBannerModerationSupport(
                        new BannerModerationSupport.Builder<Html5BannerModerationRequest, BannerWithModerationInfo,
                                BannerModerationMeta>()
                                .setBannerType(BannersBannerType.image_ad)
                                .setAdGroupType(AdGroupType.BASE)
                                .setCreativeSupport(CreativeSupport.builder()
                                        .withCreativeType(CreativeType.HTML5_CREATIVE)
                                        .build())
                                .setSender(html5BannerSender)
                                .setRequestConsumer(requestsConsumerBuilder.build(html5ModerationWriter, moderationWriterMonitoring,
                                        Labels.of("type", "html5")))
                                .setFilter(this::filterByHtml5Property)
                                .build()
                )
                .addBannerModerationSupport(
                        new BannerModerationSupport.Builder<CanvasBannerModerationRequest, BannerWithModerationInfo,
                                BannerModerationMeta>()
                                .setBannerType(BannersBannerType.image_ad)
                                .setAdGroupType(AdGroupType.MOBILE_CONTENT)
                                .setCreativeSupport(CreativeSupport.builder()
                                        .withLayoutIdsNotIn(List.of(21L, 22L, 23L, 24L, 25L, 26L, 27L))
                                        .withCreativeType(CreativeType.CANVAS)
                                        .build())
                                .setSender(mobileCanvasBannerSender)
                                .setRequestConsumer(requestsConsumerBuilder.build(canvasModerationWriter, moderationWriterMonitoring,
                                        Labels.of("type", "canvas")))
                                .setFilter(this::filterByCanvasProperty)
                                .build()
                )
                .addBannerModerationSupport(
                        new BannerModerationSupport.Builder<Html5BannerModerationRequest, BannerWithModerationInfo,
                                BannerModerationMeta>()
                                .setBannerType(BannersBannerType.image_ad)
                                .setAdGroupType(AdGroupType.MOBILE_CONTENT)
                                .setCreativeSupport(CreativeSupport.builder()
                                        .withCreativeType(CreativeType.HTML5_CREATIVE)
                                        .build())
                                .setSender(mobileHtml5BannerSender)
                                .setRequestConsumer(requestsConsumerBuilder.build(html5ModerationWriter, moderationWriterMonitoring,
                                        Labels.of("type", "html5")))
                                .setFilter(this::filterByHtml5Property)
                                .build()
                )
                .setModerationOperationModeProvider(moderationOperationModeProvider)
                .build();
    }

    private List<BannerModerationEventsWithInfo> filterByCanvasProperty(int shard,
                                                                        List<BannerModerationEventsWithInfo> events) {
        return clientIdLoadingModerationEventsService.mapEvents(shard, events).stream()
                .filter(e -> e.getClientId() % 100 < canvasTransportEnabled.getOrDefault(0))
                .map(ModerationEventWithClientInfo::getEventsObject)
                .collect(Collectors.toList());
    }

    private List<BannerModerationEventsWithInfo> filterByHtml5Property(int shard,
                                                                       List<BannerModerationEventsWithInfo> events) {
        return clientIdLoadingModerationEventsService.mapEvents(shard, events).stream()
                .filter(e -> e.getClientId() % 100 < html5TransportEnabled.getOrDefault(0))
                .map(ModerationEventWithClientInfo::getEventsObject)
                .collect(Collectors.toList());
    }

    private List<BannerModerationEventsWithInfo> filterByAdImageProperty(int shard,
                                                                         List<BannerModerationEventsWithInfo> events) {
        return clientIdLoadingModerationEventsService.mapEvents(shard, events).stream()
                .filter(e -> e.getClientId() % 100 < adImageTransportEnabled.getOrDefault(0))
                .map(ModerationEventWithClientInfo::getEventsObject)
                .collect(Collectors.toList());
    }
}
