package ru.yandex.direct.logicprocessor.processors.moderation.banner.support;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.core.entity.banner.model.ModerationableBanner;
import ru.yandex.direct.core.entity.moderation.ModerationOperationModeProvider;
import ru.yandex.direct.core.entity.moderation.model.BannerModerationMeta;
import ru.yandex.direct.core.entity.moderation.model.BannerModerationRequest;
import ru.yandex.direct.core.entity.moderation.model.BaseBannerModerationData;
import ru.yandex.direct.core.entity.moderation.model.ModerationRequest;
import ru.yandex.direct.core.entity.moderation.model.ModerationWorkflow;
import ru.yandex.direct.ess.logicobjects.moderation.banner.BannerModerationEventsObject;

import static com.google.common.base.Preconditions.checkArgument;
import static ru.yandex.direct.core.entity.moderation.ModerationOperationMode.RESTRICTED;
import static ru.yandex.direct.logicprocessor.processors.moderation.ModerationRequestUtils.getIdToEssTagMap;
import static ru.yandex.direct.logicprocessor.processors.moderation.ModerationRequestUtils.getIdToEventTimeMap;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

public class BannerModerationEventsHandler {
    private static final Logger logger = LoggerFactory.getLogger(BannerModerationEventsHandler.class);

    private List<BannerModerationSupport<? extends ModerationRequest,
            ? extends ModerationableBanner, ? extends BannerModerationMeta>> moderationSupports;

    private final ModerationOperationModeProvider moderationOperationModeProvider;

    private BannerModerationEventsHandler(List<BannerModerationSupport<? extends ModerationRequest,
            ? extends ModerationableBanner, ? extends BannerModerationMeta>> moderationSupports,
                                          ModerationOperationModeProvider moderationOperationModeProvider) {
        this.moderationSupports = moderationSupports;
        this.moderationOperationModeProvider = moderationOperationModeProvider;
    }

    private boolean filterByAdGroupType(BannerModerationSupport moderationSupport,
                                        BannerModerationEventsWithInfo ev) {
        return ev.getAdGroupType() != null &&
                (moderationSupport.getAdGroupTypes() == null
                || moderationSupport.getAdGroupTypes().contains(ev.getAdGroupType()));
    }

    private boolean filterByCampaignType(BannerModerationSupport moderationSupport,
                                         BannerModerationEventsWithInfo ev) {
        return moderationSupport.getCampaignType() == null
                || moderationSupport.getCampaignType().equals(ev.getCampaignType());
    }

    private boolean filterByBannerType(BannerModerationSupport moderationSupport,
                                       BannerModerationEventsWithInfo ev) {
        return moderationSupport.getBannerType().equals(ev.getObject().getBannersBannerType());
    }

    public void handleObjects(int shard, List<BannerModerationEventsWithInfo> eventsWithInfoList) {
        // true, если баннер должен быть отправлен в ограниченном режиме, без изменений статусов
        Function<BannerModerationEventsWithInfo, Boolean> restrictedModeClassifier =
                event -> Boolean.TRUE.equals(event.getObject().getWasCopied()) ||
                        Boolean.TRUE.equals(event.getObject().getFlagsOnly());
        Map<Boolean, List<BannerModerationEventsWithInfo>> groupedByModeBanners = StreamEx.of(eventsWithInfoList)
                .mapToEntry(restrictedModeClassifier, Function.identity())
                .grouping();

        // Order is important, because each of next calls makes its own read from the database,
        // so only the last call gets the latest object version.
        handleBannersInRestrictedMode(shard, groupedByModeBanners.getOrDefault(Boolean.TRUE, List.of()));
        filterAndSendBannersToModerate(shard, groupedByModeBanners.getOrDefault(Boolean.FALSE, List.of()));
    }

    private void handleBannersInRestrictedMode(int shard,
                                               List<BannerModerationEventsWithInfo> eventsWithInfoList) {
        try {
            this.moderationOperationModeProvider.forceMode(RESTRICTED);

            // Если поменялись только флаги, то модерация хочет получать такие объекты без изменения версии.
            // Достаточно проверить одно событие, т.к. этот код будет в конечном счете вызываться
            // из ModerationFlagsEventsProcessor, где все события про флаги.
            if (!eventsWithInfoList.isEmpty() && Boolean.TRUE.equals(eventsWithInfoList.get(0).getObject().getFlagsOnly())) {
                this.moderationOperationModeProvider.setImmutableVersionMode();
            }
            filterAndSendBannersToModerate(shard, eventsWithInfoList);
        } finally {
            this.moderationOperationModeProvider.disableForcedMode();
            this.moderationOperationModeProvider.disableImmutableVersionMode();
        }
    }

    public void filterAndSendBannersToModerate(int shard, List<BannerModerationEventsWithInfo> eventsWithInfoList) {
        Set<Long> bids = StreamEx.of(eventsWithInfoList)
                .map(BannerModerationEventsWithInfo::getObject)
                .map(BannerModerationEventsObject::getBannerId)
                .collect(Collectors.toSet());

        logger.info("{} tasks to banner moderation fetched. Making requests.", eventsWithInfoList.size());
        logger.info("Sending with bids: {}", StreamEx.of(bids).joining(","));

        moderationSupports.forEach(support -> getBannerModerationSupportConsumer(shard, support, eventsWithInfoList));
    }

    private boolean isEventSuitable(BannerModerationSupport support, BannerModerationEventsWithInfo event) {
        return filterByAdGroupType(support, event)
                && filterByBannerType(support, event)
                && filterByCreativeInfo(support, event)
                && filterByHasImage(support, event)
                && filterByCampaignType(support, event);
    }

    public List<BannerModerationSupport<?, ?, ?>> findSuitableSupportsForEvent(BannerModerationEventsWithInfo event) {
        return filterList(moderationSupports, support -> isEventSuitable(support, event));
    }

    private void getBannerModerationSupportConsumer(int shard, BannerModerationSupport<? extends ModerationRequest,
            ? extends ModerationableBanner, ? extends BannerModerationMeta> support,
                                                    List<BannerModerationEventsWithInfo> eventsWithInfoList) {

        List<BannerModerationEventsWithInfo> strategyEvents =
                filterList(eventsWithInfoList, event -> isEventSuitable(support, event));

        if (support.getFilter() != null) {
            strategyEvents = support.getFilter().apply(shard, strategyEvents);
        }

        if (strategyEvents.isEmpty()) {
            return;
        }

        sendRequests(shard, support, mapList(strategyEvents, BannerModerationEventsWithInfo::getObject));
    }

    private <D extends BannerModerationRequest<K, ? extends BaseBannerModerationData>, T extends ModerationableBanner,
            K extends BannerModerationMeta>
    void sendRequests(int shard, BannerModerationSupport<D, T, K> support,
                      List<BannerModerationEventsObject> events) {
        List<Long> bids = mapList(events, BannerModerationEventsObject::getBannerId);

        Map<Long, Long> bannerIdToEventTimeMap =
                getIdToEventTimeMap(events, BannerModerationEventsObject::getBannerId);
        var bannerIdToEssTagMap = getIdToEssTagMap(events, BannerModerationEventsObject::getBannerId);
        Map<Long, Boolean> bannerFlagsOnlyMap = getBannerFlagsOnlyMap(events);

        support.getSender().send(shard, bids,
                e -> bannerIdToEventTimeMap.get(e.getId()),
                e -> bannerIdToEssTagMap.get(e.getId()),
                e -> Boolean.TRUE.equals(bannerFlagsOnlyMap.get(e.getId())) ? ModerationWorkflow.USER_FLAGS : null,
                support.getRequestConsumer());
    }

    private static Map<Long, Boolean> getBannerFlagsOnlyMap(List<BannerModerationEventsObject> events) {
        return StreamEx.of(events)
                .mapToEntry(BannerModerationEventsObject::getBannerId, BannerModerationEventsObject::getFlagsOnly)
                .distinctKeys()
                .toMap();
    }

    private boolean filterByHasImage(BannerModerationSupport support,
                                     BannerModerationEventsWithInfo ev) {
        return support.hasImage() == null || support.hasImage() == ev.isHasImage();
    }

    private boolean filterByCreativeInfo(BannerModerationSupport moderationSupport,
                                         BannerModerationEventsWithInfo event) {

        if (moderationSupport.getCreativeSupport() == null) {
            return true;
        }

        List<BannerModerationEventsWithInfo> filtered = new ArrayList<>();
        CreativeSupport creativeSupport = moderationSupport.getCreativeSupport();

        if (event.getCreativeInfo() == null) {
            return false;
        }

        if (creativeSupport.isExpandedCreative() != null && creativeSupport.isExpandedCreative() != event.getCreativeInfo().isExpandedCreative()) {
            return false;
        }

        if (creativeSupport.getCreativeType() != null && creativeSupport.getCreativeType() != event.getCreativeInfo().getCreativeType()) {
            return false;
        }

        if (creativeSupport.getLayoutIds() != null
                && (event.getCreativeInfo().getLayoutId() == null ||
                !creativeSupport.getLayoutIds().contains(event.getCreativeInfo().getLayoutId()))
        ) {
            return false;
        }

        if (creativeSupport.getDeniedLayoutIds() != null
                && event.getCreativeInfo().getLayoutId() != null
                && creativeSupport.getDeniedLayoutIds().contains(event.getCreativeInfo().getLayoutId())
        ) {
            return false;
        }
        return true;
    }

    public static class Builder {

        private List<BannerModerationSupport<? extends ModerationRequest,
                ? extends ModerationableBanner, ? extends BannerModerationMeta>> moderationSupports = new ArrayList<>();

        public Builder setModerationOperationModeProvider(ModerationOperationModeProvider moderationOperationModeProvider) {
            this.moderationOperationModeProvider = moderationOperationModeProvider;
            return this;
        }

        private ModerationOperationModeProvider moderationOperationModeProvider;

        public Builder addBannerModerationSupport(BannerModerationSupport<? extends ModerationRequest,
                ? extends ModerationableBanner, ? extends BannerModerationMeta> bannerModerationSupport) {
            moderationSupports.add(bannerModerationSupport);
            return this;
        }

        public BannerModerationEventsHandler build() {
            checkArgument(!moderationSupports.isEmpty(), "moderationSupports is empty");
            checkArgument(moderationOperationModeProvider != null, "moderationOperationModeProvider is null");
            checkBannerModerationSupportConsistency();
            return new BannerModerationEventsHandler(moderationSupports, moderationOperationModeProvider);
        }

        private void checkBannerModerationSupportConsistency() {
            var uniqSupports = Set.copyOf(moderationSupports);
            checkArgument(moderationSupports.size() == uniqSupports.size(), "supports must be different");
        }
    }
}
