package ru.yandex.direct.logicprocessor.processors.moderation.banner.support;

import java.util.List;
import java.util.Objects;
import java.util.function.Consumer;

import javax.annotation.Nullable;

import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.banner.model.ModerationableBanner;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.moderation.model.BannerModerationMeta;
import ru.yandex.direct.core.entity.moderation.model.BannerModerationRequest;
import ru.yandex.direct.core.entity.moderation.model.BaseBannerModerationData;
import ru.yandex.direct.core.entity.moderation.service.sending.BaseBannerSender;
import ru.yandex.direct.dbschema.ppc.enums.BannersBannerType;

import static com.google.common.base.Preconditions.checkArgument;

/**
 * Определяет какие баннеры будут отправлены в модерацию
 * Пример: new BannerModerationSupport.Builder<T>()
 * .setRequestSender(requestSender)
 * .build(); - значит выполнить для всех баннеров requestSender.makeRequest
 * <p>
 * new BannerModerationSupport.Builder<T>()
 * .setRequestSender(requestSender)
 * .setAdGroupType(AdGroupType.CPM_YNDX_FRONTPAGE)
 * .build(); - фильтруем баннеры привязанные только к CPM_YNDX_FRONTPAGE группе
 */
public class BannerModerationSupport<D extends BannerModerationRequest<K, ? extends BaseBannerModerationData>,
        T extends ModerationableBanner, K extends BannerModerationMeta> {

    private final BannersBannerType bannerType;
    private final List<AdGroupType> adGroupTypes;
    private final CampaignType campaignType;
    private final CreativeSupport creativeSupport;
    private final Boolean hasImage;
    private final BaseBannerSender<D, T, K> sender;
    private final Consumer<List<D>> requestConsumer;
    private final BannerModerationEventsFilter filter;

    private BannerModerationSupport(BannersBannerType bannerType, List<AdGroupType> adGroupTypes, CampaignType campaignType,
                                    CreativeSupport creativeSupport, Boolean hasImage,
                                    BaseBannerSender<D, T, K> sender,
                                    Consumer<List<D>> requestConsumer,
                                    BannerModerationEventsFilter filter) {
        this.bannerType = bannerType;
        this.adGroupTypes = adGroupTypes;
        this.campaignType = campaignType;
        this.creativeSupport = creativeSupport;
        this.hasImage = hasImage;
        this.sender = sender;
        this.requestConsumer = requestConsumer;
        this.filter = filter;
    }

    public static <D extends BannerModerationRequest<K, ? extends BaseBannerModerationData>,
            T extends ModerationableBanner, K extends BannerModerationMeta>
    BannerModerationSupport<D, T, K> forAllEvents(BannersBannerType bannerType,
                                                  BaseBannerSender<D, T, K> requestSender, Consumer<List<D>> consumer) {
        return new BannerModerationSupport.Builder<D, T, K>()
                .setBannerType(bannerType)
                .setSender(requestSender)
                .setRequestConsumer(consumer)
                .build();
    }

    @Nullable
    BannerModerationEventsFilter getFilter() {
        return filter;
    }

    @Nullable
    CreativeSupport getCreativeSupport() {
        return creativeSupport;
    }

    @Nullable
    Boolean hasImage() {
        return hasImage;
    }

    @Nullable
    List<AdGroupType> getAdGroupTypes() {
        return adGroupTypes;
    }

    @Nullable
    CampaignType getCampaignType() {
        return campaignType;
    }

    BannersBannerType getBannerType() {
        return bannerType;
    }

    Consumer<List<D>> getRequestConsumer() {
        return requestConsumer;
    }

    public BaseBannerSender<D, T, K> getSender() {
        return sender;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        BannerModerationSupport<?, ?, ?> that = (BannerModerationSupport<?, ?, ?>) o;
        return bannerType == that.bannerType &&
                Objects.equals(adGroupTypes, that.adGroupTypes) &&
                campaignType == that.campaignType &&
                Objects.equals(creativeSupport, that.creativeSupport) &&
                Objects.equals(hasImage, that.hasImage);
    }

    @Override
    public int hashCode() {
        return Objects.hash(bannerType, adGroupTypes, campaignType, creativeSupport, hasImage);
    }

    public static class Builder<D extends BannerModerationRequest<K, ? extends BaseBannerModerationData>,
            T extends ModerationableBanner, K extends BannerModerationMeta> {
        private List<AdGroupType> adGroupTypes;
        private CampaignType campaignType;
        private CreativeSupport creativeType;
        private Boolean hasImage;
        private BaseBannerSender<D, T, K> sender;
        private Consumer<List<D>> requestConsumer;
        private BannerModerationEventsFilter filter;
        private BannersBannerType bannerType;


        /**
         * фильтрация по типу группы. значение null - фильтр не применять
         */
        public Builder<D, T, K> setAdGroupTypes(List<AdGroupType> adGroupTypes) {
            this.adGroupTypes = adGroupTypes;
            return this;
        }

        public Builder<D, T, K> setAdGroupType(AdGroupType adGroupType) {
            this.adGroupTypes = List.of(adGroupType);
            return this;
        }

        public Builder<D, T, K> setCreativeSupport(CreativeSupport creativeType) {
            this.creativeType = creativeType;
            return this;
        }

        /**
         * фильтрация по типу кампании. значение null - фильтр не применять
         */
        public Builder<D, T, K> setCampaignType(CampaignType campaignType) {
            this.campaignType = campaignType;
            return this;
        }


        public Builder<D, T, K> setSender(BaseBannerSender<D, T, K> sender) {
            this.sender = sender;
            return this;
        }

        public Builder<D, T, K> setRequestConsumer(Consumer<List<D>> requestConsumer) {
            this.requestConsumer = requestConsumer;
            return this;
        }

        public Builder<D, T, K> setFilter(BannerModerationEventsFilter filter) {
            this.filter = filter;
            return this;
        }

        public BannerModerationSupport<D, T, K> build() {
            checkArgument(bannerType != null, "bannerType is null");
            checkArgument(sender != null, "sender is null");
            checkArgument(requestConsumer != null, "requestConsumer is null");

            return new BannerModerationSupport<>(bannerType, adGroupTypes, campaignType, creativeType, hasImage,
                    sender, requestConsumer, filter);
        }

        public Builder<D, T, K> hasImage(Boolean hasImage) {
            this.hasImage = hasImage;
            return this;
        }

        public Builder<D, T, K> setBannerType(BannersBannerType bannerType) {
            this.bannerType = bannerType;
            return this;
        }
    }
}
