package ru.yandex.direct.logicprocessor.processors.moderation.banner.support;

import java.time.Duration;
import java.util.List;
import java.util.stream.Collectors;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.config.ConfigurableBeanFactory;
import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.common.db.PpcPropertyNames;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.banner.model.BannerWithModerationInfo;
import ru.yandex.direct.core.entity.creative.model.CreativeType;
import ru.yandex.direct.core.entity.moderation.ModerationOperationModeProvider;
import ru.yandex.direct.core.entity.moderation.model.BannerModerationMeta;
import ru.yandex.direct.core.entity.moderation.model.cpm.canvas.CanvasBannerModerationRequest;
import ru.yandex.direct.core.entity.moderation.model.cpm.frontpage.CpmYndxFrontpageBannerModerationRequest;
import ru.yandex.direct.core.entity.moderation.model.cpm.frontpage.FixcpmYndxFrontpageBannerModerationRequest;
import ru.yandex.direct.core.entity.moderation.model.cpm.in_banner.InBannerModerationRequest;
import ru.yandex.direct.core.entity.moderation.model.cpm.video.CpmVideoBannerModerationRequest;
import ru.yandex.direct.core.entity.moderation.model.html5.Html5BannerModerationRequest;
import ru.yandex.direct.core.entity.moderation.service.sending.CpcVideoBannerSender;
import ru.yandex.direct.core.entity.moderation.service.sending.CpmCanvasBannerSender;
import ru.yandex.direct.core.entity.moderation.service.sending.CpmFrontpageBannerSender;
import ru.yandex.direct.core.entity.moderation.service.sending.CpmHtml5BannerSender;
import ru.yandex.direct.core.entity.moderation.service.sending.CpmInBannerSender;
import ru.yandex.direct.core.entity.moderation.service.sending.CpmVideoBannerSender;
import ru.yandex.direct.core.entity.moderation.service.sending.FixcpmFrontpageBannerSender;
import ru.yandex.direct.core.entity.moderation.service.sending.GeoproductHtml5BannerSender;
import ru.yandex.direct.core.entity.moderation.service.sending.MobileCpcVideoBannerSender;
import ru.yandex.direct.dbschema.ppc.enums.BannersBannerType;
import ru.yandex.direct.logicprocessor.processors.moderation.ClientIdLoadingModerationEventsService;
import ru.yandex.direct.logicprocessor.processors.moderation.ModerationEventWithClientInfo;
import ru.yandex.direct.logicprocessor.processors.moderation.ModerationRequestsConsumerBuilder;
import ru.yandex.direct.logicprocessor.processors.moderation.writers.CanvasModerationWriter;
import ru.yandex.direct.logicprocessor.processors.moderation.writers.CpmVideoModerationWriter;
import ru.yandex.direct.logicprocessor.processors.moderation.writers.FrontpageExtendedModerationWriter;
import ru.yandex.direct.logicprocessor.processors.moderation.writers.FrontpageModerationWriter;
import ru.yandex.direct.logicprocessor.processors.moderation.writers.Html5ModerationWriter;
import ru.yandex.direct.logicprocessor.processors.moderation.writers.InBannerModerationWriter;
import ru.yandex.direct.logicprocessor.processors.moderation.writers.ModerationWriterMonitoring;
import ru.yandex.monlib.metrics.labels.Labels;

@Component
@Scope(value = ConfigurableBeanFactory.SCOPE_PROTOTYPE)
public class CpmBannerModerationEventsHandlerProvider extends BannerModerationEventsHandlerProvider {
    private final FixcpmFrontpageBannerSender fixcpmFrontpageBannerSender;
    private final CpmFrontpageBannerSender cpmFrontpageBannerSender;
    private final CpmCanvasBannerSender canvasBannerSender;
    private final CpmHtml5BannerSender html5BannerSender;
    private final GeoproductHtml5BannerSender geoproductHtml5BannerSender;
    private final CpmVideoBannerSender cpmVideoBannerSender;
    private final CpmInBannerSender inBannerSender;
    private final CpcVideoBannerSender cpcVideoBannerSender;
    private final MobileCpcVideoBannerSender mobileCpcVideoBannerSender;

    private final FrontpageModerationWriter frontpageWriter;
    private final FrontpageExtendedModerationWriter frontpageExtendedWriter;
    private final CpmVideoModerationWriter cpmVideoWriter;
    private final CanvasModerationWriter canvasWriter;
    private final Html5ModerationWriter html5Writer;
    private final InBannerModerationWriter inBannerModerationWriter;

    private final ModerationWriterMonitoring moderationWriterMonitoring;
    private final ModerationOperationModeProvider moderationOperationModeProvider;
    private final ClientIdLoadingModerationEventsService clientIdLoadingModerationEventsService;

    private final PpcProperty<Integer> canvasTransportEnabled;
    private final PpcProperty<Integer> html5TransportEnabled;
    private final PpcProperty<Integer> cpcVideoTransportEnabled;

    @Autowired
    public CpmBannerModerationEventsHandlerProvider(FixcpmFrontpageBannerSender fixcpmFrontpageBannerSender,
                                                    CpmFrontpageBannerSender cpmFrontpageBannerSender,
                                                    CpmCanvasBannerSender canvasBannerSender,
                                                    CpmHtml5BannerSender html5BannerSender,
                                                    GeoproductHtml5BannerSender geoproductHtml5BannerSender,
                                                    CpmVideoBannerSender cpmVideoBannerSender,
                                                    CpmInBannerSender inBannerSender,
                                                    CpcVideoBannerSender cpcVideoBannerSender,
                                                    MobileCpcVideoBannerSender mobileCpcVideoBannerSender,
                                                    FrontpageModerationWriter frontpageWriter,
                                                    FrontpageExtendedModerationWriter frontpageExtendedWriter,
                                                    CpmVideoModerationWriter cpmVideoWriter,
                                                    CanvasModerationWriter canvasWriter,
                                                    Html5ModerationWriter html5Writer,
                                                    InBannerModerationWriter inBannerModerationWriter,
                                                    ModerationWriterMonitoring moderationWriterMonitoring,
                                                    ModerationOperationModeProvider moderationOperationModeProvider,
                                                    ClientIdLoadingModerationEventsService clientIdLoadingModerationEventsService,
                                                    PpcPropertiesSupport ppcPropertiesSupport) {
        super(List.of(frontpageWriter, frontpageExtendedWriter, cpmVideoWriter, canvasWriter,
                html5Writer, inBannerModerationWriter));

        this.fixcpmFrontpageBannerSender = fixcpmFrontpageBannerSender;
        this.cpmFrontpageBannerSender = cpmFrontpageBannerSender;
        this.canvasBannerSender = canvasBannerSender;
        this.html5BannerSender = html5BannerSender;
        this.geoproductHtml5BannerSender = geoproductHtml5BannerSender;
        this.cpmVideoBannerSender = cpmVideoBannerSender;
        this.inBannerSender = inBannerSender;
        this.cpcVideoBannerSender = cpcVideoBannerSender;
        this.mobileCpcVideoBannerSender = mobileCpcVideoBannerSender;

        this.frontpageWriter = frontpageWriter;
        this.frontpageExtendedWriter = frontpageExtendedWriter;
        this.cpmVideoWriter = cpmVideoWriter;
        this.canvasWriter = canvasWriter;
        this.html5Writer = html5Writer;
        this.inBannerModerationWriter = inBannerModerationWriter;
        this.moderationWriterMonitoring = moderationWriterMonitoring;
        this.moderationOperationModeProvider = moderationOperationModeProvider;
        this.clientIdLoadingModerationEventsService = clientIdLoadingModerationEventsService;

        this.canvasTransportEnabled =
                ppcPropertiesSupport.get(PpcPropertyNames.ENABLE_CANVAS_TRANSPORT_NEW_MODERATION,
                        Duration.ofMinutes(1));
        this.html5TransportEnabled =
                ppcPropertiesSupport.get(PpcPropertyNames.ENABLE_HTML5_TRANSPORT_NEW_MODERATION,
                        Duration.ofMinutes(1));
        this.cpcVideoTransportEnabled =
                ppcPropertiesSupport.get(PpcPropertyNames.ENABLE_CPC_VIDEO_TRANSPORT_NEW_MODERATION,
                        Duration.ofMinutes(1));
    }

    @Override
    public BannerModerationEventsHandler build(ModerationRequestsConsumerBuilder requestsConsumerBuilder) {
        return new BannerModerationEventsHandler.Builder()
                .addBannerModerationSupport(new BannerModerationSupport.Builder<FixcpmYndxFrontpageBannerModerationRequest,
                        BannerWithModerationInfo, BannerModerationMeta>()
                        .setBannerType(BannersBannerType.cpm_banner)
                        .setAdGroupType(AdGroupType.CPM_YNDX_FRONTPAGE)
                        .setCreativeSupport(CreativeSupport.builder().withIsExtendedCreative(true).build())
                        .setSender(fixcpmFrontpageBannerSender)
                        .setRequestConsumer(requestsConsumerBuilder.build(frontpageExtendedWriter, moderationWriterMonitoring,
                                Labels.of("type", "expanded_frontpage")))
                        .build())

                .addBannerModerationSupport(new BannerModerationSupport.Builder<CpmYndxFrontpageBannerModerationRequest,
                        BannerWithModerationInfo, BannerModerationMeta>()
                        .setBannerType(BannersBannerType.cpm_banner)
                        .setAdGroupType(AdGroupType.CPM_YNDX_FRONTPAGE)
                        .setCreativeSupport(CreativeSupport.builder().withIsExtendedCreative(false).build())
                        .setSender(cpmFrontpageBannerSender)
                        .setRequestConsumer(requestsConsumerBuilder.build(frontpageWriter, moderationWriterMonitoring,
                                Labels.of("type", "frontpage")))
                        .build())

                .addBannerModerationSupport(new BannerModerationSupport.Builder<CpmVideoBannerModerationRequest,
                        BannerWithModerationInfo, BannerModerationMeta>()
                        .setBannerType(BannersBannerType.cpm_banner)
                        .setAdGroupType(AdGroupType.CPM_VIDEO)
                        .setSender(cpmVideoBannerSender)
                        .setRequestConsumer(requestsConsumerBuilder.build(cpmVideoWriter, moderationWriterMonitoring,
                                Labels.of("type", "cpm_video")))
                        .build()
                )

                .addBannerModerationSupport(new BannerModerationSupport.Builder<InBannerModerationRequest,
                        BannerWithModerationInfo, BannerModerationMeta>()
                        .setBannerType(BannersBannerType.cpm_banner)
                        .setAdGroupType(AdGroupType.CPM_BANNER)
                        .setCreativeSupport(CreativeSupport.builder()
                                .withCreativeType(CreativeType.CANVAS)
                                .withLayoutIdsIn(List.of(21L, 22L, 23L, 24L, 25L, 26L, 27L))
                                .build())
                        .setSender(inBannerSender)
                        .setRequestConsumer(requestsConsumerBuilder.build(inBannerModerationWriter, moderationWriterMonitoring,
                                Labels.of("type", "in_banner")))
                        .build())

                .addBannerModerationSupport(new BannerModerationSupport.Builder<CanvasBannerModerationRequest,
                        BannerWithModerationInfo, BannerModerationMeta>()
                        .setBannerType(BannersBannerType.cpm_banner)
                        .setAdGroupType(AdGroupType.CPM_BANNER)
                        .setCreativeSupport(CreativeSupport.builder()
                                .withCreativeType(CreativeType.CANVAS)
                                .withLayoutIdsNotIn(List.of(21L, 22L, 23L, 24L, 25L, 26L, 27L))
                                .build())
                        .setFilter(this::filterByCanvasProperty)
                        .setSender(canvasBannerSender)
                        .setRequestConsumer(requestsConsumerBuilder.build(canvasWriter, moderationWriterMonitoring,
                                Labels.of("type", "canvas")))
                        .build())

                .addBannerModerationSupport(new BannerModerationSupport.Builder<Html5BannerModerationRequest,
                        BannerWithModerationInfo, BannerModerationMeta>()
                        .setBannerType(BannersBannerType.cpm_banner)
                        .setAdGroupType(AdGroupType.CPM_BANNER)
                        .setCreativeSupport(CreativeSupport.builder()
                                .withCreativeType(CreativeType.HTML5_CREATIVE)
                                .build())
                        .setSender(html5BannerSender)
                        .setRequestConsumer(requestsConsumerBuilder.build(html5Writer, moderationWriterMonitoring,
                                Labels.of("type", "html5")))
                        .setFilter(this::filterByHtml5Property)
                        .build())

                .addBannerModerationSupport(new BannerModerationSupport.Builder<Html5BannerModerationRequest,
                        BannerWithModerationInfo, BannerModerationMeta>()
                        .setBannerType(BannersBannerType.cpm_banner)
                        .setAdGroupType(AdGroupType.CPM_GEOPRODUCT)
                        .setCreativeSupport(CreativeSupport.builder()
                                .withCreativeType(CreativeType.HTML5_CREATIVE)
                                .build())
                        .setSender(geoproductHtml5BannerSender)
                        .setRequestConsumer(requestsConsumerBuilder.build(html5Writer, moderationWriterMonitoring,
                                Labels.of("type", "cpm_geoproduct")))
                        .setFilter(this::filterByHtml5Property)
                        .build())

                .addBannerModerationSupport(new BannerModerationSupport.Builder<CpmVideoBannerModerationRequest,
                        BannerWithModerationInfo, BannerModerationMeta>()
                        .setBannerType(BannersBannerType.cpc_video)
                        .setAdGroupType(AdGroupType.BASE)
                        .setSender(cpcVideoBannerSender)
                        .setRequestConsumer(requestsConsumerBuilder.build(cpmVideoWriter, moderationWriterMonitoring,
                                Labels.of("type", "cpc_video")))
                        .setFilter(this::filterByCpcVideoProperty)
                        .build())

                .addBannerModerationSupport(new BannerModerationSupport.Builder<CpmVideoBannerModerationRequest,
                        BannerWithModerationInfo, BannerModerationMeta>()
                        .setBannerType(BannersBannerType.cpc_video)
                        .setAdGroupType(AdGroupType.MOBILE_CONTENT)
                        .setSender(mobileCpcVideoBannerSender)
                        .setRequestConsumer(requestsConsumerBuilder.build(cpmVideoWriter, moderationWriterMonitoring,
                                Labels.of("type", "mobile_cpc_video")))
                        .setFilter(this::filterByCpcVideoProperty)
                        .build())
                .setModerationOperationModeProvider(moderationOperationModeProvider)
                .build();
    }

    private List<BannerModerationEventsWithInfo> filterByCanvasProperty(int shard,
                                                                        List<BannerModerationEventsWithInfo> events) {
        return clientIdLoadingModerationEventsService.mapEvents(shard, events).stream()
                .filter(e -> e.getClientId() % 100 < canvasTransportEnabled.getOrDefault(0))
                .map(ModerationEventWithClientInfo::getEventsObject)
                .collect(Collectors.toList());
    }

    private List<BannerModerationEventsWithInfo> filterByHtml5Property(int shard,
                                                                       List<BannerModerationEventsWithInfo> events) {
        return clientIdLoadingModerationEventsService.mapEvents(shard, events).stream()
                .filter(e -> e.getClientId() % 100 < html5TransportEnabled.getOrDefault(0))
                .map(ModerationEventWithClientInfo::getEventsObject)
                .collect(Collectors.toList());
    }

    private List<BannerModerationEventsWithInfo> filterByCpcVideoProperty(Integer shard,
                                                                          List<BannerModerationEventsWithInfo> events) {
        return clientIdLoadingModerationEventsService.mapEvents(shard, events).stream()
                .filter(e -> e.getClientId() % 100 < cpcVideoTransportEnabled.getOrDefault(0))
                .map(ModerationEventWithClientInfo::getEventsObject)
                .collect(Collectors.toList());
    }
}
