package ru.yandex.direct.logicprocessor.processors.moderation.banner.support;

import java.time.Duration;
import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.config.ConfigurableBeanFactory;
import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.common.db.PpcPropertyNames;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.banner.model.BannerWithTextAndImageModerationInfo;
import ru.yandex.direct.core.entity.moderation.ModerationOperationModeProvider;
import ru.yandex.direct.core.entity.moderation.model.BannerModerationMeta;
import ru.yandex.direct.core.entity.moderation.model.text.TextBannerModerationRequest;
import ru.yandex.direct.core.entity.moderation.service.sending.DynamicBannerSender;
import ru.yandex.direct.dbschema.ppc.enums.BannersBannerType;
import ru.yandex.direct.logicprocessor.processors.moderation.ClientIdLoadingModerationEventsService;
import ru.yandex.direct.logicprocessor.processors.moderation.ModerationEventWithClientInfo;
import ru.yandex.direct.logicprocessor.processors.moderation.ModerationRequestsConsumerBuilder;
import ru.yandex.direct.logicprocessor.processors.moderation.writers.DynamicModerationWriter;
import ru.yandex.direct.logicprocessor.processors.moderation.writers.ModerationWriterMonitoring;
import ru.yandex.monlib.metrics.labels.Labels;

@Component
@Scope(value = ConfigurableBeanFactory.SCOPE_PROTOTYPE)
public class DynamicBannerModerationEventsHandlerProvider extends BannerModerationEventsHandlerProvider {
    private final DynamicBannerSender sender;
    private final DynamicModerationWriter writer;

    private final ModerationWriterMonitoring moderationWriterMonitoring;
    private final ModerationOperationModeProvider moderationOperationModeProvider;
    private final ClientIdLoadingModerationEventsService clientIdLoadingModerationEventsService;

    private final PpcProperty<Integer> dynamicBannerTransportEnabled;

    @Autowired
    public DynamicBannerModerationEventsHandlerProvider(DynamicBannerSender sender,
                                                        DynamicModerationWriter writer,
                                                        ModerationWriterMonitoring moderationWriterMonitoring,
                                                        ModerationOperationModeProvider moderationOperationModeProvider,
                                                        PpcPropertiesSupport ppcPropertiesSupport,
                                                        ClientIdLoadingModerationEventsService clientIdLoadingModerationEventsService) {
        super(Collections.singletonList(writer));

        this.sender = sender;
        this.writer = writer;
        this.moderationWriterMonitoring = moderationWriterMonitoring;
        this.moderationOperationModeProvider = moderationOperationModeProvider;
        this.clientIdLoadingModerationEventsService = clientIdLoadingModerationEventsService;

        this.dynamicBannerTransportEnabled =
                ppcPropertiesSupport.get(PpcPropertyNames.ENABLE_DYNAMIC_TRANSPORT_NEW_MODERATION,
                        Duration.ofMinutes(1));
    }

    @Override
    public BannerModerationEventsHandler build(ModerationRequestsConsumerBuilder requestsConsumerBuilder) {
        return new BannerModerationEventsHandler.Builder()
                .addBannerModerationSupport(
                        new BannerModerationSupport.Builder<TextBannerModerationRequest,
                                BannerWithTextAndImageModerationInfo, BannerModerationMeta>()
                                .setBannerType(BannersBannerType.dynamic)
                                .setSender(sender)
                                .setAdGroupTypes(List.of(AdGroupType.DYNAMIC, AdGroupType.BASE))
                                .setRequestConsumer(requestsConsumerBuilder.build(writer, moderationWriterMonitoring,
                                        Labels.of("type", "dynamic")))
                                .setFilter(this::filterByProperty)
                                .build()
                )
                .setModerationOperationModeProvider(moderationOperationModeProvider)
                .build();
    }

    private List<BannerModerationEventsWithInfo> filterByProperty(Integer shard,
                                                                  List<BannerModerationEventsWithInfo> events) {
        return clientIdLoadingModerationEventsService.mapEvents(shard, events).stream()
                .filter(e -> e.getClientId() % 100 < dynamicBannerTransportEnabled.getOrDefault(0))
                .map(ModerationEventWithClientInfo::getEventsObject)
                .collect(Collectors.toList());
    }
}
