package ru.yandex.direct.logicprocessor.processors.moderation.banner.support;

import java.time.Duration;
import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.config.ConfigurableBeanFactory;
import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.common.db.PpcPropertyNames;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.banner.model.BannerWithTextAndImageModerationInfo;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.moderation.ModerationOperationModeProvider;
import ru.yandex.direct.core.entity.moderation.model.BannerModerationMeta;
import ru.yandex.direct.core.entity.moderation.model.text.TextBannerModerationRequest;
import ru.yandex.direct.core.entity.moderation.service.sending.MobileContentBannerSender;
import ru.yandex.direct.dbschema.ppc.enums.BannersBannerType;
import ru.yandex.direct.logicprocessor.processors.moderation.ClientIdLoadingModerationEventsService;
import ru.yandex.direct.logicprocessor.processors.moderation.ModerationEventWithClientInfo;
import ru.yandex.direct.logicprocessor.processors.moderation.ModerationRequestsConsumerBuilder;
import ru.yandex.direct.logicprocessor.processors.moderation.writers.MobileContentModerationWriter;
import ru.yandex.direct.logicprocessor.processors.moderation.writers.ModerationWriterMonitoring;
import ru.yandex.monlib.metrics.labels.Labels;

@Component
@Scope(value = ConfigurableBeanFactory.SCOPE_PROTOTYPE)
public class MobileContentBannerModerationEventsHandlerProvider extends BannerModerationEventsHandlerProvider {
    private final MobileContentBannerSender sender;
    private final MobileContentModerationWriter writer;
    private final ModerationWriterMonitoring moderationWriterMonitoring;
    private final ModerationOperationModeProvider moderationOperationModeProvider;
    private final ClientIdLoadingModerationEventsService clientIdLoadingModerationEventsService;

    private final PpcProperty<Integer> mobileContentTransportEnabled;

    @Autowired
    public MobileContentBannerModerationEventsHandlerProvider(MobileContentBannerSender sender,
                                                              MobileContentModerationWriter writer,
                                                              ModerationWriterMonitoring moderationWriterMonitoring,
                                                              ModerationOperationModeProvider moderationOperationModeProvider,
                                                              PpcPropertiesSupport ppcPropertiesSupport,
                                                              ClientIdLoadingModerationEventsService clientIdLoadingModerationEventsService) {
        super(Collections.singletonList(writer));

        this.sender = sender;
        this.writer = writer;
        this.moderationWriterMonitoring = moderationWriterMonitoring;
        this.moderationOperationModeProvider = moderationOperationModeProvider;
        this.clientIdLoadingModerationEventsService = clientIdLoadingModerationEventsService;

        this.mobileContentTransportEnabled =
                ppcPropertiesSupport.get(PpcPropertyNames.ENABLE_MOBILE_CONTENT_TRANSPORT_NEW_MODERATION,
                        Duration.ofMinutes(1));
    }

    @Override
    public BannerModerationEventsHandler build(ModerationRequestsConsumerBuilder requestsConsumerBuilder) {
        return new BannerModerationEventsHandler.Builder()
                .addBannerModerationSupport(
                        new BannerModerationSupport.Builder<TextBannerModerationRequest,
                                BannerWithTextAndImageModerationInfo, BannerModerationMeta>()
                                .setBannerType(BannersBannerType.mobile_content)
                                .setSender(sender)
                                .setAdGroupType(AdGroupType.MOBILE_CONTENT)
                                .setCampaignType(CampaignType.MOBILE_CONTENT)
                                .setRequestConsumer(requestsConsumerBuilder.build(writer, moderationWriterMonitoring,
                                        Labels.of("type", "mobile_content")))
                                .setFilter(this::filterByMobileContentProperty)
                                .build()
                )
                .setModerationOperationModeProvider(moderationOperationModeProvider)
                .build();
    }

    private List<BannerModerationEventsWithInfo> filterByMobileContentProperty(int shard, List<BannerModerationEventsWithInfo> events) {
        return clientIdLoadingModerationEventsService.mapEvents(shard, events).stream()
                .filter(e -> e.getClientId() % 100 < mobileContentTransportEnabled.getOrDefault(0))
                .map(ModerationEventWithClientInfo::getEventsObject)
                .collect(Collectors.toList());
    }
}
