package ru.yandex.direct.logicprocessor.processors.moderation.bannerstorage;

import java.util.List;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.bannerstorage.client.model.Creative;
import ru.yandex.direct.bannerstorage.client.model.File;
import ru.yandex.direct.bannerstorage.client.model.Template;
import ru.yandex.direct.core.entity.moderation.repository.sending.BannerstorageCreativesSendingRepository;
import ru.yandex.direct.core.entity.moderation.service.sending.bannerstorage.BannerstorageCreativesSender;
import ru.yandex.direct.core.entity.moderation.service.sending.bannerstorage.BannerstorageCreativesService;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.env.NonProductionEnvironment;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.ess.config.moderation.banner.BannerstorageCreativeModerationConfig;
import ru.yandex.direct.ess.logicobjects.moderation.bannerstorage.BannerstorageCreativeModerationEventsObject;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.logicprocessor.common.EssLogicProcessor;
import ru.yandex.direct.logicprocessor.common.EssLogicProcessorContext;
import ru.yandex.direct.logicprocessor.processors.moderation.BaseModerationEventsProcessor;
import ru.yandex.direct.logicprocessor.processors.moderation.writers.BannerstorageCreativesModerationWriter;
import ru.yandex.direct.logicprocessor.processors.moderation.writers.ModerationRequestsWriter;
import ru.yandex.direct.logicprocessor.processors.moderation.writers.ModerationWriterMonitoring;
import ru.yandex.direct.tracing.Trace;

import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_MODERATION;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_0;
import static ru.yandex.direct.logicprocessor.processors.moderation.ModerationRequestUtils.getIdToEssTagMap;
import static ru.yandex.direct.logicprocessor.processors.moderation.ModerationRequestUtils.getIdToEventTimeMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@JugglerCheck(
        ttl = @JugglerCheck.Duration(minutes = 5),
        needCheck = ProductionOnly.class,
        tags = {DIRECT_PRIORITY_0, DIRECT_MODERATION})
@JugglerCheck(ttl = @JugglerCheck.Duration(minutes = 10), needCheck = NonProductionEnvironment.class, tags =
        {DIRECT_PRIORITY_0})
@EssLogicProcessor(BannerstorageCreativeModerationConfig.class)
public class BannerstorageCreativesModerationEventsProcessor extends BaseModerationEventsProcessor<
        BannerstorageCreativeModerationEventsObject> {

    private static final Logger logger = LoggerFactory.getLogger(BannerstorageCreativesModerationEventsProcessor.class);

    private final BannerstorageCreativesService bannerstorageCreativesService;
    private final BannerstorageCreativesModerationWriter moderationWriter;
    private final ModerationWriterMonitoring moderationWriterMonitoring;
    private final BannerstorageCreativesSendingRepository sendingRepository;
    private final DslContextProvider dslContextProvider;

    @Autowired
    public BannerstorageCreativesModerationEventsProcessor(
            DslContextProvider dslContextProvider,
            EssLogicProcessorContext essLogicProcessorContext,
            BannerstorageCreativesModerationWriter moderationWriter,
            ModerationWriterMonitoring moderationWriterMonitoring,
            BannerstorageCreativesSendingRepository sendingRepository,
            BannerstorageCreativesService bannerstorageCreativesService) {
        super(essLogicProcessorContext);

        this.moderationWriter = moderationWriter;
        this.moderationWriterMonitoring = moderationWriterMonitoring;
        this.sendingRepository = sendingRepository;
        this.dslContextProvider = dslContextProvider;
        this.bannerstorageCreativesService = bannerstorageCreativesService;
    }

    @Override
    protected void processObjects(List<BannerstorageCreativeModerationEventsObject> events) {
        try (var profile = Trace.current().profile(
                "bannerstorage_creatives_moderation_requests.make", String.valueOf(getShard()))) {

            var idsToProcess = mapList(events, BannerstorageCreativeModerationEventsObject::getCreativeId)
                    .stream()
                    .distinct()
                    .collect(toList());
            logger.info("Taken {} creative events to process: {}", idsToProcess.size(), idsToProcess);

            // Подготовим креативы для отправки
            var preparedCreatives = bannerstorageCreativesService.prepareCreatives(getShard(), idsToProcess);

            if (preparedCreatives.getCreatives().isEmpty()) {
                logger.info("No creatives to send, finish iteration");
                return;
            }

            // Отправляем в модерацию с созданием новой версии отправки в таблице
            // bannerstorage_creatives_moderation_versions
            sendToModeration(
                    events,
                    preparedCreatives.getCreatives(),
                    preparedCreatives.getTemplatesMap(),
                    preparedCreatives.getFilesMap(),
                    preparedCreatives.getPerfCreativesMap()
            );
        }
    }

    private void sendToModeration(List<BannerstorageCreativeModerationEventsObject> eventsToProcess,
                                  List<Creative> creatives,
                                  Map<Integer, Template> templatesMap,
                                  Map<Integer, File> filesMap,
                                  Map<Long, ru.yandex.direct.core.entity.creative.model.Creative> perfCreativesMap) {
        logger.info("Starting to send {} objects", creatives.size());
        Map<Long, Long> creativeIdToEventTimeMap =
                getIdToEventTimeMap(eventsToProcess, BannerstorageCreativeModerationEventsObject::getCreativeId);
        var creativeIdToEssTagMap =
                getIdToEssTagMap(eventsToProcess, BannerstorageCreativeModerationEventsObject::getCreativeId);
        var sender = new BannerstorageCreativesSender(
                dslContextProvider,
                sendingRepository,
                creatives,
                templatesMap,
                filesMap,
                perfCreativesMap
        );
        sender.send(
                getShard(),
                creatives.stream().map(c -> c.getId().longValue()).collect(toList()),
                (o) -> creativeIdToEventTimeMap.get(o.getCreativeId()),
                o -> creativeIdToEssTagMap.get(o.getCreativeId()),
                makeRequestsConsumer(moderationWriter, moderationWriterMonitoring)
        );
        logger.info("Objects sent");
    }

    @Override
    protected List<? extends ModerationRequestsWriter> getWriters() {
        return List.of(moderationWriter);
    }
}
