package ru.yandex.direct.logicprocessor.processors.moderation.callout;

import java.time.Duration;
import java.util.List;
import java.util.Map;

import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.common.db.PpcPropertyNames;
import ru.yandex.direct.core.entity.addition.callout.repository.CalloutRepository;
import ru.yandex.direct.core.entity.moderation.ModerationOperationMode;
import ru.yandex.direct.core.entity.moderation.ModerationOperationModeProvider;
import ru.yandex.direct.core.entity.moderation.service.sending.CalloutSender;
import ru.yandex.direct.env.NonProductionEnvironment;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.ess.config.moderation.callout.CalloutModerationConfig;
import ru.yandex.direct.ess.logicobjects.moderation.callout.CalloutModerationEventsObject;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.logicprocessor.common.EssLogicProcessor;
import ru.yandex.direct.logicprocessor.common.EssLogicProcessorContext;
import ru.yandex.direct.logicprocessor.processors.moderation.BaseModerationEventsProcessor;
import ru.yandex.direct.logicprocessor.processors.moderation.writers.CalloutModerationWriter;
import ru.yandex.direct.logicprocessor.processors.moderation.writers.ModerationRequestsWriter;
import ru.yandex.direct.logicprocessor.processors.moderation.writers.ModerationWriterMonitoring;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceProfile;

import static java.util.Collections.emptyList;
import static java.util.function.Function.identity;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_MODERATION;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_1;
import static ru.yandex.direct.logicprocessor.processors.moderation.ModerationRequestUtils.getIdToEssTagMap;
import static ru.yandex.direct.logicprocessor.processors.moderation.ModerationRequestUtils.getIdToEventTimeMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@JugglerCheck(
        ttl = @JugglerCheck.Duration(minutes = 5),
        needCheck = ProductionOnly.class,
        tags = {DIRECT_PRIORITY_1, DIRECT_MODERATION})
@JugglerCheck(ttl = @JugglerCheck.Duration(minutes = 10), needCheck = NonProductionEnvironment.class, tags =
        {DIRECT_PRIORITY_1})
@EssLogicProcessor(CalloutModerationConfig.class)
public class CalloutModerationEventsProcessor extends BaseModerationEventsProcessor<CalloutModerationEventsObject> {

    private final CalloutModerationWriter moderationWriter;
    private final CalloutSender sender;
    private final ModerationWriterMonitoring moderationWriterMonitoring;
    private final ModerationOperationModeProvider moderationOperationModeProvider;
    private final CalloutRepository calloutRepository;

    private final PpcProperty<Integer> newTransportEnabled;

    @Autowired
    public CalloutModerationEventsProcessor(
            EssLogicProcessorContext essLogicProcessorContext,
            CalloutSender sender,
            CalloutModerationWriter moderationWriter,
            ModerationWriterMonitoring moderationWriterMonitoring,
            ModerationOperationModeProvider moderationOperationModeProvider,
            CalloutRepository calloutRepository,
            PpcPropertiesSupport ppcPropertiesSupport) {
        super(essLogicProcessorContext);
        this.sender = sender;
        this.moderationWriter = moderationWriter;
        this.moderationWriterMonitoring = moderationWriterMonitoring;
        this.moderationOperationModeProvider = moderationOperationModeProvider;
        this.calloutRepository = calloutRepository;

        this.newTransportEnabled = ppcPropertiesSupport.get(PpcPropertyNames.ENABLE_CALLOUT_TRANSPORT_NEW_MODERATION,
                Duration.ofMinutes(1));
    }

    @Override
    protected List<? extends ModerationRequestsWriter> getWriters() {
        return List.of(moderationWriter);
    }

    @Override
    protected void processObjects(List<CalloutModerationEventsObject> events) {
        try (TraceProfile profile = Trace.current().profile(
                sender.typeName() + "callout_moderation_requests.make", String.valueOf(getShard()))) {

            List<CalloutModerationEventsObject> filteredEvents = filterCalloutEventsForProcessing(events);

            Map<Boolean, List<CalloutModerationEventsObject>> restrictedModeToCalloutEventsMap =
                    StreamEx.of(filteredEvents)
                            .mapToEntry(CalloutModerationEventsObject::getWasCopied, identity())
                            .grouping();

            processObjectsInRestrictedMode(restrictedModeToCalloutEventsMap.getOrDefault(Boolean.TRUE, emptyList()));
            processObjectsInternal(restrictedModeToCalloutEventsMap.getOrDefault(Boolean.FALSE, emptyList()));
        }
    }

    protected List<CalloutModerationEventsObject> filterCalloutEventsForProcessing(
            List<CalloutModerationEventsObject> events) {
        List<Long> calloutIds = mapList(events, CalloutModerationEventsObject::getCalloutId);
        Map<Long, Long> calloutIdToClientId = calloutRepository.getClientIdsByCalloutIds(getShard(), calloutIds);

        return StreamEx.of(events)
                .mapToEntry(CalloutModerationEventsObject::getCalloutId, identity())
                .filterKeys(calloutIdToClientId::containsKey)
                .filterKeys(calloutId -> isNewTransportEnabledForClient(calloutIdToClientId.get(calloutId)))
                .values()
                .toList();
    }

    private boolean isNewTransportEnabledForClient(Long clientId) {
        return clientId % 100 < newTransportEnabled.getOrDefault(0);
    }

    protected void processObjectsInRestrictedMode(List<CalloutModerationEventsObject> events) {
        try {
            moderationOperationModeProvider.forceMode(ModerationOperationMode.RESTRICTED);
            processObjectsInternal(events);
        } finally {
            moderationOperationModeProvider.disableForcedMode();
        }
    }

    protected void processObjectsInternal(List<CalloutModerationEventsObject> events) {
        if (events.isEmpty()) {
            return;
        }

        List<Long> calloutIds = mapList(events, CalloutModerationEventsObject::getCalloutId);

        Map<Long, Long> idToEventTimeMap =
                getIdToEventTimeMap(events, CalloutModerationEventsObject::getCalloutId);

        Map<Long, String> idToEssTagMap =
                getIdToEssTagMap(events, CalloutModerationEventsObject::getCalloutId);

        sender.send(getShard(), calloutIds,
                (o) -> idToEventTimeMap.get(o.getCalloutId()),
                (o) -> idToEssTagMap.get(o.getCalloutId()),
                makeRequestsConsumer(moderationWriter, moderationWriterMonitoring));
    }
}
