package ru.yandex.direct.logicprocessor.processors.moderation.campaign;

import java.time.Duration;
import java.util.List;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.common.db.PpcPropertyNames;
import ru.yandex.direct.env.NonProductionEnvironment;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.ess.config.moderation.campaign.CampaignModerationConfig;
import ru.yandex.direct.ess.logicobjects.moderation.campaign.CampaignModerationEvent;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.logicprocessor.common.EssLogicProcessor;
import ru.yandex.direct.logicprocessor.common.EssLogicProcessorContext;
import ru.yandex.direct.logicprocessor.processors.moderation.BaseModerationEventsProcessor;
import ru.yandex.direct.logicprocessor.processors.moderation.ModerationRequestsConsumerBuilder;
import ru.yandex.direct.logicprocessor.processors.moderation.campaign.copy.CampaignCopyModerationHandler;
import ru.yandex.direct.logicprocessor.processors.moderation.writers.ModerationRequestsWriter;

import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_MODERATION;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_1;

@JugglerCheck(
        ttl = @JugglerCheck.Duration(minutes = 5),
        needCheck = ProductionOnly.class,
        tags = {DIRECT_PRIORITY_1, DIRECT_MODERATION})
@JugglerCheck(ttl = @JugglerCheck.Duration(minutes = 10), needCheck = NonProductionEnvironment.class, tags = {DIRECT_PRIORITY_1})
@EssLogicProcessor(CampaignModerationConfig.class)
public class CampaignModerationEventsProcessor extends BaseModerationEventsProcessor<CampaignModerationEvent> {
    private static final Logger logger = LoggerFactory.getLogger(CampaignModerationEventsProcessor.class);

    private final CampaignCopyModerationHandler campaignCopyModerationHandler;
    private final CampaignModerationHandler campaignModerationHandler;
    private final ModerationRequestsConsumerBuilder moderationRequestsConsumerBuilder;

    private final PpcProperty<Integer> newTransportEnabled;

    @Autowired
    public CampaignModerationEventsProcessor(
            EssLogicProcessorContext essLogicProcessorContext,
            CampaignCopyModerationHandler campaignCopyModerationHandler,
            CampaignModerationHandler campaignModerationHandler,
            PpcPropertiesSupport ppcPropertiesSupport) {
        super(essLogicProcessorContext);

        this.campaignCopyModerationHandler = campaignCopyModerationHandler;
        this.campaignModerationHandler = campaignModerationHandler;
        this.moderationRequestsConsumerBuilder = createModerationRequestsConsumerBuilder();

        this.newTransportEnabled = ppcPropertiesSupport.get(PpcPropertyNames.ENABLE_CAMPAIGN_TRANSPORT_NEW_MODERATION,
                Duration.ofMinutes(1));
    }

    @Override
    protected List<? extends ModerationRequestsWriter> getWriters() {
        return List.of(campaignCopyModerationHandler.getWriter(), campaignModerationHandler.getWriter());
    }

    @Override
    protected void processObjects(List<CampaignModerationEvent> events) {
        var copyEvents = events.stream()
                .filter(e -> e.getFromCampaignId() != null && e.getFromCampaignId() != 0)
                .collect(Collectors.toList());
        if (!copyEvents.isEmpty()) {
            campaignCopyModerationHandler.process(copyEvents, getShard(), moderationRequestsConsumerBuilder);
        }

        var moderationEvents = events.stream()
                .filter(e -> e.getCampaignId() != null && e.getClientId() != null)
                .filter(e -> isNewTransportEnabledForClient(e.getClientId()))
                .collect(Collectors.toList());
        if (!moderationEvents.isEmpty()) {
            campaignModerationHandler.process(moderationEvents, getShard(), moderationRequestsConsumerBuilder);
        }
    }

    private boolean isNewTransportEnabledForClient(Long clientId) {
        return clientId % 100 < newTransportEnabled.getOrDefault(0);
    }
}
