package ru.yandex.direct.logicprocessor.processors.moderation.campaign;

import java.time.Duration;
import java.util.List;
import java.util.Map;
import java.util.function.Function;

import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.common.db.PpcPropertyNames;
import ru.yandex.direct.core.entity.moderation.ModerationOperationMode;
import ru.yandex.direct.core.entity.moderation.ModerationOperationModeProvider;
import ru.yandex.direct.core.entity.moderation.service.sending.CampaignSender;
import ru.yandex.direct.ess.logicobjects.moderation.campaign.CampaignModerationEvent;
import ru.yandex.direct.logicprocessor.processors.moderation.ModerationRequestsConsumerBuilder;
import ru.yandex.direct.logicprocessor.processors.moderation.writers.ModerationWriterMonitoring;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceProfile;

import static java.util.Collections.emptyList;
import static ru.yandex.direct.logicprocessor.processors.moderation.ModerationRequestUtils.getIdToEssTagMap;
import static ru.yandex.direct.logicprocessor.processors.moderation.ModerationRequestUtils.getIdToEventTimeMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Component
public class CampaignModerationHandler {
    private final CampaignSender campaignSender;
    private final CampaignModerationWriter campaignModerationWriter;
    private final ModerationWriterMonitoring moderationWriterMonitoring;
    private final ModerationOperationModeProvider moderationOperationModeProvider;

    private final PpcProperty<Integer> updateStatusEnabled;

    @Autowired
    CampaignModerationHandler(CampaignSender campaignSender,
                              CampaignModerationWriter campaignModerationWriter,
                              ModerationWriterMonitoring moderationWriterMonitoring,
                              ModerationOperationModeProvider moderationOperationModeProvider,
                              PpcPropertiesSupport ppcPropertiesSupport) {
        this.campaignSender = campaignSender;
        this.campaignModerationWriter = campaignModerationWriter;
        this.moderationWriterMonitoring = moderationWriterMonitoring;
        this.moderationOperationModeProvider = moderationOperationModeProvider;

        this.updateStatusEnabled = ppcPropertiesSupport.get(PpcPropertyNames.CAMPAIGN_TRANSPORT_NEW_MODERATION_MODIFY_STATUS,
                Duration.ofMinutes(1));
    }

    public void process(List<CampaignModerationEvent> events,
                        int shard,
                        ModerationRequestsConsumerBuilder moderationRequestsConsumerBuilder) {
        try (TraceProfile profile = Trace.current().profile(
                "moderation_campaign-requests.make", String.valueOf(shard))) {

            // RESTRICTED - режим, в котором транспорт не обновляет статусы модерации объектов.
            // Делим список кампаний по признаку "разрешено обновление статусов",
            // кампании, для которых обновление запрещено обрабатываем в ограниченном режиме.
            Map<Boolean, List<CampaignModerationEvent>> eventsByRestrictedMode =
                    StreamEx.of(events)
                            .mapToEntry(e -> !isUpdateStatusEnabled(e), Function.identity())
                            .grouping();

            processInRestrictedMode(eventsByRestrictedMode.getOrDefault(Boolean.TRUE, emptyList()),
                    shard,
                    moderationRequestsConsumerBuilder);
            processInternal(eventsByRestrictedMode.getOrDefault(Boolean.FALSE, emptyList()),
                    shard,
                    moderationRequestsConsumerBuilder);
        }
    }

    private boolean isUpdateStatusEnabled(CampaignModerationEvent event) {
        return event.getClientId() % 100 < updateStatusEnabled.getOrDefault(0);
    }

    private void processInRestrictedMode(List<CampaignModerationEvent> events,
                                 int shard,
                                 ModerationRequestsConsumerBuilder moderationRequestsConsumerBuilder) {
        try {
            moderationOperationModeProvider.forceMode(ModerationOperationMode.RESTRICTED);
            processInternal(events, shard, moderationRequestsConsumerBuilder);
        } finally {
            moderationOperationModeProvider.disableForcedMode();
        }
    }

    private void processInternal(List<CampaignModerationEvent> events,
                                 int shard,
                                 ModerationRequestsConsumerBuilder moderationRequestsConsumerBuilder) {
        if (events.isEmpty()) {
            return;
        }

        List<Long> ids = mapList(events, CampaignModerationEvent::getCampaignId);

        Map<Long, Long> idToEventTimeMap =
                getIdToEventTimeMap(events, CampaignModerationEvent::getCampaignId);

        Map<Long, String> idToEssTagMap =
                getIdToEssTagMap(events, CampaignModerationEvent::getCampaignId);

        campaignSender.send(shard, ids,
                (o) -> idToEventTimeMap.get(o.getId()),
                (o) -> idToEssTagMap.get(o.getId()),
                moderationRequestsConsumerBuilder.build(campaignModerationWriter, moderationWriterMonitoring));
    }

    public CampaignModerationWriter getWriter() {
        return campaignModerationWriter;
    }
}
