package ru.yandex.direct.logicprocessor.processors.moderation.campaign.copy;

import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.moderation.model.campaign.CampaignCopyModerationRequest;
import ru.yandex.direct.core.entity.user.repository.UserRepository;
import ru.yandex.direct.ess.logicobjects.moderation.campaign.CampaignModerationEvent;
import ru.yandex.direct.logicprocessor.processors.moderation.ModerationRequestsConsumerBuilder;
import ru.yandex.direct.logicprocessor.processors.moderation.writers.ModerationWriterMonitoring;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceProfile;

import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Component
public class CampaignCopyModerationHandler {
    private static final Logger logger = LoggerFactory.getLogger(CampaignCopyModerationHandler.class);

    private final CampaignCopyModerationWriter moderationWriter;
    private final ModerationWriterMonitoring moderationWriterMonitoring;
    private final CampaignCopyModerationEventConverter eventConverter;
    private final UserRepository userRepository;

    @Autowired
    CampaignCopyModerationHandler(CampaignCopyModerationWriter moderationWriter,
                                  ModerationWriterMonitoring moderationWriterMonitoring,
                                  CampaignCopyModerationEventConverter eventConverter,
                                  UserRepository userRepository) {
        this.moderationWriter = moderationWriter;
        this.moderationWriterMonitoring = moderationWriterMonitoring;
        this.eventConverter = eventConverter;
        this.userRepository = userRepository;
    }

    public void process(List<CampaignModerationEvent> events,
                 int shard,
                 ModerationRequestsConsumerBuilder moderationRequestsConsumerBuilder) {
        try (TraceProfile profile = Trace.current().profile(
                "moderation_campaign-copy_requests.make", String.valueOf(shard))) {
            List<CampaignCopyModerationRequest> requests = mapList(events, eventConverter::convert);
            List<Long> toUids = events.stream()
                    .map(CampaignModerationEvent::getToUid)
                    .collect(Collectors.toList());
            Map<Long, String> loginsByUids = userRepository.getLoginsByUids(toUids);
            requests.forEach(request -> {
                String login = loginsByUids.get(request.getToUid());
                if (login == null) {
                    logger.warn("Login for uid = " + request.getToUid() + " is not found");
                } else {
                    request.setToLogin(loginsByUids.get(request.getToUid()));
                }
            });
            requests = requests.stream()
                    .filter(request -> request.getToLogin() != null)
                    .collect(Collectors.toList());
            moderationRequestsConsumerBuilder.build(moderationWriter, moderationWriterMonitoring).accept(requests);
        }
    }

    public CampaignCopyModerationWriter getWriter() {
        return moderationWriter;
    }
}
