package ru.yandex.direct.logicprocessor.processors.moderation.mobilecontenticon;

import java.time.Duration;
import java.util.List;
import java.util.Map;

import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.common.db.PpcPropertyNames;
import ru.yandex.direct.core.entity.mobilecontent.repository.MobileContentRepository;
import ru.yandex.direct.core.entity.moderation.ModerationOperationMode;
import ru.yandex.direct.core.entity.moderation.ModerationOperationModeProvider;
import ru.yandex.direct.core.entity.moderation.service.sending.MobileContentIconSender;
import ru.yandex.direct.env.NonProductionEnvironment;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.ess.config.moderation.mobilecontenticon.MobileContentIconModerationConfig;
import ru.yandex.direct.ess.logicobjects.moderation.mobilecontenticon.MobileContentIconModerationEventsObject;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.logicprocessor.common.EssLogicProcessor;
import ru.yandex.direct.logicprocessor.common.EssLogicProcessorContext;
import ru.yandex.direct.logicprocessor.processors.moderation.BaseModerationEventsProcessor;
import ru.yandex.direct.logicprocessor.processors.moderation.writers.MobileContentIconModerationWriter;
import ru.yandex.direct.logicprocessor.processors.moderation.writers.ModerationRequestsWriter;
import ru.yandex.direct.logicprocessor.processors.moderation.writers.ModerationWriterMonitoring;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceProfile;

import static java.util.Collections.emptyList;
import static java.util.function.Function.identity;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_MODERATION;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_1;
import static ru.yandex.direct.logicprocessor.processors.moderation.ModerationRequestUtils.getIdToEssTagMap;
import static ru.yandex.direct.logicprocessor.processors.moderation.ModerationRequestUtils.getIdToEventTimeMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@JugglerCheck(
        ttl = @JugglerCheck.Duration(minutes = 5),
        needCheck = ProductionOnly.class,
        tags = {DIRECT_PRIORITY_1, DIRECT_MODERATION})
@JugglerCheck(ttl = @JugglerCheck.Duration(minutes = 10), needCheck = NonProductionEnvironment.class, tags =
        {DIRECT_PRIORITY_1})
@EssLogicProcessor(MobileContentIconModerationConfig.class)
public class MobileContentIconModerationEventsProcessor
        extends BaseModerationEventsProcessor<MobileContentIconModerationEventsObject> {

    private final MobileContentRepository mobileContentRepository;
    private final ModerationOperationModeProvider moderationOperationModeProvider;
    private final MobileContentIconSender mobileContentIconSender;
    private final MobileContentIconModerationWriter mobileContentIconModerationWriter;
    private final ModerationWriterMonitoring moderationWriterMonitoring;

    private final PpcProperty<Integer> newTransportEnabled;

    @Autowired
    public MobileContentIconModerationEventsProcessor(
            EssLogicProcessorContext essLogicProcessorContext,
            MobileContentRepository mobileContentRepository,
            ModerationOperationModeProvider moderationOperationModeProvider,
            MobileContentIconSender mobileContentIconSender,
            MobileContentIconModerationWriter mobileContentIconModerationWriter,
            ModerationWriterMonitoring moderationWriterMonitoring,
            PpcPropertiesSupport ppcPropertiesSupport) {
        super(essLogicProcessorContext);

        this.mobileContentRepository = mobileContentRepository;
        this.moderationOperationModeProvider = moderationOperationModeProvider;
        this.mobileContentIconSender = mobileContentIconSender;
        this.mobileContentIconModerationWriter = mobileContentIconModerationWriter;
        this.moderationWriterMonitoring = moderationWriterMonitoring;

        this.newTransportEnabled = ppcPropertiesSupport.get(
                PpcPropertyNames.ENABLE_MOBILE_CONTENT_ICON_TRANSPORT_NEW_MODERATION,
                Duration.ofMinutes(1));
    }

    @Override
    protected List<? extends ModerationRequestsWriter> getWriters() {
        return List.of(mobileContentIconModerationWriter);
    }

    @Override
    protected void processObjects(List<MobileContentIconModerationEventsObject> events) {
        try (TraceProfile profile = Trace.current().profile(
                mobileContentIconSender.typeName() + "mobile_content_icon_moderation_requests.make",
                String.valueOf(getShard()))) {

            List<MobileContentIconModerationEventsObject> filteredEvents = filterEventsForProcessing(events);

            Map<Boolean, List<MobileContentIconModerationEventsObject>> restrictedModeToEventsMap =
                    StreamEx.of(filteredEvents)
                            .mapToEntry(MobileContentIconModerationEventsObject::getWasCopied, identity())
                            .grouping();

            processObjectsInRestrictedMode(restrictedModeToEventsMap.getOrDefault(Boolean.TRUE, emptyList()));
            processObjectsInternal(restrictedModeToEventsMap.getOrDefault(Boolean.FALSE, emptyList()));
        }
    }

    private List<MobileContentIconModerationEventsObject> filterEventsForProcessing(
            List<MobileContentIconModerationEventsObject> events) {
        List<Long> ids = mapList(events, MobileContentIconModerationEventsObject::getObjectId);
        Map<Long, Long> mobileContentIdToClientId = mobileContentRepository.getClientIdsByMobileContentIds(getShard(), ids);

        return StreamEx.of(events)
                .mapToEntry(MobileContentIconModerationEventsObject::getObjectId, identity())
                .filterKeys(mobileContentIdToClientId::containsKey)
                .filterKeys(id -> isNewTransportEnabledForClient(mobileContentIdToClientId.get(id)))
                .values()
                .toList();
    }

    private boolean isNewTransportEnabledForClient(Long clientId) {
        return clientId % 100 < newTransportEnabled.getOrDefault(0);
    }

    protected void processObjectsInRestrictedMode(List<MobileContentIconModerationEventsObject> events) {
        try {
            moderationOperationModeProvider.forceMode(ModerationOperationMode.RESTRICTED);
            processObjectsInternal(events);
        } finally {
            moderationOperationModeProvider.disableForcedMode();
        }
    }

    protected void processObjectsInternal(List<MobileContentIconModerationEventsObject> events) {
        if (events.isEmpty()) {
            return;
        }

        List<Long> mobileContentIds = mapList(events, MobileContentIconModerationEventsObject::getObjectId);

        Map<Long, Long> idToEventTimeMap =
                getIdToEventTimeMap(events, MobileContentIconModerationEventsObject::getObjectId);

        Map<Long, String> idToEssTagMap =
                getIdToEssTagMap(events, MobileContentIconModerationEventsObject::getObjectId);

        mobileContentIconSender.send(getShard(), mobileContentIds,
                (o) -> idToEventTimeMap.get(o.getId()),
                (o) -> idToEssTagMap.get(o.getId()),
                makeRequestsConsumer(mobileContentIconModerationWriter, moderationWriterMonitoring));
    }
}
