package ru.yandex.direct.logicprocessor.processors.moderation.special.archiving;

import java.util.List;

import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod;
import ru.yandex.direct.env.NonProductionEnvironment;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.ess.config.moderation.special.ModerationArchivingConfig;
import ru.yandex.direct.ess.logicobjects.moderation.special.ModerationArchivingEvent;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification;
import ru.yandex.direct.logicprocessor.common.EssLogicProcessor;
import ru.yandex.direct.logicprocessor.common.EssLogicProcessorContext;
import ru.yandex.direct.logicprocessor.processors.moderation.BaseModerationEventsProcessor;
import ru.yandex.direct.logicprocessor.processors.moderation.writers.ModerationRequestsWriter;
import ru.yandex.direct.logicprocessor.processors.moderation.writers.ModerationWriterMonitoring;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceProfile;
import ru.yandex.monlib.metrics.labels.Labels;

import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_MODERATION;
import static ru.yandex.direct.juggler.check.model.NotificationRecipient.LOGIN_STASIS93;

@JugglerCheck(
        ttl = @JugglerCheck.Duration(minutes = 5),
        needCheck = ProductionOnly.class,
        notifications = {
                @OnChangeNotification(recipient = LOGIN_STASIS93,
                        status = {JugglerStatus.OK, JugglerStatus.CRIT},
                        method = NotificationMethod.TELEGRAM),
        },
        tags = {DIRECT_MODERATION})
@JugglerCheck(ttl = @JugglerCheck.Duration(minutes = 10), needCheck = NonProductionEnvironment.class)
@EssLogicProcessor(ModerationArchivingConfig.class)
public class ModerationArchivingEventsProcessor extends BaseModerationEventsProcessor<ModerationArchivingEvent> {

    private final ModerationArchivingWriter writer;
    private final ModerationWriterMonitoring monitoring;
    private final ModerationArchivingEventsService archivingEventsService;

    @Autowired
    public ModerationArchivingEventsProcessor(EssLogicProcessorContext essLogicProcessorContext,
                                              ModerationArchivingWriter writer,
                                              ModerationWriterMonitoring monitoring,
                                              ModerationArchivingEventsService archivingEventsService) {
        super(essLogicProcessorContext);
        this.writer = writer;
        this.monitoring = monitoring;
        this.archivingEventsService = archivingEventsService;
    }

    @Override
    protected List<? extends ModerationRequestsWriter> getWriters() {
        return List.of(writer);
    }

    @Override
    protected void processObjects(List<ModerationArchivingEvent> events) {
        try (TraceProfile profile = Trace.current().profile(
                "moderation_archiving_requests.make", String.valueOf(getShard()))) {
            archivingEventsService.processEvents(getShard(), events, requests -> {
                int sentRequestsCount = writer.writeRequests(requests);
                monitoring.incModerationRequestCount(sentRequestsCount, getLabelsForMonitoring(Labels.empty()));
            });
        }
    }
}
