package ru.yandex.direct.logicprocessor.processors.moderation.special.archiving;

import java.time.Duration;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.BinaryOperator;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.google.common.collect.Lists;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.common.db.PpcPropertyNames;
import ru.yandex.direct.core.entity.banner.repository.BannerModerationRepository;
import ru.yandex.direct.core.entity.banner.repository.BannerTypedRepository;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.ess.common.circuits.moderation.ModerationArchivingObjectType;
import ru.yandex.direct.ess.logicobjects.moderation.banner.BannerModerationEventsObject;
import ru.yandex.direct.ess.logicobjects.moderation.special.ModerationArchivingEvent;
import ru.yandex.direct.logicprocessor.processors.moderation.banner.BannerModerationEventsService;
import ru.yandex.direct.logicprocessor.processors.moderation.banner.support.BannerModerationEventsWithInfo;

import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapList;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
public class ModerationArchivingEventsService {

    private static final int DEFAULT_BATCH_SIZE = 1000;
    private int batchSize;

    private static final Logger logger = LoggerFactory.getLogger(ModerationArchivingEventsService.class);
    private final DslContextProvider dslContextProvider;
    private final BannerModerationRepository bannerModerationRepository;
    private final BannerTypedRepository bannerTypedRepository;
    private final BannerModerationEventsService bannerModerationEventsService;
    private final BannerInitialVersionEvaluator bannerInitialVersionEvaluator;
    private final PpcProperty<Boolean> sendBannerArchiving;
    private final PpcProperty<Boolean> sendBannerArchivingByCampaign;

    public ModerationArchivingEventsService(DslContextProvider dslContextProvider,
                                            BannerModerationRepository bannerModerationRepository,
                                            BannerTypedRepository bannerTypedRepository,
                                            PpcPropertiesSupport ppcPropertiesSupport,
                                            BannerModerationEventsService bannerModerationEventsService,
                                            BannerInitialVersionEvaluator bannerInitialVersionEvaluator) {
        this.batchSize = DEFAULT_BATCH_SIZE;
        this.dslContextProvider = dslContextProvider;
        this.bannerModerationRepository = bannerModerationRepository;
        this.bannerTypedRepository = bannerTypedRepository;
        this.bannerModerationEventsService = bannerModerationEventsService;
        this.bannerInitialVersionEvaluator = bannerInitialVersionEvaluator;

        this.sendBannerArchiving = ppcPropertiesSupport.get(
                PpcPropertyNames.ENABLE_BANNER_ARCHIVING_TRANSPORT,
                Duration.ofMinutes(1));

        this.sendBannerArchivingByCampaign = ppcPropertiesSupport.get(
                PpcPropertyNames.ENABLE_BANNER_ARCHIVING_BY_CAMPAIGN_TRANSPORT,
                Duration.ofMinutes(1));
    }

    public void setBatchSize(int batchSize) {
        this.batchSize = batchSize;
    }

    public void processEvents(int shard,
                              List<ModerationArchivingEvent> events,
                              Consumer<List<ModerationArchivingRequest>> requestsConsumer) {
        var eventsByType = events.stream()
                .filter(this::filterEvent)
                .collect(Collectors.groupingBy(ModerationArchivingEvent::getType));

        List<ModerationArchivingEvent> bannerEvents = getBannerEvents(eventsByType);
        Map<Long, ModerationArchivingEvent> campaignEvents = getCampaignEvents(eventsByType);

        for (var eventsPart : Lists.partition(bannerEvents, batchSize)) {
            logger.info("Processing {} banner archiving events", eventsPart.size());
            processBannerEventsBatch(shard, eventsPart, requestsConsumer);
        }

        if (!campaignEvents.isEmpty()) {
            List<BannerTypedRepository.BannerData> bannersData;
            Long lastBid = null;
            do {
                bannersData = bannerTypedRepository.getOrderedBannersByCampaignIds(
                        shard,
                        campaignEvents.keySet(),
                        batchSize,
                        lastBid);

                var generatedBannerEvents = generateBannerEvents(bannersData, campaignEvents);

                logger.info("Processing {} generated banner archiving events", generatedBannerEvents.size());
                processBannerEventsBatch(shard, generatedBannerEvents, requestsConsumer);
                if (!bannersData.isEmpty()) {
                    lastBid = bannersData.get(bannersData.size() - 1).bannerId;
                }
            } while (bannersData.size() >= batchSize);
        }
    }

    private boolean filterEvent(ModerationArchivingEvent event) {
        if (event.getType() == ModerationArchivingObjectType.BANNER) {
            return sendBannerArchiving.getOrDefault(false);
        }
        if (event.getType() == ModerationArchivingObjectType.CAMPAIGN) {
            return sendBannerArchivingByCampaign.getOrDefault(false);
        }
        return false;
    }

    private Map<Long, ModerationArchivingEvent> getLatestEvents(List<ModerationArchivingEvent> events,
                                                                Function<ModerationArchivingEvent, Long> keyExtractor) {
        return events.stream().collect(Collectors.toMap(keyExtractor, Function.identity(),
                BinaryOperator.maxBy(Comparator.comparing(ModerationArchivingEvent::getEventTime))));
    }

    private List<ModerationArchivingEvent> getBannerEvents(
            Map<ModerationArchivingObjectType, List<ModerationArchivingEvent>> allEvents) {
        List<ModerationArchivingEvent> bannerEvents = allEvents.getOrDefault(
                ModerationArchivingObjectType.BANNER, Collections.emptyList());
        return new ArrayList<>(getLatestEvents(bannerEvents, ModerationArchivingEvent::getBannerId).values());
    }

    private Map<Long, ModerationArchivingEvent> getCampaignEvents(
            Map<ModerationArchivingObjectType, List<ModerationArchivingEvent>> allEvents) {
        List<ModerationArchivingEvent> campaignEvents = allEvents.getOrDefault(
                ModerationArchivingObjectType.CAMPAIGN, Collections.emptyList());
        return getLatestEvents(campaignEvents, ModerationArchivingEvent::getCampaignId);
    }

    private List<ModerationArchivingEvent> generateBannerEvents(List<BannerTypedRepository.BannerData> bannersData,
                                                                Map<Long, ModerationArchivingEvent> campaignEvents) {
        return mapList(bannersData, bannerData -> {
            ModerationArchivingEvent campaignEvent = campaignEvents.get(bannerData.campaignId);

            return ModerationArchivingEvent.bannerArchivingEvent(
                    campaignEvent.getEssTag(),
                    campaignEvent.getEventTime(),
                    campaignEvent.getCampaignId(),
                    bannerData.adgroupId,
                    bannerData.bannerId,
                    bannerData.bannerType);
        });
    }

    private void processBannerEventsBatch(int shard,
                                          List<ModerationArchivingEvent> events,
                                          Consumer<List<ModerationArchivingRequest>> requestsConsumer) {
        if (events.isEmpty()) {
            return;
        }
        Map<Long, Long> archivingVersions = getArchivingVersions(shard, events);

        List<ModerationArchivingRequest> requests = filterAndMapList(
                events,
                ev -> archivingVersions.containsKey(ev.getBannerId()),
                ev -> new ModerationArchivingRequest(
                        ev.getType(),
                        ev.getBannerId(),
                        archivingVersions.get(ev.getBannerId()),
                        ev.getEventTime()));

        if (!requests.isEmpty()) {
            requestsConsumer.accept(requests);
        }
    }

    private Map<Long, Long> getArchivingVersions(int shard, List<ModerationArchivingEvent> events) {
        Map<Long, Long> moderationVersions = bannerModerationRepository.getBannerModerateVersions(
                dslContextProvider.ppc(shard),
                mapList(events, ModerationArchivingEvent::getBannerId));

        List<ModerationArchivingEvent> eventsWithoutModerationVersion = filterList(
                events,
                ev -> !moderationVersions.containsKey(ev.getBannerId()));

        Map<Long, Long> preInitialVersions = getPreInitialVersions(shard, eventsWithoutModerationVersion);

        Set<Long> bidsWithoutInitialVersions = eventsWithoutModerationVersion.stream()
                .map(ModerationArchivingEvent::getBannerId)
                .filter(bid -> !preInitialVersions.containsKey(bid))
                .collect(Collectors.toSet());

        if (!bidsWithoutInitialVersions.isEmpty()) {
            String bids = bidsWithoutInitialVersions.stream()
                    .map(Object::toString)
                    .collect(Collectors.joining(", "));
            logger.warn("Could not determine initial moderation version for bids {}", bids);
        }

        return Stream.concat(moderationVersions.entrySet().stream(), preInitialVersions.entrySet().stream())
                .collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue));
    }

    private Map<Long, Long> getPreInitialVersions(int shard, List<ModerationArchivingEvent> events) {
        List<BannerModerationEventsObject> convertedEvents = mapList(events,
                ModerationArchivingEventsService::convertBannerArchivingEvent);

        List<BannerModerationEventsWithInfo> enrichedEvents =
                bannerModerationEventsService.mapEventsToBannerWithInfo(shard, convertedEvents);

        return StreamEx.of(enrichedEvents).mapToEntry(
                        BannerModerationEventsWithInfo::getBannerId,
                        bannerInitialVersionEvaluator::getInitialVersion)
                .nonNullValues()
                .mapValues(version -> (version > 0) ? version - 1 : version)
                .toMap();
    }

    private static BannerModerationEventsObject convertBannerArchivingEvent(ModerationArchivingEvent event) {
        return new BannerModerationEventsObject(
                event.getEssTag(),
                event.getEventTime(),
                event.getCampaignId(),
                event.getAdGroupId(),
                event.getBannerId(),
                event.getBannersBannerType(),
                false,
                false);
    }
}
