package ru.yandex.direct.logicprocessor.processors.moderation.special.flags;

import java.time.Duration;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.common.db.PpcPropertyNames;
import ru.yandex.direct.core.entity.banner.repository.BannerCommonRepository;
import ru.yandex.direct.dbschema.ppc.enums.BannersBannerType;
import ru.yandex.direct.env.NonProductionEnvironment;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.ess.config.moderation.special.ModerationFlagsConfig;
import ru.yandex.direct.ess.logicobjects.moderation.banner.BannerModerationEventsObject;
import ru.yandex.direct.ess.logicobjects.moderation.special.ModerationFlagsEvent;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.logicprocessor.common.EssLogicProcessor;
import ru.yandex.direct.logicprocessor.common.EssLogicProcessorContext;
import ru.yandex.direct.logicprocessor.processors.moderation.BaseModerationEventsProcessor;
import ru.yandex.direct.logicprocessor.processors.moderation.banner.BannerModerationEventsService;
import ru.yandex.direct.logicprocessor.processors.moderation.banner.support.AdImageBannerModerationEventsHandlerProvider;
import ru.yandex.direct.logicprocessor.processors.moderation.banner.support.BannerModerationEventsHandler;
import ru.yandex.direct.logicprocessor.processors.moderation.banner.support.BannerModerationEventsHandlerProvider;
import ru.yandex.direct.logicprocessor.processors.moderation.banner.support.BannerModerationEventsWithInfo;
import ru.yandex.direct.logicprocessor.processors.moderation.banner.support.ContentPromotionBannerModerationEventsHandlerProvider;
import ru.yandex.direct.logicprocessor.processors.moderation.banner.support.CpmAudioBannerModerationEventsHandlerProvider;
import ru.yandex.direct.logicprocessor.processors.moderation.banner.support.CpmBannerModerationEventsHandlerProvider;
import ru.yandex.direct.logicprocessor.processors.moderation.banner.support.CpmGeoPinBannerModerationEventsHandlerProvider;
import ru.yandex.direct.logicprocessor.processors.moderation.banner.support.InternalBannerModerationEventsHandlerProvider;
import ru.yandex.direct.logicprocessor.processors.moderation.banner.support.TextBannerModerationEventsHandlerProvider;
import ru.yandex.direct.logicprocessor.processors.moderation.writers.ModerationRequestsWriter;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceProfile;

import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_MODERATION;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_1;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@JugglerCheck(
        ttl = @JugglerCheck.Duration(minutes = 5),
        needCheck = ProductionOnly.class,
        tags = {DIRECT_PRIORITY_1, DIRECT_MODERATION})
@JugglerCheck(ttl = @JugglerCheck.Duration(minutes = 10), needCheck = NonProductionEnvironment.class, tags = {DIRECT_PRIORITY_1})
@EssLogicProcessor(ModerationFlagsConfig.class)
public class ModerationFlagsEventsProcessor extends BaseModerationEventsProcessor<ModerationFlagsEvent> {
    private final BannerCommonRepository bannerCommonRepository;
    private final BannerModerationEventsService bannerModerationService;
    private final Collection<BannerModerationEventsHandlerProvider> providers;
    private final List<BannerModerationEventsHandler> handlers;

    private final PpcProperty<Boolean> flagsTransportEnabled;

    @Autowired
    public ModerationFlagsEventsProcessor(
            EssLogicProcessorContext essLogicProcessorContext,
            BannerCommonRepository bannerCommonRepository,
            BannerModerationEventsService bannerModerationService,
            ApplicationContext applicationContext,
            AdImageBannerModerationEventsHandlerProvider adImageBannerModerationEventsHandlerProvider,
            ContentPromotionBannerModerationEventsHandlerProvider contentPromotionBannerModerationEventsHandlerProvider,
            CpmAudioBannerModerationEventsHandlerProvider cpmAudioBannerModerationEventsHandlerProvider,
            CpmBannerModerationEventsHandlerProvider cpmBannerModerationEventsHandlerProvider,
            CpmGeoPinBannerModerationEventsHandlerProvider cpmGeoPinBannerModerationEventsHandlerProvider,
            InternalBannerModerationEventsHandlerProvider internalBannerModerationEventsHandlerProvider,
            TextBannerModerationEventsHandlerProvider textBannerModerationEventsHandlerProvider,
            PpcPropertiesSupport ppcPropertiesSupport) {
        super(essLogicProcessorContext);

        this.bannerCommonRepository = bannerCommonRepository;
        this.bannerModerationService = bannerModerationService;

        this.providers = applicationContext.getBeansOfType(BannerModerationEventsHandlerProvider.class).values();
        this.handlers = this.providers.stream()
                .map(p -> p.build(createModerationRequestsConsumerBuilder()))
                .collect(Collectors.toList());

        this.flagsTransportEnabled = ppcPropertiesSupport.get(PpcPropertyNames.ENABLE_FLAGS_TRANSPORT_NEW_MODERATION,
                Duration.ofMinutes(1));
    }

    @Override
    protected void processObjects(List<ModerationFlagsEvent> events) {
        if (!flagsTransportEnabled.getOrDefault(false)) {
            return;
        }

        try (TraceProfile profile = Trace.current().profile(
                "moderation_flags_requests.make", String.valueOf(getShard()))) {

            List<Long> bannerIds = mapList(events, ModerationFlagsEvent::getBannerId);
            Map<Long, BannersBannerType> bannerIdToType = bannerCommonRepository.getBannerTypesByIds(getShard(), bannerIds);

            // Конвертируем события изменения флагов пользователем в событие изменения баннера с атрибутом isFlagsOnly = true.
            // Новые события посылаем в обработчики событий баннеров.
            // В обработчиках есть фильтрация по типу баннеров, поэтому не пытаемся заранее сматчить обработчик и событие.
            List<BannerModerationEventsObject> bannerEvents = convertEvents(events, bannerIdToType);
            List<BannerModerationEventsWithInfo> bannerModerationEventsWithInfos =
                    bannerModerationService.mapEventsToBannerWithInfo(getShard(), bannerEvents);
            handlers.forEach(h-> h.handleObjects(getShard(), bannerModerationEventsWithInfos));
        }
    }

    private List<BannerModerationEventsObject> convertEvents(List<ModerationFlagsEvent> events,
                                                             Map<Long, BannersBannerType> bannerIdToType) {
        return events.stream()
                .filter(e -> bannerIdToType.containsKey(e.getBannerId()))
                .map(e -> new BannerModerationEventsObject(
                        e.getEssTag(),
                        e.getEventTime(),
                        e.getCampaignId(),
                        e.getAdGroupId(),
                        e.getBannerId(),
                        bannerIdToType.get(e.getBannerId()),
                        false,
                        true))
                .collect(Collectors.toList());
    }

    @Override
    protected List<? extends ModerationRequestsWriter> getWriters() {
        return providers.stream()
                .map(BannerModerationEventsHandlerProvider::getWriters)
                .flatMap(Collection::stream)
                .collect(Collectors.toList());
    }
}
