package ru.yandex.direct.logicprocessor.processors.moderation.writers;

import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.function.Supplier;

import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.binlogbroker.logbroker_utils.models.SourceType;
import ru.yandex.direct.binlogbroker.logbroker_utils.writer.JsonLinesLogbrokerWriter;
import ru.yandex.direct.binlogbroker.logbroker_utils.writer.LogbrokerWriter;
import ru.yandex.direct.binlogbroker.logbroker_utils.writer.LogbrokerWriterCloseException;
import ru.yandex.direct.common.log.container.ModerationLogEntry;
import ru.yandex.direct.common.log.service.ModerationLogService;
import ru.yandex.direct.env.EnvironmentType;
import ru.yandex.direct.ess.common.logbroker.LogbrokerClientFactoryFacade;
import ru.yandex.direct.ess.common.logbroker.LogbrokerProducerProperties;
import ru.yandex.direct.logicprocessor.processors.moderation.ModerationRequestFilter;
import ru.yandex.direct.logicprocessor.processors.moderation.ModerationRequestLogEntryCreator;
import ru.yandex.direct.logicprocessor.processors.moderation.PartitionGroupComputer;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceProfile;
import ru.yandex.kikimr.persqueue.producer.AsyncProducer;

import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Создаёт logBrokerWriter для отправки moderationRequests. Вызывающий код должен передать нужный producerProperties
 * чтобы писать в нужный ему topic logbrokerа. Чтобы отправить moderationRequests надо вызвать
 * метод writeRequests().
 */
public class ModerationRequestsWriterImpl<T> implements ModerationRequestsWriter<T> {
    private static final Logger logger = LoggerFactory.getLogger(ModerationRequestsWriterImpl.class);

    private static final String SOURCE_TEMPLATE = "%s:%d";
    private static final int MODERATION_CHUNK_SIZE = 100;

    private final EnvironmentType environmentType;
    private final ModerationRequestFilter filter;
    private final LogbrokerClientFactoryFacade logbrokerClientFactory;
    private final ModerationRequestLogEntryCreator<? super T> logEntryCreator;
    private final ModerationLogService moderationLogService;
    private final List<LogbrokerProducerProperties> producersProperties;
    private final PartitionGroupComputer<? super T> partitionGroupComputer;
    // префикс, название заявки в модерацию
    private final String sourceIdPrefix;
    // postfix, который добавляется в конце sourceId в logbroker writer-е
    // важно, чтобы конечный sourceId не пересекался с lb-moderation
    private final String sourceIdPostfix;

    private List<LogbrokerWriter<T>> moderationRequestsWriters;
    private List<Integer> partitions;
    private int shard;

    protected ModerationRequestsWriterImpl(EnvironmentType environmentType,
                                           ModerationRequestFilter filter,
                                           LogbrokerClientFactoryFacade logbrokerClientFactory,
                                           ModerationRequestLogEntryCreator<? super T> logEntryCreator,
                                           ModerationLogService moderationLogService,
                                           List<LogbrokerProducerProperties> producersProperties,
                                           PartitionGroupComputer<? super T> partitionGroupComputer,
                                           String sourceIdPrefix,
                                           String sourceIdPostfix) {
        this.environmentType = environmentType;
        this.filter = filter;
        this.logbrokerClientFactory = logbrokerClientFactory;
        this.logEntryCreator = logEntryCreator;
        this.moderationLogService = moderationLogService;
        this.producersProperties = producersProperties;
        this.partitionGroupComputer = partitionGroupComputer;
        this.sourceIdPrefix = sourceIdPrefix;
        this.sourceIdPostfix = sourceIdPostfix;
    }

    @Override
    public void initialize(String processorName, int shard) {
        this.shard = shard;
        SourceType sourceType = getSourceType(shard);

        // Для каждого набора свойств создаем отдельный writer
        this.moderationRequestsWriters = mapList(producersProperties, props ->
                createLogbrokerWriter(props, sourceIdPrefix, sourceType, sourceIdPostfix, processorName));
        this.partitions = mapList(producersProperties, LogbrokerProducerProperties::getGroup);
    }

    /**
     * sourceId должен быть уникальным в рамках каждого запуска джобы.
     * Например, для обработки текстового баннера на первом шарде будет: banner-moderationproduction:ppc:1text
     */
    private LogbrokerWriter<T> createLogbrokerWriter(
            LogbrokerProducerProperties properties, String prefix, SourceType sourceType, String postfix,
            String processorName) {
        String sourceId = new StringBuilder()
                .append(processorName.toLowerCase() + ":")
                .append(prefix)
                .append(sourceType.getSourceName())
                .append(postfix)
                .append(":partition")
                .append(properties.getGroup())
                .toString();

        Supplier<CompletableFuture<AsyncProducer>> asyncProducerSupplier = logbrokerClientFactory
                .createProducerSupplier(properties, sourceId);

        return new JsonLinesLogbrokerWriter<>(asyncProducerSupplier, Duration.ofSeconds(properties.getTimeoutSec()),
                properties.getRetries(), MODERATION_CHUNK_SIZE);
    }

    private SourceType getSourceType(int shard) {
        EnvironmentType dbEnv = environmentType == EnvironmentType.DEVELOPMENT ? EnvironmentType.DEVTEST :
                environmentType;
        return SourceType.fromType(dbEnv, String.format(SOURCE_TEMPLATE, "ppc", shard));
    }

    @Override
    public void finish() {
        if (Objects.nonNull(moderationRequestsWriters)) {
            try {
                moderationRequestsWriters.forEach(LogbrokerWriter::close);
            } catch (LogbrokerWriterCloseException ex) {
                logger.error("Error while closing moderation logbroker writer", ex);
            }
        }
    }

    @Override
    public int writeRequests(List<T> requests) {
        List<ModerationLogEntry<T>> logEntries = new ArrayList<>();

        try (TraceProfile profile = Trace.current().profile("base_moderation_requests.write",
                String.valueOf(shard))) {

            List<T> filteredRequests = filter.filter(requests);

            filteredRequests.stream()
                    .map(logEntryCreator::createLogEntry)
                    .forEach(logEntries::add);

            StreamEx.of(filteredRequests)
                    .groupingBy(this::computePartitionGroup)
                    .forEach((group, chunk) -> {
                        logger.info("Writing {} requests to partition {}", chunk.size(), partitions.get(group));
                        moderationRequestsWriters.get(group).writeSync(chunk);
                    });

            return filteredRequests.size();

        } catch (Exception e) {
            logEntries.forEach(logEntry -> logEntry.setSuccess(false));
            throw e;
        } finally {
            logEntries.forEach(moderationLogService::logEvent);
        }
    }

    private int computePartitionGroup(T moderationRequest) {
        return partitionGroupComputer.computePartitionGroup(moderationRequest, moderationRequestsWriters.size());
    }
}
