package ru.yandex.direct.logicprocessor.processors.mysql2grut.replicationwriter

import org.slf4j.Logger
import org.slf4j.LoggerFactory
import org.springframework.stereotype.Component
import ru.yandex.direct.common.db.PpcPropertiesSupport
import ru.yandex.direct.common.db.PpcProperty
import ru.yandex.direct.common.db.PpcPropertyNames
import ru.yandex.direct.core.entity.adgroup.model.AdGroup
import ru.yandex.direct.core.entity.adgroup.model.AdGroupPriceSales
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType
import ru.yandex.direct.core.entity.adgroup.model.AdGroupWithPageBlocks
import ru.yandex.direct.core.entity.adgroup.model.DynamicAdGroup
import ru.yandex.direct.core.entity.adgroup.model.InternalAdGroup
import ru.yandex.direct.core.entity.adgroup.model.MobileContentAdGroup
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.AdGroupAdditionalTargeting
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.repository.AdGroupAdditionalTargetingRepository
import ru.yandex.direct.core.entity.retargeting.model.Goal
import ru.yandex.direct.core.grut.api.AdGroupGrut
import ru.yandex.direct.core.grut.api.BiddableShowConditionGrutApi
import ru.yandex.direct.core.grut.api.BiddableShowConditionType
import ru.yandex.direct.core.grut.api.InternalAdGroupOptions
import ru.yandex.direct.core.grut.api.MobileContentDetails
import ru.yandex.direct.core.grut.replication.GrutApiService
import ru.yandex.direct.core.mysql2grut.CryptaSegmentsCache
import ru.yandex.direct.core.mysql2grut.repository.BiddableShowConditionsRepository
import ru.yandex.direct.ess.logicobjects.mysql2grut.BidModifierTableType
import ru.yandex.direct.ess.logicobjects.mysql2grut.Mysql2GrutReplicationObject
import ru.yandex.direct.logicprocessor.processors.mysql2grut.service.BidModifierId
import ru.yandex.direct.logicprocessor.processors.mysql2grut.service.BidModifierReplicationService
import java.time.Duration

data class AdGroupWriterObject(
    val adGroupId: Long,
    val adGroup: AdGroup? = null, // not used in delete operation
    val biddableShowConditionsGrutIds: List<BiddableShowConditionId> = emptyList(), // not used in delete operation
    val bidModifiersIds: Set<BidModifierId> = emptySet(), // not used in delete operation
)

data class BiddableShowConditionId(val grutId: Long, val type: BiddableShowConditionType)

@Component
class AdGroupReplicationWriter(
    private val objectApiService: GrutApiService,
    private val adGroupRepository: AdGroupRepository,
    private val targetingRepository: AdGroupAdditionalTargetingRepository,
    private val biddableShowConditionsRepository: BiddableShowConditionsRepository,
    private val cryptaSegmentsCache: CryptaSegmentsCache,
    private val bidModifierReplicationService: BidModifierReplicationService,
    ppcPropertiesSupport: PpcPropertiesSupport,
) : BaseReplicationWriter<AdGroupWriterObject>() {

    companion object {
        private val adGroupLogger = LoggerFactory.getLogger(AdGroupReplicationWriter::class.java)

        fun convertAdGroup(
            adGroup: AdGroup,
            orderId: Long,
            targeting: List<AdGroupAdditionalTargeting>,
            biddableShowConditionGrutIds: List<Long>,
            cryptaSegmentsMapping: Map<Long, Goal>,
            bidModifiersIds: Collection<Long>,
        ): AdGroupGrut {
            val relevanceMatchData = (adGroup as? DynamicAdGroup)?.let { adGroup.relevanceMatchCategories }.orEmpty()
            val rfOptions = (adGroup as? InternalAdGroup)?.let {
                InternalAdGroupOptions(
                    adGroup.rf,
                    adGroup.rfReset,
                    adGroup.maxClicksCount,
                    adGroup.maxClicksPeriod,
                    adGroup.maxStopsCount,
                    adGroup.maxStopsPeriod,
                    adGroup.startTime,
                    adGroup.finishTime
                )
            }
            val internalLevel = (adGroup as? InternalAdGroup)?.let { adGroup.level }

            val mobileContentDetails = (adGroup as? MobileContentAdGroup)?.let {
                MobileContentDetails(
                    adGroup.mobileContentId,
                    adGroup.deviceTypeTargeting,
                    adGroup.networkTargeting,
                    adGroup.storeUrl,
                    adGroup.minimalOperatingSystemVersion,
                )
            }

            val pageBlocks = (adGroup as? AdGroupWithPageBlocks)?.let { adGroup.pageBlocks }.orEmpty()

            val priority = (adGroup as? AdGroupPriceSales)?.let { adGroup.priority }

            return AdGroupGrut(
                adGroup.id,
                orderId,
                adGroup.name,
                adGroup.type,
                adGroup.pageGroupTags,
                adGroup.targetTags,
                relevanceMatchData,
                rfOptions,
                internalLevel,
                getMinusPhrasesIds(adGroup),
                adGroup.geo,
                targeting,
                mobileContentDetails,
                pageBlocks,
                adGroup.trackingParams,
                priority,
                biddableShowConditionGrutIds,
                cryptaSegmentsMapping,
                bidModifiersIds,
            )
        }

        fun getMinusPhrasesIds(adGroup: AdGroup): Collection<Long> {
            return if (adGroup.minusKeywordsId == null) {
                adGroup.libraryMinusKeywordsIds
            } else {
                adGroup.libraryMinusKeywordsIds + adGroup.minusKeywordsId
            }
        }

        /**
         * Конвертирует идентификаторы biddableShowConditions Директа (сгруппированные по айди группы и типу) в идентификаторы BiddableShowConditions Грута (сгруппированные по группам)
         */
        fun directIdsToGrutIds(biddableShowConditionIds: Map<Long, Map<BiddableShowConditionType, List<Long>>>): Map<Long, List<BiddableShowConditionId>> {
            val result = mutableMapOf<Long, List<BiddableShowConditionId>>()
            for ((adGroupId, biddableShowConditionIdsByType) in biddableShowConditionIds) {
                val grutIds = biddableShowConditionIdsByType.flatMap { (type, ids) ->
                    ids.map {
                        val grutId = BiddableShowConditionGrutApi.directIdToGrutId(
                            type,
                            it
                        )
                        BiddableShowConditionId(grutId, type)
                    }
                }
                result[adGroupId] = grutIds
            }
            return result
        }
    }

    override val logger: Logger = adGroupLogger

    private val fillBiddableShowConditionsProperty =
        ppcPropertiesSupport.get(PpcPropertyNames.GRUT_AD_GROUPS_FILL_BIDDABLE_SHOW_CONDITIONS, Duration.ofSeconds(20))
    private val property: PpcProperty<Set<Int>> =
        ppcPropertiesSupport.get(PpcPropertyNames.GRUT_SKIP_ADGROUPS_REPLICATION_SHARDS, Duration.ofSeconds(20))

    private val asyncUpdate =
        ppcPropertiesSupport.get(PpcPropertyNames.GRUT_CAMP_REPL_ASYNC_UPDATE, Duration.ofSeconds(20))
    private val filterForeignEntities =
        ppcPropertiesSupport.get(PpcPropertyNames.ADGROUP_REPLICATION_FILTER_WITHOUT_FOREIGN, Duration.ofSeconds(20))

    override fun isDisabledInShard(shard: Int): Boolean {
        return property.get()
            .orEmpty()
            .contains(shard)
    }

    override fun getLogicObjectsToWrite(
        shard: Int,
        logicObjects: Collection<Mysql2GrutReplicationObject>
    ): ObjectsForUpdateAndDelete<AdGroupWriterObject> {

        val logicObjectsForUpdate = logicObjects
            .filter { !it.isDeleted }
            // Помимо изменения самих групп мы отслеживаем изменения в корректировках, чтобы обновить связи группы->корректировки
            .filter { (it.adGroupId != null || it.bidModifierIdForRelation != null) }

        val logicObjectsForDelete = logicObjects
            .filter { it.isDeleted && it.adGroupId != null }

        val objectForUpdate = createObjectsForUpdate(shard, logicObjectsForUpdate)
        val objectForDelete = logicObjectsForDelete.map { AdGroupWriterObject(it.adGroupId!!) }
            .distinct()

        return ObjectsForUpdateAndDelete(objectForUpdate, objectForDelete)
    }

    private fun createObjectsForUpdate(
        shard: Int,
        logicObjectsForUpdate: List<Mysql2GrutReplicationObject>
    ): List<AdGroupWriterObject> {
        if (logicObjectsForUpdate.isEmpty()) {
            return emptyList()
        }
        val adGroupIdsFromAdGroups = logicObjectsForUpdate.mapNotNull { it.adGroupId }

        // Идентификаторы групп, чьи корректировки изменились - необходимы для обновления связей на корректировки
        val bidModifiers = logicObjectsForUpdate
            .filter { it.bidModifierIdForRelation != null }
            .groupBy({ it.bidModifierTableType!! }, { it.bidModifierIdForRelation!! })
        val adGroupIdsFromBidModifiers = bidModifierReplicationService.getBidModifiers(shard, bidModifiers)
            .mapNotNull { it.adGroupId }
        if (adGroupIdsFromBidModifiers.isNotEmpty()) {
            logger.info("${adGroupIdsFromBidModifiers.size} groups to update from bid_modifiers changes: $adGroupIdsFromBidModifiers")
        }

        val adGroupIdsToLoad = (adGroupIdsFromAdGroups + adGroupIdsFromBidModifiers).distinct()
        if (adGroupIdsToLoad.isEmpty()) {
            return emptyList()
        }

        val adGroupsFromMySql = adGroupRepository.getAdGroups(shard, adGroupIdsToLoad)

        val adGroupBiddableShowConditionsGrutIds = if (fillBiddableShowConditionsProperty.getOrDefault(false)) {
            // Получить идентификаторы biddableShowConditions из директа, сгруппированые по типу и по группе
            val adGroupToBiddableShowConditionDirectIds =
                biddableShowConditionsRepository.getAdGroupsBiddableShowConditions(shard, adGroupIdsToLoad)
            // Конвертировать идентификаторы BiddableShowCondition директа в идентификаторы Грута
            directIdsToGrutIds(adGroupToBiddableShowConditionDirectIds)
        } else mapOf()

        val adGroupsToBidModifiers =
            bidModifierReplicationService.getGrutIdsForAdGroups(shard, adGroupIdsToLoad.toSet())


        return adGroupsFromMySql.map {
            AdGroupWriterObject(
                adGroupId = it.id!!,
                adGroup = it!!,
                biddableShowConditionsGrutIds = adGroupBiddableShowConditionsGrutIds.getOrDefault(it.id, emptyList()),
                bidModifiersIds = adGroupsToBidModifiers.getOrDefault(it.id, emptySet()),
            )
        }
    }

    override fun filterObjectsWithParent(
        shard: Int,
        objects: Collection<AdGroupWriterObject>
    ): Collection<AdGroupWriterObject> {
        // сейчас obtainCampaignToOrderIdMapping вызывается 2 раза: тут и при записи в грут
        // когда все кампании будут пролиты, этот можно будет убрать
        // но так как реплицируются не все типы кампаний(mcb, deals, billing_aggregate)
        // их нужно будет отфильтровать
        val campaignIdToOrderId = obtainCampaignToOrderIdMapping(objects.map { it.adGroup!! })
        val (adGroupsWithExistingCampaignsInGrut, adGroupsWithMissingCampaignsInGrut) = objects.partition {
            campaignIdToOrderId.containsKey(
                it.adGroup!!.campaignId
            )
        }
        val notExistingInGrutCampaigns = adGroupsWithMissingCampaignsInGrut.map { it.adGroup!!.campaignId }.distinct()
        if (notExistingInGrutCampaigns.isNotEmpty()) {
            logger.info("${notExistingInGrutCampaigns.size} group campaigns don't exist, directIds: $notExistingInGrutCampaigns")
        }
        val adGroupIdsWithMissingFKs =
            getMissingForeignEntitiesObjectsByAdGroupId(adGroupsWithExistingCampaignsInGrut).keys
        val (notOk, ok) = adGroupsWithExistingCampaignsInGrut.partition { adGroupIdsWithMissingFKs.contains(it.adGroupId) }
        if (notOk.isNotEmpty()) {
            logger.info("Filtered ${notOk.size} adGroups with missing foreign entities: ${notOk.map { it.adGroupId }}")
        }
        return ok
    }

    /**
     * Получить объекты репликации мобильного контента, которых не хватает, чтобы создать связи из групп.
     */
    private fun getMissingMobileContent(adGroups: Collection<AdGroupWriterObject>): Map<Long, List<Mysql2GrutReplicationObject>> {
        val mobileContentAdGroups = adGroups.filter { it.adGroup!!.type == AdGroupType.MOBILE_CONTENT }
        if (mobileContentAdGroups.isEmpty()) {
            return emptyMap()
        }
        val mobileContentIds = mobileContentAdGroups
            .map { (it.adGroup as MobileContentAdGroup).mobileContentId }
            .toSet()
        val existingInGrutMobileContentIds =
            objectApiService.mobileContentGrutDao
                .getExistingObjects(mobileContentIds)
                .toSet()
        val adGroupsWithMissingMobileContent = mobileContentAdGroups
            .filterNot { existingInGrutMobileContentIds.contains((it.adGroup as MobileContentAdGroup).mobileContentId) }
        warnOnMissingForeignEntities(
            "mobileContent",
            adGroupsWithMissingMobileContent,
            existingInGrutMobileContentIds,
        ) { getMinusPhrasesIds(it.adGroup!!).toSet() }
        return adGroupsWithMissingMobileContent.associate {
            it.adGroupId to listOf(
                Mysql2GrutReplicationObject(mobileContentId = (it.adGroup as MobileContentAdGroup).mobileContentId)
            )
        }
    }

    /**
     * Получить объекты репликации минус фраз, которых не хватает, чтобы создать связи из групп.
     */
    private fun getMissingMinusPhrases(adGroups: Collection<AdGroupWriterObject>): Map<Long, List<Mysql2GrutReplicationObject>> {
        val minusPhrasesIds = adGroups.flatMap { getMinusPhrasesIds(it.adGroup!!) }.toSet()
        if (minusPhrasesIds.isEmpty()) return emptyMap()

        val existingMinusPhrases = objectApiService.minusPhrasesGrutDao.getExistingObjects(minusPhrasesIds).toSet()
        val adGroupsWithMissingMinusPhrases = adGroups
            .filterNot { existingMinusPhrases.containsAll(getMinusPhrasesIds(it.adGroup!!)) }
        warnOnMissingForeignEntities(
            "minusPhrases",
            adGroupsWithMissingMinusPhrases,
            existingMinusPhrases,
        ) { getMinusPhrasesIds(it.adGroup!!).toSet() }
        return adGroupsWithMissingMinusPhrases.associate { group ->
            val missingMwIds = getMinusPhrasesIds(group.adGroup!!).toSet()
            group.adGroupId to missingMwIds.map { Mysql2GrutReplicationObject(minusPhraseId = it) }
        }
    }

    /**
     * Получить объекты репликации корректировок, которых не хватает, чтобы создать связи группам.
     */
    private fun getMissingBidModifiers(adGroups: Collection<AdGroupWriterObject>): Map<Long, List<Mysql2GrutReplicationObject>> {
        val bidModifiersIds = adGroups.flatMap { it.bidModifiersIds }.toSet()
        if (bidModifiersIds.isEmpty()) return emptyMap()
        val existingBidModifiers = objectApiService.bidModifierGrutApi
            .getExistingObjects(bidModifiersIds.map { it.grutId })
            .toSet()
        val adGroupsWithMissingBidModifiers = adGroups.filterNot { g ->
            existingBidModifiers.containsAll(
                g.bidModifiersIds.map { it.grutId }
            )
        }
        warnOnMissingForeignEntities(
            "bidModifiers",
            adGroupsWithMissingBidModifiers,
            existingBidModifiers,
        ) { g -> g.bidModifiersIds.map { it.grutId }.toSet() }

        return adGroupsWithMissingBidModifiers
            .associate { group ->
                val missingBidModifiers = group.bidModifiersIds
                    .filterNot { existingBidModifiers.contains(it.grutId) }
                group.adGroupId to missingBidModifiers
                    .map {
                        Mysql2GrutReplicationObject(
                            bidModifierId = it.hierarchicalMultiplierId,
                            bidModifierTableType = BidModifierTableType.PARENT
                        )
                    }
            }
    }

    /**
     * Получить список объектов репликации BiddableShowConditions, которых не хватает в GrUT, чтобы создать связи группам
     */
    private fun getMissingBiddableShowConditions(adGroups: Collection<AdGroupWriterObject>): Map<Long, List<Mysql2GrutReplicationObject>> {
        val allBiddableShowConditionIds = adGroups.flatMap { it.biddableShowConditionsGrutIds }
        val allGrutIds = allBiddableShowConditionIds.map { it.grutId }
        val existingInGrutIds =
            objectApiService.biddableShowConditionReplicationGrutDao
                .getExistingBiddableShowConditionsParallel(allGrutIds)
                .toSet()
        val adGroupsWithMissingConditions = adGroups
            .filterNot { id -> existingInGrutIds.containsAll(id.biddableShowConditionsGrutIds.map { it.grutId }) }
        warnOnMissingForeignEntities(
            "biddableShowConditions",
            adGroupsWithMissingConditions, existingInGrutIds
        )
        { it.biddableShowConditionsGrutIds.map { it.grutId }.toSet() }

        return adGroupsWithMissingConditions.associate { group ->
            val missingFKs = group.biddableShowConditionsGrutIds.filterNot { existingInGrutIds.contains(it.grutId) }
            group.adGroupId to missingFKs.map {
                Mysql2GrutReplicationObject(
                    biddableShowConditionType = BiddableShowConditionReplicationWriter.REVERSE_MAPPING[it.type],
                    biddableShowConditionId = BiddableShowConditionGrutApi.grutIdToDirectId(it.type, it.grutId),
                )
            }
        }
    }

    private fun warnOnMissingForeignEntities(
        entityName: String,
        adGroupsWithMissingFk: List<AdGroupWriterObject>,
        existingFkIds: Set<Long>,
        fkExtractor: (group: AdGroupWriterObject) -> Set<Long>,
    ) {
        if (adGroupsWithMissingFk.isNotEmpty()) {
            for (adGroup in adGroupsWithMissingFk) {
                val missingFkIds = fkExtractor(adGroup).minus(existingFkIds)
                logger.warn("FK missing: pid ${adGroup.adGroupId}, $entityName: $missingFkIds")
            }
        }
    }

    override fun getMissingForeignEntitiesObjects(
        shard: Int,
        objects: Collection<AdGroupWriterObject>
    ): Collection<Mysql2GrutReplicationObject> {
        // Если включена фильтрация объектов с отсутствующими внешними ключами, то создание таких сущностей не делаем
        if (filterForeignEntities.getOrDefault(true)) {
            return listOf()
        }
        return getMissingForeignEntitiesObjectsByAdGroupId(objects).flatMap { it.value }
    }

    private fun getMissingForeignEntitiesObjectsByAdGroupId(objects: Collection<AdGroupWriterObject>): Map<Long, Collection<Mysql2GrutReplicationObject>> {
        if (objects.isEmpty()) {
            return emptyMap()
        }
        val adGroupsWithMissingBidModifiers = getMissingBidModifiers(objects)
        val adGroupsWithMissingMobileContent = getMissingMobileContent(objects)
        val adGroupsWithMissingMinusPhrases = getMissingMinusPhrases(objects)
        val adGroupsWithMissingBiddableShowConditions = getMissingBiddableShowConditions(objects)
        val keys = adGroupsWithMissingBidModifiers.keys
            .plus(adGroupsWithMissingMobileContent.keys)
            .plus(adGroupsWithMissingMinusPhrases.keys)
            .plus(adGroupsWithMissingBiddableShowConditions.keys)
        return keys.associateWith {
            adGroupsWithMissingBidModifiers[it].orEmpty()
                .plus(adGroupsWithMissingMobileContent[it].orEmpty())
                .plus(adGroupsWithMissingMinusPhrases[it].orEmpty())
                .plus(adGroupsWithMissingBiddableShowConditions[it].orEmpty())
        }
    }

    override fun getNotExistingInMysqlObjects(
        shard: Int,
        objects: Collection<AdGroupWriterObject>
    ): Collection<AdGroupWriterObject> {
        val adGroupIds = objects.map { it.adGroupId }.toSet()
        val existingAdGroupIds = adGroupRepository.getExistingAdGroupIds(shard, adGroupIds)
        return objects.filterNot { existingAdGroupIds.contains(it.adGroupId) }
    }

    override fun writeObjectsToGrut(shard: Int, objects: Collection<AdGroupWriterObject>) {
        val adGroupIds = objects.map { it.adGroupId }
        val adGroupToTargeting = getAdGroupToTargetingMapping(shard, adGroupIds)
        val cryptaSegments = cryptaSegmentsCache.getCryptaSegments()

        val campaignIdToOrderId = obtainCampaignToOrderIdMapping(objects.map { it.adGroup!! })
        val adGroupsToUpdate = objects.map { group ->
            convertAdGroup(
                group.adGroup!!,
                campaignIdToOrderId[group.adGroup.campaignId]!!,
                adGroupToTargeting.getOrDefault(group.adGroupId, emptyList()),
                group.biddableShowConditionsGrutIds.map { it.grutId },
                cryptaSegments,
                group.bidModifiersIds.map { it.grutId },
            )
        }

        val biddableShowConditionFKCount = adGroupsToUpdate.sumOf { it.biddableShowConditionsIds.size }
        if(biddableShowConditionFKCount > 10_000) {
            logger.warn("Too many adGroups biddable show conditions fk count: $biddableShowConditionFKCount")
        }

        if (asyncUpdate.getOrDefault(false)) {
            objectApiService.adGroupGrutDao.createOrUpdateAdGroupsParallel(adGroupsToUpdate)
        } else {
            objectApiService.adGroupGrutDao.createOrUpdateAdGroups(adGroupsToUpdate)
        }
    }

    override fun deleteObjectsInGrut(objects: Collection<AdGroupWriterObject>) {
        objectApiService.adGroupGrutDao.deleteObjects(objects.map { it.adGroupId })
    }

    private fun getAdGroupToTargetingMapping(
        shard: Int,
        adGroupIds: List<Long>
    ): Map<Long, List<AdGroupAdditionalTargeting>> {
        return targetingRepository.getByAdGroupIds(shard, adGroupIds)
            .groupBy { (it.adGroupId) }
    }

    private fun obtainCampaignToOrderIdMapping(adGroups: Collection<AdGroup>): Map<Long, Long?> {
        val campaignIds = adGroups.map { it.campaignId }.distinct()
        return objectApiService.campaignGrutDao.getCampaignIdsByDirectIds(campaignIds)
    }
}
