package ru.yandex.direct.logicprocessor.processors.mysql2grut.replicationwriter

import org.slf4j.Logger
import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Value
import org.springframework.context.annotation.Lazy
import org.springframework.stereotype.Component
import ru.yandex.direct.common.db.PpcPropertiesSupport
import ru.yandex.direct.common.db.PpcProperty
import ru.yandex.direct.common.db.PpcPropertyNames
import ru.yandex.direct.common.util.DirectThreadPoolExecutor
import ru.yandex.direct.core.entity.banner.model.Banner
import ru.yandex.direct.core.entity.banner.model.BannerStatusModerate
import ru.yandex.direct.core.entity.banner.model.BannerWithBannerImage
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields
import ru.yandex.direct.core.entity.banner.model.ContentPromotionBanner
import ru.yandex.direct.core.entity.banner.model.CpcVideoBanner
import ru.yandex.direct.core.entity.banner.model.CpmAudioBanner
import ru.yandex.direct.core.entity.banner.model.CpmBanner
import ru.yandex.direct.core.entity.banner.model.CpmIndoorBanner
import ru.yandex.direct.core.entity.banner.model.CpmOutdoorBanner
import ru.yandex.direct.core.entity.banner.model.DynamicBanner
import ru.yandex.direct.core.entity.banner.model.ImageBanner
import ru.yandex.direct.core.entity.banner.model.InternalBanner
import ru.yandex.direct.core.entity.banner.model.McBanner
import ru.yandex.direct.core.entity.banner.model.MobileAppBanner
import ru.yandex.direct.core.entity.banner.model.PerformanceBanner
import ru.yandex.direct.core.entity.banner.model.PerformanceBannerMain
import ru.yandex.direct.core.entity.banner.model.TextBanner
import ru.yandex.direct.core.entity.banner.repository.BannerTypedRepository
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository
import ru.yandex.direct.core.entity.feature.service.FeatureService
import ru.yandex.direct.core.entity.image.model.BannerImageFormat
import ru.yandex.direct.core.entity.image.repository.BannerImageFormatRepository
import ru.yandex.direct.core.grut.api.BannerGrut
import ru.yandex.direct.core.grut.api.utils.extractVcardId
import ru.yandex.direct.core.grut.api.utils.waitFutures
import ru.yandex.direct.core.grut.replication.GrutApiService
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.ess.logicobjects.mysql2grut.Mysql2GrutReplicationObject
import ru.yandex.direct.feature.FeatureName
import java.time.Duration
import java.util.concurrent.CompletableFuture
import java.util.concurrent.ExecutorService

data class BannerWriterData(
    val bannerId: Long,
    val banner: BannerWithSystemFields? = null, // not needed for deletions
    val imageFormat: BannerImageFormat? = null,
    var dropVcard: Boolean = false,
    var dropImage: Boolean = false,
    var skipModeration: Boolean = true,
)

@Component
@Lazy   // из-за чтения настройки размера тредпула из конфига, а процессоры используются в oneshot (где настройка не задана)
class BannerReplicationWriter(
    private val objectApiService: GrutApiService,
    private val bannerTypedRepository: BannerTypedRepository,
    private val bannerImageFormatRepository: BannerImageFormatRepository,
    private val campaignRepository: CampaignRepository,
    private val featureService: FeatureService,
    @Value("\${grut_replication.get_banners_threads:1}") private val threadPoolSize: Int,
    ppcPropertiesSupport: PpcPropertiesSupport,
) : BaseReplicationWriter<BannerWriterData>(), AutoCloseable {
    companion object {
        private val bannerLogger = LoggerFactory.getLogger(BannerReplicationWriter::class.java)
        private val bannersDbLoadTimeout = Duration.ofMinutes(2)
        private const val defaultDbLoadChunkSize = 2000
    }

    // выбираем количество тредов примерно по числу vCPU в продакшене jobs
    // очередь делаем побольше, при нашем использованииa не видно разницы блокироваться на постановке задачи
    // или на ожидании выполнения
    private val executorService: ExecutorService =
        DirectThreadPoolExecutor(threadPoolSize, 500, "grut-get-typed-banners")

    private val supportedBannerTypes = setOf(
        TextBanner::class.java, CpmBanner::class.java, DynamicBanner::class.java,
        CpcVideoBanner::class.java, CpmOutdoorBanner::class.java, CpmIndoorBanner::class.java,
        InternalBanner::class.java, CpmAudioBanner::class.java, ContentPromotionBanner::class.java,
        PerformanceBannerMain::class.java,
        PerformanceBanner::class.java, ImageBanner::class.java, MobileAppBanner::class.java, McBanner::class.java
    )
    private val property: PpcProperty<Set<Int>> =
        ppcPropertiesSupport.get(PpcPropertyNames.GRUT_SKIP_BANNERS_REPLICATION_SHARDS, Duration.ofSeconds(20))

    private val asyncUpdate =
        ppcPropertiesSupport.get(PpcPropertyNames.GRUT_CAMP_REPL_ASYNC_UPDATE, Duration.ofSeconds(20))
    private val loadBannersChunkSize =
        ppcPropertiesSupport.get(PpcPropertyNames.GRUT_REPLICATION_BANNERS_DB_LOAD_CHUNK_SIZE, Duration.ofSeconds(60))

    override val logger: Logger = bannerLogger

    override fun getLogicObjectsToWrite(
        shard: Int,
        logicObjects: Collection<Mysql2GrutReplicationObject>
    ): ObjectsForUpdateAndDelete<BannerWriterData> {
        val (logicObjectsForUpdate, logicObjectsForDelete) = logicObjects
            .filter { it.bannerId != null }
            .partition { !it.isDeleted }
        val bannerIdsForUpdate = logicObjectsForUpdate
            .map { it.bannerId!! }
            .distinct()

        val banners = getBannersAsync(shard, bannerIdsForUpdate)
            .filter { supportedBannerTypes.contains(it::class.java) }
            .map { it as BannerWithSystemFields }

        val imageHashes = banners
            .filter { isModerated(it) }
            .mapNotNull { it as? BannerWithBannerImage }
            // сэкономим и не будем грузить формат для тех картинок, которые точно не пошлем
            .filter { objectApiService.bannerReplicationApiService.canSendImageId(it) }
            .mapNotNull { it.imageHash }
            .distinct()
            .toList()

        val bannerImageFormats = getBannerImagesFormat(shard, imageHashes)

        val enabledModerationInGrutByBannerId = getEnabledModerationInGrutByBannerId(shard, banners)

        // реплицируем только промодерированные баннеры, чтобы выставить им статус skipModeration
        val moderatedBannersForUpdate = banners
            .filter { isModerated(it) }
            .map {
                BannerWriterData(
                    bannerId = it.id,
                    banner = it,
                    imageFormat = bannerImageFormats[(it as? BannerWithBannerImage)?.imageHash],
                    dropImage = objectApiService.bannerReplicationApiService.canSendImageId(it).not(),
                    // Баннеры, для которых нет записей в enabledModerationInGrutByBannerId, отфильтруются
                    // в методе filterObjectsWithParent
                    skipModeration = enabledModerationInGrutByBannerId.getOrDefault(it.id, false).not(),
                )
            }
            .toList()

        val bannerIdsToDelete =
            logicObjectsForDelete.map { it.bannerId!! }

        val objectForDelete = bannerIdsToDelete
            .distinct()
            .map {
                BannerWriterData(bannerId = it)
            }

        return ObjectsForUpdateAndDelete(moderatedBannersForUpdate, objectForDelete)
    }

    private fun isModerated(it: BannerWithSystemFields) =
        it.statusModerate == BannerStatusModerate.YES || it.statusModerate == BannerStatusModerate.NO

    override fun filterObjectsWithParent(
        shard: Int,
        objects: Collection<BannerWriterData>
    ): Collection<BannerWriterData> {
        val adGroupIds = objects.map { it.banner!!.adGroupId }.distinct()
        val existingAdGroups = objectApiService.adGroupGrutDao.getExistingAdGroupsParallel(adGroupIds)
            .toSet()

        // проверяем существование визикти.
        // в https://st.yandex-team.ru/DIRECT-168208#627ccc7876d7e3003a46c416 договорились,
        // что можно делать вид отсутствия визитки в баннере (реплицировать vcard_id = 0)

        val vcardIds = objects.mapNotNull { extractVcardId(it.banner!!) }
            .distinct()
        val existingVcards = objectApiService.vcardGrutDao.getExistingObjects(vcardIds).toSet()

        val result = objects
            .filter { existingAdGroups.contains(it.banner!!.adGroupId) }
            .onEach { obj ->
                extractVcardId(obj.banner!!)?.takeUnless { existingVcards.contains(it) }?.let {
                    obj.dropVcard = true
                    logger.warn(
                        "Banner ${obj.banner.id} has vcard_id=$it which not exists in GrUT. " +
                            "Relation with this vcard will be removed from GrUT."
                    )
                }
            }

        if (objects.size != result.size) {
            logger.info("${objects.size - result.size} banners skipped due to absent foreign entities")
        }
        return result
    }

    override fun getNotExistingInMysqlObjects(
        shard: Int,
        objects: Collection<BannerWriterData>
    ): Collection<BannerWriterData> {
        val bannerIds = objects.map { it.bannerId }.toSet()
        val existingBannersMap = bannerTypedRepository.getSafely(shard, bannerIds, BannerWithSystemFields::class.java)
            .associateBy { it.id }
        return objects.filter { it.bannerId !in existingBannersMap }
    }

    override fun writeObjectsToGrut(shard: Int, objects: Collection<BannerWriterData>) {
        val orderIdsByCampaignIds =
            objectApiService.campaignGrutDao.getCampaignIdsByDirectIds(objects.map { it.banner!!.campaignId }
                .distinct())
        val objectsToWrite = objects.map {
            if (it.dropImage) {
                logger.warn("Banner ${it.banner?.id} has banner_image but will be sent without it.")
            }
            BannerGrut(
                banner = it.banner!!,
                orderId = orderIdsByCampaignIds[it.banner.campaignId]!!,
                sitelinkSet = null,
                turbolandingsById = mapOf(),
                bannerImageFormat = it.imageFormat,
                dropVcard = it.dropVcard,
                dropImage = it.dropImage,
                skipModeration = it.skipModeration,
            )
        }
        if (asyncUpdate.getOrDefault(false)) {
            objectApiService.bannerReplicationApiService
                .createOrUpdateBannersParallel(objectsToWrite)
        } else {
            objectApiService.bannerReplicationApiService
                .createOrUpdateBanners(objectsToWrite)
        }
    }

    override fun deleteObjectsInGrut(objects: Collection<BannerWriterData>) {
        objectApiService.bannerReplicationApiService.deleteBannersByDirectIds(objects.map { it.bannerId })
    }

    override fun isDisabledInShard(shard: Int): Boolean {
        return property.get()
            .orEmpty()
            .contains(shard)
    }

    private fun getBannerSync(shard: Int, bannerIds: List<Long>): Sequence<Banner> {
        return bannerTypedRepository.getTyped(shard, bannerIds)
            .asSequence()
    }

    private fun getBannerImagesFormat(shard: Int, imageHashes: List<String>): Map<String, BannerImageFormat> {
        if (imageHashes.isEmpty()) {
            return emptyMap()
        }

        val chunkSize = loadBannersChunkSize.getOrDefault(defaultDbLoadChunkSize)
        val completableFutures = imageHashes.chunked(chunkSize)
            .map {
                CompletableFuture.supplyAsync(
                    { bannerImageFormatRepository.getBannerImageFormats(shard, it).values },
                    executorService
                )
            }.toTypedArray()
        waitFutures(completableFutures, bannersDbLoadTimeout)
        return completableFutures.asSequence()
            .map { it.get() }
            .flatten()
            .associateBy { it.imageHash }
    }

    private fun getBannersAsync(shard: Int, bannerIds: List<Long>): Sequence<Banner> {
        val chunkSize = loadBannersChunkSize.getOrDefault(defaultDbLoadChunkSize)
        if (bannerIds.size <= 1.05 * chunkSize) {
            // TODO DIRECT-168914: убрать возможность синхронного чтения баннеров из базы
            // когда убедимся, что асинхронно все хорошо работает
            return getBannerSync(shard, bannerIds)
        }

        val completableFutures = bannerIds.chunked(chunkSize)
            .map {
                CompletableFuture.supplyAsync({ bannerTypedRepository.getTyped(shard, it) }, executorService)
            }
            .toTypedArray()

        waitFutures(completableFutures, bannersDbLoadTimeout)
        return completableFutures.asSequence()
            .map { it.get() }
            .flatten()
    }

    /**
     * Возвращает признак включенности модерации в груте по bannerId.
     * Получает clientIds по campaignId баннеров, далее для них получает включенность фичи:
     * FeatureName.MODERATION_BANNERS_IN_GRUT_ENABLED
     */
    private fun getEnabledModerationInGrutByBannerId(
        shard: Int,
        banners: Sequence<BannerWithSystemFields>
    ): Map<Long, Boolean> {
        val bannerIdToCampaignId = banners
            .map { it.id to it.campaignId }
            .toMap()
        val campaignIdToClientId = campaignRepository.getClientIdsForCids(shard, bannerIdToCampaignId.values)
            .mapValues { ClientId.fromLong(it.value) }

        val clientsEnabledModerationInGrut = featureService.isEnabledForClientIdsOnlyFromDb(
            campaignIdToClientId.values.toSet(),
            FeatureName.MODERATION_BANNERS_IN_GRUT_ENABLED.getName()
        )

        return bannerIdToCampaignId
            .filterValues { campaignIdToClientId.containsKey(it) }
            .mapValues { campaignIdToClientId[it.value!!]!! }
            .mapValues { clientsEnabledModerationInGrut[it.value]!! }
    }

    override fun close() {
        executorService.shutdown()
    }
}
