package ru.yandex.direct.logicprocessor.processors.mysql2grut.replicationwriter

import java.time.Duration
import java.util.concurrent.CompletableFuture
import java.util.concurrent.ExecutorService
import ru.yandex.direct.autobudget.restart.model.CampStrategyRestartData
import ru.yandex.direct.autobudget.restart.model.StrategyDto
import ru.yandex.direct.autobudget.restart.repository.CampRestartData
import ru.yandex.direct.autobudget.restart.repository.RestartTimes
import ru.yandex.direct.autobudget.restart.service.Reason
import ru.yandex.direct.autobudget.restart.service.StrategyState
import ru.yandex.direct.common.db.PpcPropertiesSupport
import ru.yandex.direct.common.db.PpcPropertyNames
import ru.yandex.direct.common.util.DirectThreadPoolExecutor
import ru.yandex.direct.core.entity.additionaltargetings.repository.CampAdditionalTargetingsRepository
import ru.yandex.direct.core.entity.bs.common.service.BsOrderIdCalculator
import ru.yandex.direct.core.entity.campaign.CampaignAutobudgetRestartUtils.getStrategyDto
import ru.yandex.direct.core.entity.campaign.model.CampaignTypeKinds
import ru.yandex.direct.core.entity.campaign.model.CampaignWithMinusKeywords
import ru.yandex.direct.core.entity.campaign.model.CampaignWithPricePackage
import ru.yandex.direct.core.entity.campaign.model.CampaignWithSkadNetwork
import ru.yandex.direct.core.entity.campaign.model.CampaignsPlatform
import ru.yandex.direct.core.entity.campaign.model.CommonCampaign
import ru.yandex.direct.core.entity.campaign.model.StrategyName
import ru.yandex.direct.core.entity.campaign.model.WalletTypedCampaign
import ru.yandex.direct.core.entity.campaign.service.CampaignAutobudgetRestartService
import ru.yandex.direct.core.entity.campaign.service.WalletHasMoneyChecker
import ru.yandex.direct.core.entity.client.model.Client
import ru.yandex.direct.core.entity.metrika.repository.MetrikaCampaignRepository
import ru.yandex.direct.core.entity.mobileapp.service.IosSkAdNetworkSlotManager
import ru.yandex.direct.core.grut.api.AutoBudgetRestartData
import ru.yandex.direct.core.grut.api.CampaignGrutModel
import ru.yandex.direct.core.grut.api.utils.bigDecimalFromGrut
import ru.yandex.direct.core.grut.api.utils.moneyFromGrut
import ru.yandex.direct.core.grut.api.utils.moscowDateTimeFromGrut
import ru.yandex.direct.core.grut.api.utils.moscowLocalDateFromGrut
import ru.yandex.direct.core.grut.api.utils.waitFutures
import ru.yandex.direct.core.grut.replication.GrutApiService
import ru.yandex.direct.core.mysql2grut.repository.MinusPhrase
import ru.yandex.direct.ess.logicobjects.mysql2grut.Mysql2GrutReplicationObject
import ru.yandex.direct.logicprocessor.processors.bsexport.utils.SupportedCampaignsService
import ru.yandex.direct.logicprocessor.processors.mysql2grut.replicationwriter.BaseCampaignReplicationWriter.Companion.BEGUN_AGENCY_IDS
import ru.yandex.direct.logicprocessor.processors.mysql2grut.replicationwriter.BaseCampaignReplicationWriter.Companion.TEST_BEGUN_CAMPAIGN_ID
import ru.yandex.direct.logicprocessor.processors.mysql2grut.repository.CampOrderTypesRepository
import ru.yandex.direct.mysql2grut.enummappers.CampaignEnumMappers
import ru.yandex.direct.mysql2grut.enummappers.CampaignEnumMappers.Companion.autobudgetRestartReasonFromGrut
import ru.yandex.direct.mysql2grut.enummappers.CampaignEnumMappers.Companion.fromGrutPlatform
import ru.yandex.direct.mysql2grut.enummappers.CampaignEnumMappers.Companion.fromGrutStrategyType
import ru.yandex.grut.objects.proto.CampaignV2
import ru.yandex.grut.objects.proto.client.Schema

data class CampaignWriterData(
    val campaignId: Long,
    val campaign: CommonCampaign? = null, //not needed for deletions
    val clientId: Long? = null, // not needed for deletions
)

abstract class BaseCampaignReplicationWriter(
    private val campaignTypedRepository: SupportedCampaignsService,
    private val objectApiService: GrutApiService,
    private val timezoneCache: TimezoneCache,
    private val campOrderTypesRepository: CampOrderTypesRepository,
    private val autobudgetRestartService: CampaignAutobudgetRestartService,
    private val metrikaCampaignRepository: MetrikaCampaignRepository,
    private val bsOrderIdCalculator: BsOrderIdCalculator,
    private val walletHasMoneyChecker: WalletHasMoneyChecker,
    private val campaignAdditionalTargetingsRepository: CampAdditionalTargetingsRepository,
    private val calcAutobudgetRestartsThreadPoolSize: Int,
    private val iosSkAdNetworkSlotManager: IosSkAdNetworkSlotManager,
    ppcPropertiesSupport: PpcPropertiesSupport,
) : BaseReplicationWriter<CampaignWriterData>() {
    companion object {
        /**
         * AgencyID Бегун'а. Получился из старого спискаа по принзнаку is_begun.
         * https://a.yandex-team.ru/arc/trunk/arcadia/direct/perl/protected/API/Settings.pm?rev=6219335#L187
         */
        val BEGUN_AGENCY_IDS = setOf(
            1647047L,  // begun-2011
            1618235L // begun-test-agency
        )
        const val TEST_BEGUN_CAMPAIGN_ID = 2834265L
        const val AUTOBUDGET_CALCULATE_CHUNK = 5_000
        private const val defaultAutobudgetRestartChunkSize = 10_000
        private val calcAutobudgetRestartTimeout = Duration.ofMinutes(2)
    }

    private val useOnlyMysqlTableAutobudgetRestartProperty = ppcPropertiesSupport.get(
        PpcPropertyNames.GRUT_USE_MYSQL_TABLE_FOR_AUTOBUDGET_RESTART_REPLICATION,
        Duration.ofSeconds(20)
    )
    private val calcAutobudgetRestartChunkSize =
        ppcPropertiesSupport.get(
            PpcPropertyNames.GRUT_REPLICATION_CALC_AUTOBUDGET_RESTART_CHUNK_SIZE,
            Duration.ofSeconds(60)
        )

    private val filterForeignEntities =
        ppcPropertiesSupport.get(PpcPropertyNames.CAMPAIGN_REPLICATION_FILTER_WITHOUT_FOREIGN)
    private val asyncUpdate =
        ppcPropertiesSupport.get(PpcPropertyNames.GRUT_CAMP_REPL_ASYNC_UPDATE, Duration.ofSeconds(20))
    private val calcAutobudgetRestartPercent =
        ppcPropertiesSupport.get(PpcPropertyNames.GRUT_CAMP_REPL_CALC_AUTOBUDGET_RESTART_PERCENT)

    // выбираем количество тредов примерно по числу vCPU в продакшене jobs
    // очередь делаем побольше, при нашем использованииa не видно разницы блокироваться на постановке задачи
    // или на ожидании выполнения
    private val executorService: ExecutorService =
        DirectThreadPoolExecutor(calcAutobudgetRestartsThreadPoolSize, 500, "calc-autobudget-restart")

    abstract fun getObjectsForDelete(objects: Collection<Mysql2GrutReplicationObject>): List<Mysql2GrutReplicationObject>
    abstract fun getCampaigns(
        shard: Int,
        campaignIds: Collection<Long>
    ): List<CommonCampaign>

    override fun isDisabledInShard(shard: Int): Boolean {
        return false
    }

    override fun getLogicObjectsToWrite(
        shard: Int,
        logicObjects: Collection<Mysql2GrutReplicationObject>
    ): ObjectsForUpdateAndDelete<CampaignWriterData> {
        val logicObjectsForUpdate = logicObjects
            .filter { it.campaignId != null }
            .filter { !it.isDeleted }

        val logicObjectsForDelete = getObjectsForDelete(logicObjects)
        val clientsWithAutoOverDraftLimitChanged = logicObjects
            .mapNotNull { it.clientAutoOverdraftLimitChanged }
            .distinct()

        val campaignsWithAutoOverdraftLimitChanged =
            campaignTypedRepository.getClientsCampaignIds(shard, clientsWithAutoOverDraftLimitChanged)
        val campaignIdsForUpdate = logicObjectsForUpdate
            .map { it.campaignId!! }
            .union(campaignsWithAutoOverdraftLimitChanged)
            .distinct()

        val campaignsFromDb = getCampaigns(shard, campaignIdsForUpdate)

        val objectForUpdate = campaignsFromDb
            .filter { !CampaignEnumMappers.CAMPAIGN_TYPE_BLACKLIST.contains(it.type) }
            .map {
                CampaignWriterData(campaignId = it.id, campaign = it, clientId = it.clientId)
            }
            .toList()

        val objectForDelete = logicObjectsForDelete
            .map {
                CampaignWriterData(campaignId = it.campaignId!!)
            }

        return ObjectsForUpdateAndDelete(objectForUpdate, objectForDelete.distinct())
    }

    abstract fun filterSpecificEntities(
        shard: Int,
        objects: Collection<CampaignWriterData>
    ): Collection<CampaignWriterData>

    override fun filterObjectsWithParent(
        shard: Int,
        objects: Collection<CampaignWriterData>
    ): Collection<CampaignWriterData> {
        if (!filterForeignEntities.getOrDefault(true)) {
            return objects
        }
        val gotClientIds = objects.map { it.clientId!! }.distinct()
        val gotAgencyClientIds = objects.mapNotNull { it.campaign!!.agencyId }.filter { it != 0L }.distinct()
        val existingClients =
            objectApiService.clientGrutDao.getExistingObjects((gotClientIds + gotAgencyClientIds).distinct()).toSet()

        val commonResult = objects.filter {
            existingClients.contains(it.clientId)
                && (it.campaign!!.agencyId == null || it.campaign.agencyId == 0L || existingClients.contains(it.campaign.agencyId))
        }.associateBy { it.campaignId }

        val specificResult = filterSpecificEntities(shard, objects).associateBy { it.campaignId }
        val result = commonResult.filter { specificResult.containsKey(it.key) }
            .map { it.value }
        if (objects.size != result.size) {
            logger.info("${objects - result} campaigns skipped due to absent foreign entities")
        }
        return result
    }

    abstract fun getMissingSpecificForeignEntities(
        shard: Int,
        objects: Collection<CampaignWriterData>
    ): Collection<Mysql2GrutReplicationObject>;

    override fun getMissingForeignEntitiesObjects(
        shard: Int,
        objects: Collection<CampaignWriterData>
    ): Collection<Mysql2GrutReplicationObject> {
        // Если включена фильтрация объектов с отсутствующими внешними ключами, то создание таких сущностей не делаем
        if (filterForeignEntities.getOrDefault(true)) {
            return listOf()
        }
        return getMissingSpecificForeignEntities(shard, objects)
    }

    override fun writeObjectsToGrut(shard: Int, objects: Collection<CampaignWriterData>) {
        val campaignsToWrite = objects.map { it.campaign!! }
        if (campaignsToWrite.isEmpty()) {
            return
        }
        val grutCampaignsByDirectIdMap = objectApiService.campaignGrutDao.getCampaignsByDirectIds(
            campaignsToWrite.map { it.id },
            listOf(
                "/meta/direct_id",
                "/meta/order_type",
                "/spec/minus_phrases_ids",
                "/spec/platform",
                "/spec/start_date",
                "/spec/end_date",
                "/spec/strategy",
                "/spec/status",
                "/spec/flags/enable_cpc_hold",
                "/spec/time_target_str",
                "/spec/autobudget_restart"
            )
        ).associateBy { it.meta.directId }

        val campaignDirectWalletIds = objects.mapNotNull { it.campaign!!.walletId }.filter { it != 0L }.distinct()
        val walletsMap =
            campaignTypedRepository.getSafely(shard, campaignDirectWalletIds, WalletTypedCampaign::class.java)
                .associateBy { it.id }
        val directWalletIdToGrutId =
            walletsMap.values.associate { it.id to if (it.orderId ?: 0L == 0L) bsOrderIdCalculator.calculateOrderId(it.id) else it.orderId!! }

        logger.info("Handle campaigns minus phrases")
        val campaignIdToMinusPhraseId = handleCampaignMinusPhrases(campaignsToWrite, grutCampaignsByDirectIdMap.values)
        logger.info("Handle campaigns order types")
        val campaignIdToOrderType = handleOrderType(shard, campaignsToWrite)

        val chunkSize = calcAutobudgetRestartChunkSize.getOrDefault(defaultAutobudgetRestartChunkSize)
        val completableFutures = campaignsToWrite.chunked(chunkSize)
            .map {
                val chunkedWallets = it.mapNotNull { camp -> walletsMap[camp.walletId] }
                val grutCampaignsChunked = it.mapNotNull { camp -> grutCampaignsByDirectIdMap[camp.id] }
                CompletableFuture.supplyAsync(
                    { calculateAutoBudgetRestart(shard, it, chunkedWallets, grutCampaignsChunked) },
                    executorService
                )
            }
            .toTypedArray()

        waitFutures(completableFutures, calcAutobudgetRestartTimeout)
        val autoBudgetRestartsMap = completableFutures
            .map { it.get() }
            .flatMap { it.map { elem -> elem.key to elem.value } }
            .toMap()

        val additionalTargetings =
            campaignAdditionalTargetingsRepository.findByCids(shard, campaignsToWrite.map { it.id })
                .groupBy { it.cid }

        val campaignWithSkadNetwork = campaignsToWrite.mapNotNull { it as? CampaignWithSkadNetwork }.map { it.id }
        val campaignsToAllocatedSlots =
            iosSkAdNetworkSlotManager.getAllocatedSlotsByCampaignIds(campaignWithSkadNetwork)
                .associateBy { it.campaignId }

        val campaignsWithPricePackages = campaignsToWrite
            .mapNotNull { it as? CampaignWithPricePackage }
            .filter { it.pricePackageId ?: 0L != 0L }

        val pricePackagesIdsByCampaigns =
            campaignsWithPricePackages.associate { it.id to it.pricePackageId }

        val campaignGrutModels = campaignsToWrite.map {
            val walletGrutId = if (it.walletId ?: 0L == 0L) 0L else directWalletIdToGrutId[it.walletId]!!
            CampaignGrutModel(
                campaign = it,
                minusPhraseId = campaignIdToMinusPhraseId[it.id],
                walletId = walletGrutId,
                timeZone = timezoneCache[it.timeZoneId],
                autoBudgetRestart = autoBudgetRestartsMap[it.id],
                orderType = campaignIdToOrderType[it.id],
                additionalTargetings = additionalTargetings[it.id],
                skAdNetworkSlot = campaignsToAllocatedSlots[it.id],
                pricePackageId = pricePackagesIdsByCampaigns[it.id]
            )
        }

        if (asyncUpdate.getOrDefault(false)) {
            logger.info("Start writing campaigns to GRuT")
            objectApiService.campaignGrutDao.createOrUpdateCampaignsParallel(campaignGrutModels)
        } else {
            logger.info("Start writing campaigns to GRuT")
            objectApiService.campaignGrutDao.createOrUpdateCampaigns(campaignGrutModels)
        }
    }

    /**
     * Берем значение, сохраненное старым транспортом в таблице CAMP_ORDER_TYPES
     * Здесь не рассчитываем и не сохраняем, так как есть в случае, если на кампании еще нет денег, она не отправится старым транспортом,
     * но отправится здесь, при этом orderType не учтет поле paidBySertificate так как оно проставляется после начисления денег на кампанию,
     * а так как orderType иммутабельный, то для таких кампаний он всегда будет 1, хотя старым транспотом получалось бы 6(внутренняя реклама)
     */
    private fun handleOrderType(
        shard: Int,
        directCampaigns: Collection<CommonCampaign>
    ): Map<Long, Int?> {

        val calculatedOrderTypes =
            campOrderTypesRepository.getCampsOrderType(shard, directCampaigns.map { it.id })
        return directCampaigns
            .associate { it.id to calculatedOrderTypes[it.id] }
    }

    /**
     * Расчет рестарта автобюджета. Сравнивает настройки кампании до изменений и после, на основе этого считает рестарт
     * Получает информацию о кампании из GRuT
     * Если в GRuT кампании нет, значит возможны два варианта:
     * а) кампания еще не реплицировалась, но в mysql она уже давно
     * б) кампания только создана
     *
     * В первом случае прошлое состояние будет взято из таблицы camp_autobudget_restart, куда сохраняет старый транспорт
     * После репликации всех кампаний в продакшене этот код можно будет убрать
     *
     * Так как не все поля для рассчета сейчас есть в GRuT(денежных полей нет),
     * то в GRuT еще сохраняется рассчитаное на основе этой полей поле hasMoney
     */
    private fun calculateAutoBudgetRestart(
        shard: Int,
        allCampaigns: List<CommonCampaign>,
        wallets: List<WalletTypedCampaign>,
        grutCampaigns: Collection<Schema.TCampaignV2>
    ): Map<Long, AutoBudgetRestartData> {
        logger.info("Start calculate autobudget restart")
        val campaignHasMoneyMap = walletHasMoneyChecker.calcHasMoney(shard, allCampaigns, wallets)

        val campaignIdsWithCombinedGoals =
            metrikaCampaignRepository.getCampaignIdsWithCombinedGoals(shard, allCampaigns.map { it.id })
        val grutCampaignsMap = grutCampaigns.associateBy { it.meta.directId }

        val (campaignsWithGrutAutobudgetRestart, campaignWithoutGrutAutobudgetRestart) = allCampaigns
            .partition { grutCampaignsMap[it.id] != null && grutCampaignsMap[it.id]!!.spec.hasAutobudgetRestart() }

        val allNewRestartData: MutableList<CampStrategyRestartData> = mutableListOf()
        val allOldRestartData: MutableList<CampRestartData> = mutableListOf()

        // для кампаний, которые есть в груте, заполняем прошлое состояние оттуда
        campaignsWithGrutAutobudgetRestart.forEach {
            allNewRestartData.add(
                CampStrategyRestartData(
                    it.id,
                    getStrategyDto(it, campaignHasMoneyMap[it.id]!!, campaignIdsWithCombinedGoals.contains(it.id))
                )
            )

            // на будущее, когда все кампании будут в груте, а в старом транспорте реастар автобюджета вычисляться не будет,
            // нужно будет брать прошлую стратегию кампанию из грута
            // пока всегда берем из таблицы camp_autobudget_restart
            allOldRestartData.add(getStrategyDtoForGrutCampaign(grutCampaignsMap[it.id]!!))
        }

        campaignWithoutGrutAutobudgetRestart.forEach {
            allNewRestartData.add(
                CampStrategyRestartData(
                    it.id,
                    getStrategyDto(it, campaignHasMoneyMap[it.id]!!, campaignIdsWithCombinedGoals.contains(it.id))
                )
            )
        }

        val percent = calcAutobudgetRestartPercent.getOrDefault(0).toLong()
        // так как начинаем совместно со старым транспортом использовать таблицу camp_autobudget_restart на запись,
        // то хочется иметь возможность отключить новый функционал, если что то пойдет не так
        val filteredRestartData = allNewRestartData.filter {
            it.cid % 100 < percent
        }.toMutableList()

        val result = filteredRestartData.chunked(AUTOBUDGET_CALCULATE_CHUNK)
            .flatMap {
                autobudgetRestartService.calculateRestartsAndSave(shard, it)
            }

        logger.info("Finish calculate autobudget restart")
        return result.associate {
            it.restartResult.cid to
                AutoBudgetRestartData(
                    restartTime = it.restartResult.restartTime,
                    softRestartTime = it.restartResult.softRestartTime,
                    restartReason = Reason.valueOf(it.restartResult.restartReason),
                    hasMoney = campaignHasMoneyMap[it.restartResult.cid]!!,
                    stopTime = it.state.stopTime,
                )
        }
    }

    private fun getStrategyDtoForGrutCampaign(campaign: Schema.TCampaignV2): CampRestartData {
        val autobudgetRestartData = campaign.spec.autobudgetRestart
        var dto = StrategyDto(
            platform = fromGrutPlatform(campaign.spec.platform)?.let { CampaignsPlatform.toSource(it) }?.literal,
            timeTarget = campaign.spec.timeTargetStr,
            startTime = moscowLocalDateFromGrut(campaign.spec.startDate),
            finishTime = if (campaign.spec.hasEndDate()) moscowLocalDateFromGrut(campaign.spec.endDate) else null,
            enableCpcHold = campaign.spec.flags.enableCpcHold,
            statusShow = campaign.spec.status == CampaignV2.ECampaignStatus.CST_ACTIVE.number,
            strategy = campaign.spec.strategy?.type?.let { StrategyName.toSource(fromGrutStrategyType(it))?.literal }
                ?: "default",
            hasMoney = autobudgetRestartData.hasMoney,
        )

        if (campaign.spec.hasStrategy()) {
            val strategy = campaign.spec.strategy
            dto = dto.copy(
                payForConversion = strategy.payForConversion,
                goalId = if (strategy.hasGoalId()) strategy.goalId else null,
                roiCoef = if (strategy.hasRoiCoef()) bigDecimalFromGrut(strategy.roiCoef) else null,
                avgCpm = if (strategy.hasAvgCpm()) moneyFromGrut(strategy.avgCpm) else null,
                avgBid = if (strategy.hasAvgBid()) moneyFromGrut(strategy.avgBid) else null,
                avgCpa = if (strategy.hasAvgCpa()) moneyFromGrut(strategy.avgCpa) else null,
                avgCpv = if (strategy.hasAvgCpv()) moneyFromGrut(strategy.avgCpv) else null,
                strategyStart = if (strategy.hasStartDate()) moscowLocalDateFromGrut(strategy.startDate) else null,
                strategyFinish = if (strategy.hasFinishDate()) moscowLocalDateFromGrut(strategy.finishDate) else null,
                autoBudgetSum = if (strategy.hasBudget()) moneyFromGrut(strategy.budget) else null,
            )
        }
        val restartTimes = RestartTimes(
            restartTime = moscowDateTimeFromGrut(autobudgetRestartData.restartTime),
            softRestartTime = moscowDateTimeFromGrut(autobudgetRestartData.softRestartTime),
            restartReason = autobudgetRestartReasonFromGrut(autobudgetRestartData.restartReason)!!.name
        )
        val stopTime =
            if (autobudgetRestartData.hasStopTime()) moscowDateTimeFromGrut(autobudgetRestartData.stopTime) else null
        val strategyState = StrategyState(stopTime = stopTime)
        return CampRestartData(
            cid = campaign.meta.directId,
            strategyData = dto,
            times = restartTimes,
            state = strategyState
        )
    }

    /**
     * Обработка минус фраз на кампании
     * В директе минус фразы хранятся в самой кампании, а в груте лежат в отдельной таблице, а в кампании хранится только id
     * Этот метод:
     * - удаляет минус фразы из таблицы minus_phrases, если на кампании минус фразы удалились
     * - обновляет минус фразы для тех кампаний, на которых они изменились
     * - добавляет новые минус фразы, если раньше их на кампании не было
     *
     * Возвращает мапу campignId->minusPhraseId с актуальными id набора минус фраз
     */
    private fun handleCampaignMinusPhrases(
        allCampaigns: List<CommonCampaign>,
        grutCampaigns: Collection<Schema.TCampaignV2>
    ): Map<Long, Long> {
        val campaigns = allCampaigns.filterIsInstance<CampaignWithMinusKeywords>()
        if (campaigns.isEmpty()) {
            return mapOf()
        }
        // получаем текущие id минус фраз кампаний
        val minusPhrasesIdByDirectCampaignId = grutCampaigns
            .associate { it.meta.directId to it.spec.minusPhrasesIdsList }

        deleteRemovedMinusPhrases(campaigns, minusPhrasesIdByDirectCampaignId)

        // выбираем кампании с существующими минус фразами и в груте и в mysql
        // и кампании, для которых в груте еще не создан набор минус фраз
        val (campaignsWithExistingMinusPhrases, campaignsWithNewMinusPhrases) = campaigns
            .filter { !it.minusKeywords.isNullOrEmpty() }
            .partition { !minusPhrasesIdByDirectCampaignId[it.id].isNullOrEmpty() }

        // получаем из грута сами значения минус фраз для кампаний, у которых они есть
        val minusPhrasesIdsToExtractByCampaignId = campaignsWithExistingMinusPhrases
            .associate { it.id to minusPhrasesIdByDirectCampaignId[it.id]!![0] }

        val grutMinusPhrasesById =
            objectApiService.minusPhrasesGrutDao.getMinusPhrases(minusPhrasesIdsToExtractByCampaignId.values)
                .associateBy { it.meta.id }

        // выбираем только те минус фразы, которые отличаются от техкущих значений и обновляем их
        val minusPhrasesToUpdate = campaignsWithExistingMinusPhrases
            .filter {
                val minusPhraseId = minusPhrasesIdsToExtractByCampaignId[it.id]
                val grutMinusPhrases = grutMinusPhrasesById[minusPhraseId]!!.spec.phrasesList
                it.minusKeywords != grutMinusPhrases
            }
            .map {
                MinusPhrase(
                    id = minusPhrasesIdsToExtractByCampaignId[it.id],
                    clientId = it.clientId,
                    name = null,
                    phrases = it.minusKeywords,
                    isLibrary = false
                )
            }
        logger.info("Getting minus phrases from GRuT")
        objectApiService.minusPhrasesGrutDao.createOrUpdateMinusPhrases(minusPhrasesToUpdate)

        // создаем новые минус фразы, полученные id приписываем кампаниям в порядке их следования в списке
        val minusPhrasesToCreate = campaignsWithNewMinusPhrases
            .map {
                MinusPhrase(
                    id = null,
                    clientId = it.clientId,
                    name = null,
                    phrases = it.minusKeywords,
                    isLibrary = false
                )
            }

        logger.info("Creating minus phrases in GRuT")
        val createdMinusPhrasesIdByCampaignId = objectApiService.minusPhrasesGrutDao.createObjects(minusPhrasesToCreate)
            .mapIndexed { idx, minusPhraseId ->
                campaignsWithNewMinusPhrases[idx].id to minusPhraseId
            }

        return createdMinusPhrasesIdByCampaignId
            .union(campaignsWithExistingMinusPhrases.map { it.id to minusPhrasesIdByDirectCampaignId[it.id]!![0] })
            .toMap()
    }

    /**
     * Выбирает те кампании, на которых уже нет минус фраз, но в груте они есть
     * Удаляет из грута их минус фразы
     */
    private fun deleteRemovedMinusPhrases(
        campaigns: List<CommonCampaign>,
        minusPhrasesIdByCampaignId: Map<Long, List<Long>>
    ) {
        // выбираем те кампании, на которых уже нет минус фраз, но в груте они есть
        val minusPhrasesToDelete = campaigns
            .filterIsInstance<CampaignWithMinusKeywords>()
            .filter { it.minusKeywords.isNullOrEmpty() && !minusPhrasesIdByCampaignId[it.id].isNullOrEmpty() }
            .map { minusPhrasesIdByCampaignId[it.id]!![0] }

        // удаляем минус фразы кампаний с предыдущего шага
        if (minusPhrasesToDelete.isNotEmpty()) {
            objectApiService.minusPhrasesGrutDao.deleteObjects(minusPhrasesToDelete)
        }
    }
}

/**
 * Оригинал https://a.yandex-team.ru/arc/trunk/arcadia/direct/perl/protected/BS/ExportQuery.pm?rev=r9182513#L8464
 */
fun calculateOrderType(
    campaign: CommonCampaign,
    wallet: WalletTypedCampaign?,
    client: Client,
    agency: Client?
): Int {
    if (client.isBusinessUnit == true) {
        // реклама бизнес-юнитов
        return 9
    }
    if (client.socialAdvertising == true || (agency != null && agency.socialAdvertising == true)) {
        // социальная реклама
        return 10
    }
    if (CampaignTypeKinds.INTERNAL.contains(campaign.type) || campaign.paidByCertificate == true || (wallet != null && wallet.paidByCertificate == true)) {
        // внутренняя реклама
        return 6
    }

    if (BEGUN_AGENCY_IDS.contains(campaign.agencyId) || campaign.id == TEST_BEGUN_CAMPAIGN_ID) {
        // специальные агентские кампании
        return 8
    }
    // обычный Директ - коммерция
    return 1
}

