package ru.yandex.direct.logicprocessor.processors.mysql2grut.replicationwriter

import org.slf4j.Logger
import ru.yandex.direct.ess.logicobjects.mysql2grut.Mysql2GrutReplicationObject
import ru.yandex.direct.tracing.Trace

data class ObjectsForUpdateAndDelete<T>(
    val objectsToUpdate: List<T>,
    val objectsToDelete: List<T>,
)

data class ReplicationWriterResultWithCallbacks(
    val missingForeignEntities: Collection<Mysql2GrutReplicationObject>,
    val writeObjectsCallback: Runnable,
    val deleteObjectsCallback: Runnable,
)

abstract class BaseReplicationWriter<T> {
    abstract val logger: Logger

    abstract fun getLogicObjectsToWrite(
        shard: Int, logicObjects: Collection<Mysql2GrutReplicationObject>
    ): ObjectsForUpdateAndDelete<T>

    /**
     * Внешние сущности, которые нужно досоздать для корректного создания объектов
     * Этот метод нужно использовать, когда все объекты и их внешние сущности уже лежат в груте
     * Этот метод нужен для таких случаев:
     * 1) Изменилось какое-либо поле кампании
     * 2) Создалась и привязалась к кампании внешняя сущность
     * После попадания первого события в пачку, может произойти второе, но в пачку не попасть, так как стейт будет взят из mysql,
     * он уже будет с добавленной новой связью и при обновлении будет ошибка
     *
     * Если объекты и их внешние связи не пролиты до конца, то этот метод НЕЛЬЗЯ использовать, надо использовать filterObjectsWithParent
     * Иначе это очень сильно и неконтролируемо замедлит репликацию
     */
    open fun getMissingForeignEntitiesObjects(
        shard: Int,
        objects: Collection<T>
    ): Collection<Mysql2GrutReplicationObject> =
        listOf()

    // Оставить только те объекты, для которых существует родительский объект (в иерархии грута)
    // остальные объекты будут пропущены
    abstract fun filterObjectsWithParent(shard: Int, objects: Collection<T>): Collection<T>
    abstract fun getNotExistingInMysqlObjects(shard: Int, objects: Collection<T>): Collection<T>
    abstract fun writeObjectsToGrut(shard: Int, objects: Collection<T>)
    abstract fun deleteObjectsInGrut(objects: Collection<T>)
    abstract fun isDisabledInShard(shard: Int): Boolean

    fun getReplicationWriterResultWithCallbacks(
        shard: Int,
        logicObjects: Collection<Mysql2GrutReplicationObject>
    ): ReplicationWriterResultWithCallbacks {
        logger.info("Start get logic objects to write")
        val filteredObjects = getLogicObjectsToWrite(shard, logicObjects)
        val objectsForUpdate = filteredObjects.objectsToUpdate
        val objectsForDelete = filteredObjects.objectsToDelete
        if (objectsForUpdate.isEmpty() && objectsForDelete.isEmpty()) {
            return ReplicationWriterResultWithCallbacks(listOf(), {}, {})
        }
        logger.info("Start filtering objects without parent")

        val missingForeignEntities = getMissingForeignEntitiesObjects(shard, objectsForUpdate)
        val writeObjectsCallback = {
            val objectsWithExistingParent = Trace.current().profile("filter_objects_with_parent").use {
                filterObjectsWithParent(shard, objectsForUpdate)
            }
            if (objectsWithExistingParent.size != objectsForUpdate.size) {
                logger.info("There is ${objectsForUpdate.size - objectsWithExistingParent.size} objects without parent")
            }
            logger.info("Start prepare objects and writing to grut")
            if (objectsWithExistingParent.isNotEmpty()) {
                logger.info("Update ${objectsWithExistingParent.size} objects")
                Trace.current().profile("write_objects_grut").use {
                    writeObjectsToGrut(shard, objectsWithExistingParent)
                }
            }
        }

        val deleteObjectsCallback = {
            logger.info("Start deleting objects")
            if (objectsForDelete.isNotEmpty()) {
                Trace.current().profile("delete_objects_grut").use {
                    val notExistingObjectsToDelete = getNotExistingInMysqlObjects(shard, objectsForDelete)
                    if (objectsForDelete.size != notExistingObjectsToDelete.size) {
                        logger.error("Objects ${objectsForDelete - notExistingObjectsToDelete} that wanted to be deleted, but exists in mysql")
                    }
                    if (notExistingObjectsToDelete.isNotEmpty()) {
                        logger.info("Delete ${notExistingObjectsToDelete.size} objects")
                        deleteObjectsInGrut(notExistingObjectsToDelete)
                    }
                }
            }
        }
        return ReplicationWriterResultWithCallbacks(missingForeignEntities, writeObjectsCallback, deleteObjectsCallback)
    }
}
