package ru.yandex.direct.logicprocessor.processors.mysql2grut.replicationwriter

import org.slf4j.Logger
import org.slf4j.LoggerFactory
import org.springframework.stereotype.Component
import ru.yandex.direct.common.db.PpcPropertiesSupport
import ru.yandex.direct.common.db.PpcProperty
import ru.yandex.direct.common.db.PpcPropertyNames
import ru.yandex.direct.core.entity.bidmodifier.AbstractBidModifierRetargeting
import ru.yandex.direct.core.entity.bidmodifier.AgeType
import ru.yandex.direct.core.entity.bidmodifier.BidModifier
import ru.yandex.direct.core.entity.bidmodifier.BidModifierABSegment
import ru.yandex.direct.core.entity.bidmodifier.BidModifierDemographicsAdjustment
import ru.yandex.direct.core.entity.bidmodifier.BidModifierType
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository
import ru.yandex.direct.core.grut.api.BidModifierGrut
import ru.yandex.direct.core.grut.replication.GrutApiService
import ru.yandex.direct.ess.logicobjects.mysql2grut.BidModifierTableType
import ru.yandex.direct.ess.logicobjects.mysql2grut.Mysql2GrutReplicationObject
import ru.yandex.direct.logicprocessor.processors.mysql2grut.service.BidModifierReplicationService
import java.time.Duration

data class BidModifierWriterObject(
    val bidModifier: BidModifier? = null,
    val id: Long,
    val tableType: BidModifierTableType? = null, // Используется при удалении корректировок
    val clientId: Long? = null,
)

@Component
class BidModifierReplicationWriter(
    private val objectApiService: GrutApiService,
    private val bidModifierReplicationService: BidModifierReplicationService,
    private val campaignRepository: CampaignRepository,
    ppcPropertiesSupport: PpcPropertiesSupport,
) : BaseReplicationWriter<BidModifierWriterObject>() {

    companion object {
        private val logger = LoggerFactory.getLogger(BidModifierReplicationWriter::class.java)

        /**
         * Создает объект BidModifierGrut по корректировке, у которой не бывает несколько записей в *-values таблице.
         */
        fun createGrutObjectBySingleValuedAdjustment(bidModifier: BidModifier, clientId: Long): BidModifierGrut {
            val adjustment = BidModifierReplicationService.extractSingleAdjustment(bidModifier)
            return BidModifierGrut(
                id = adjustment.id,
                clientId = clientId,
                type = bidModifier.type!!,
                adjustment = adjustment,
            )
        }

        /**
         * Создает объекты BidModifierGrut по корректировке, у которой бывают множественные значения в *-values таблице.
         */
        fun createGrutObjectsByMultiValuedAdjustment(bidModifier: BidModifier, clientId: Long): List<BidModifierGrut> {
            val adjustments = BidModifierReplicationService.extractMultipleAdjustments(bidModifier)
            return adjustments.map {
                BidModifierGrut(
                    id = it.id,
                    clientId = clientId,
                    type = bidModifier.type!!,
                    adjustment = it,
                )
            }
        }
    }

    override val logger: Logger = BidModifierReplicationWriter.logger
    private val skipShardsProperty: PpcProperty<Set<Int>> =
        ppcPropertiesSupport.get(PpcPropertyNames.GRUT_SKIP_BID_MODIFIERS_REPLICATION_SHARDS, Duration.ofSeconds(20))
    private val asyncUpdate =
        ppcPropertiesSupport.get(PpcPropertyNames.GRUT_CAMP_REPL_ASYNC_UPDATE, Duration.ofSeconds(20))

    private val bidModifierTypesWithRetargetingId = setOf(
        BidModifierType.RETARGETING_MULTIPLIER,
        BidModifierType.RETARGETING_FILTER,
        BidModifierType.AB_SEGMENT_MULTIPLIER,
    )

    override fun getLogicObjectsToWrite(shard: Int, logicObjects: Collection<Mysql2GrutReplicationObject>)
        : ObjectsForUpdateAndDelete<BidModifierWriterObject> {
        val (toUpsert, toDelete) = logicObjects.filter { it.bidModifierId != null }.partition { !it.isDeleted }
        val upsertObjects = createObjectsForUpdate(shard, toUpsert)
        // Предполагаем, что для hierachical_mulitpliers
        // удаления для дочерних строк в *-values таблицах уже пришли.
        val deleteObjects = toDelete.map {
            BidModifierWriterObject(
                id = it.bidModifierId!!,
                tableType = it.bidModifierTableType
            )
        }
        return ObjectsForUpdateAndDelete(upsertObjects, deleteObjects)
    }

    private fun createObjectsForUpdate(
        shard: Int,
        objects: List<Mysql2GrutReplicationObject>
    ): List<BidModifierWriterObject> {
        val bidModifiers = objects
            .filter { it.bidModifierId != null }
            .groupBy({ it.bidModifierTableType!! }, { it.bidModifierId!! })
        val bidModifiersWithAdjustments = bidModifierReplicationService.getBidModifiers(shard, bidModifiers)
        // Получаем маппинг кампаний к клиентам
        val campaignsIds = bidModifiersWithAdjustments.map { it.campaignId!! }.distinct()
        val campaignIdsToClientIds = campaignRepository.getClientIdsForCids(shard, campaignsIds)

        return bidModifiersWithAdjustments
            .filter { campaignIdsToClientIds.containsKey(it.campaignId) } // Удаляем корректировки, чьи кампании уже удалились
            .map {
                BidModifierWriterObject(
                    bidModifier = it,
                    id = it.id,
                    clientId = campaignIdsToClientIds[it.campaignId]!!,
                )
            }
    }

    override fun filterObjectsWithParent(
        shard: Int,
        objects: Collection<BidModifierWriterObject>
    ): Collection<BidModifierWriterObject> {
        val clientsIds = objects.map { it.clientId!! }.toSet()
        val existingInGrutClientsIds = objectApiService.clientGrutDao.getExistingObjects(clientsIds).toSet()

        if (existingInGrutClientsIds.size != clientsIds.size) {
            logger.info(
                "${clientsIds.size - existingInGrutClientsIds.size} clients don't exist: ${
                    clientsIds.minus(
                        existingInGrutClientsIds
                    )
                }"
            )
        }
        val filteredByClient = objects.filter { existingInGrutClientsIds.contains(it.clientId) }
        val retCondIds = extractRetargetingConditionIds(filteredByClient)
        if (retCondIds.isEmpty()) {
            return filteredByClient
        }
        // Фильруем корректировки, для которых не существует условий ретаргетинга в GrUT
        val retCondInGrut = objectApiService.retargetingConditionGrutApi
            .getExistingObjects(retCondIds)
            .toSet()
        return filteredByClient.filter {
            !bidModifierTypesWithRetargetingId.contains(it.bidModifier!!.type)
                || retCondInGrut.containsAll(extractRetargetingConditionIds(it.bidModifier))
        }
    }

    /**
     * Достать все идентификаторы условий ретаргетинга из корректировок
     */
    private fun extractRetargetingConditionIds(objects: List<BidModifierWriterObject>): Set<Long> {
        return objects
            .filter { bidModifierTypesWithRetargetingId.contains(it.bidModifier!!.type) }
            .flatMap { extractRetargetingConditionIds(it.bidModifier!!) }
            .toSet()
    }

    private fun extractRetargetingConditionIds(bidModifier: BidModifier): List<Long> {
        return when (bidModifier) {
            is AbstractBidModifierRetargeting -> bidModifier.retargetingAdjustments.map { it.retargetingConditionId }
            is BidModifierABSegment -> bidModifier.abSegmentAdjustments.map { it.abSegmentRetargetingConditionId }
            else -> emptyList()
        }
    }

    override fun getNotExistingInMysqlObjects(
        shard: Int,
        objects: Collection<BidModifierWriterObject>
    ): Collection<BidModifierWriterObject> {
        val existingBidModifiersIds = bidModifierReplicationService.getExisingInMysqlBidModifiers(shard, objects)
        return objects.filterNot { existingBidModifiersIds.contains(it.id) }
    }

    fun createGrutObjects(objects: Collection<BidModifierWriterObject>): Collection<BidModifierGrut> {
        val (singleValued, multiValued) = objects.partition { bidModifierReplicationService.isSingleValued(it.bidModifier!!) }
        val singleValuedAdjustments =
            singleValued.map { createGrutObjectBySingleValuedAdjustment(it.bidModifier!!, it.clientId!!) }
        val multivaluedAdjustments = multiValued
            .flatMap { createGrutObjectsByMultiValuedAdjustment(it.bidModifier!!, it.clientId!!) }

        return singleValuedAdjustments + multivaluedAdjustments
    }

    override fun writeObjectsToGrut(shard: Int, objects: Collection<BidModifierWriterObject>) {
        val grutObject = createGrutObjects(objects)

        if (asyncUpdate.getOrDefault(false)) {
            objectApiService.bidModifierGrutApi.createOrUpdateBidModifiersParallel(grutObject)
        } else {
            objectApiService.bidModifierGrutApi.createOrUpdateBidModifiers(grutObject)
        }
    }

    override fun deleteObjectsInGrut(objects: Collection<BidModifierWriterObject>) {
        val bidModifiersIds = objects.map { it.id }.distinct()
        logger.info("Delete requests for bidModifiers: $bidModifiersIds")
        objectApiService.bidModifierGrutApi.deleteObjects(bidModifiersIds)
    }

    override fun isDisabledInShard(shard: Int): Boolean {
        return skipShardsProperty.get()
            .orEmpty()
            .contains(shard)
    }
}
