package ru.yandex.direct.logicprocessor.processors.mysql2grut.replicationwriter

import org.slf4j.Logger
import org.slf4j.LoggerFactory
import org.springframework.stereotype.Component
import ru.yandex.direct.common.db.PpcPropertiesSupport
import ru.yandex.direct.common.db.PpcProperty
import ru.yandex.direct.common.db.PpcPropertyNames
import ru.yandex.direct.core.grut.api.BiddableShowConditionType
import ru.yandex.direct.core.grut.replication.GrutApiService
import ru.yandex.direct.core.mysql2grut.repository.BiddableShowCondition
import ru.yandex.direct.core.mysql2grut.repository.BiddableShowConditionsRepository
import ru.yandex.direct.ess.logicobjects.mysql2grut.BiddableShowConditionChangeType
import ru.yandex.direct.ess.logicobjects.mysql2grut.Mysql2GrutReplicationObject
import java.time.Duration

data class BiddableShowConditionWriterObject(
    val type: BiddableShowConditionType,
    val obj: BiddableShowCondition? = null,
    val idToDelete: Long? = null,
)

@Component
class BiddableShowConditionReplicationWriter(
    private val biddableShowConditionsRepository: BiddableShowConditionsRepository,
    private val objectApiService: GrutApiService,
    private val ppcPropertiesSupport: PpcPropertiesSupport,
) : BaseReplicationWriter<BiddableShowConditionWriterObject>() {

    companion object {
        private val logger = LoggerFactory.getLogger(BiddableShowConditionReplicationWriter::class.java)

        val TYPE_MAPPING = mapOf(
            BiddableShowConditionChangeType.KEYWORD to BiddableShowConditionType.KEYWORD,
            BiddableShowConditionChangeType.OFFER_RETARGETING to BiddableShowConditionType.OFFER_RETARGETING,
            BiddableShowConditionChangeType.RELEVANCE_MATCH to BiddableShowConditionType.RELEVANCE_MATCH,
            BiddableShowConditionChangeType.PERFORMANCE to BiddableShowConditionType.PERFORMANCE,
            BiddableShowConditionChangeType.DYNAMIC to BiddableShowConditionType.DYNAMIC,
            BiddableShowConditionChangeType.RETARGETING to BiddableShowConditionType.RETARGETING,
        )

        val REVERSE_MAPPING = mapOf(
            BiddableShowConditionType.KEYWORD to BiddableShowConditionChangeType.KEYWORD,
            BiddableShowConditionType.OFFER_RETARGETING to BiddableShowConditionChangeType.OFFER_RETARGETING,
            BiddableShowConditionType.RELEVANCE_MATCH to BiddableShowConditionChangeType.RELEVANCE_MATCH,
            BiddableShowConditionType.PERFORMANCE to BiddableShowConditionChangeType.PERFORMANCE,
            BiddableShowConditionType.DYNAMIC to BiddableShowConditionChangeType.DYNAMIC,
            BiddableShowConditionType.RETARGETING to BiddableShowConditionChangeType.RETARGETING,
        )
    }

    override val logger: Logger = BiddableShowConditionReplicationWriter.logger
    private val skipShardsProperty: PpcProperty<Set<Int>> =
        ppcPropertiesSupport.get(PpcPropertyNames.GRUT_SKIP_BIDDABLE_SHOW_CONDITION_REPLICATION_SHARDS, Duration.ofSeconds(20))
    private val asyncUpdate = ppcPropertiesSupport.get(PpcPropertyNames.GRUT_CAMP_REPL_ASYNC_UPDATE, Duration.ofSeconds(20))
    // Общий флаг с группами на возможность догружать FK объекты, потому что biddable_show_conditions бывают только у групп.
    private val filterForeignEntities = ppcPropertiesSupport.get(PpcPropertyNames.ADGROUP_REPLICATION_FILTER_WITHOUT_FOREIGN, Duration.ofSeconds(20))

    override fun getLogicObjectsToWrite(shard: Int, logicObjects: Collection<Mysql2GrutReplicationObject>)
        : ObjectsForUpdateAndDelete<BiddableShowConditionWriterObject> {

        val (toUpsert, toDelete) = logicObjects
            .filter { it.biddableShowConditionType != null }
            .partition { !it.isDeleted }
        val upsertObjects = createUpsertObjects(shard, toUpsert)
        val deleteObjects = createDeleteObjects(toDelete)
        return ObjectsForUpdateAndDelete(upsertObjects, deleteObjects)
    }

    private fun createUpsertObjects(
        shard: Int,
        replicationObjects: Collection<Mysql2GrutReplicationObject>)
        : List<BiddableShowConditionWriterObject> {
        if (replicationObjects.isEmpty()) return emptyList()
        val conditionsByType = replicationObjects
            .groupBy { it.biddableShowConditionType }
        val biddableShowConditionIdsByType = mutableMapOf<BiddableShowConditionType, Set<Long>>()

        // keyword
        val keywordIds = conditionsByType.getOrDefault(BiddableShowConditionChangeType.KEYWORD, emptyList()).map { it.biddableShowConditionId!! }
        val hrefParamIds = conditionsByType.getOrDefault(BiddableShowConditionChangeType.HREF_PARAM, emptyList()).map { it.biddableShowConditionId!! }
        biddableShowConditionIdsByType[BiddableShowConditionType.KEYWORD] = (keywordIds + hrefParamIds).toSet()

        // offer retargeting
        biddableShowConditionIdsByType[BiddableShowConditionType.OFFER_RETARGETING] = conditionsByType
            .getOrDefault(BiddableShowConditionChangeType.OFFER_RETARGETING, emptyList())
            .map { it.biddableShowConditionId!! }
            .toSet()

        // relevance match
        biddableShowConditionIdsByType[BiddableShowConditionType.RELEVANCE_MATCH] = conditionsByType
            .getOrDefault(BiddableShowConditionChangeType.RELEVANCE_MATCH, emptyList())
            .map { it.biddableShowConditionId!! }
            .plus(hrefParamIds)
            .toSet()

        // dynamic
        val dynamicIds = conditionsByType.getOrDefault(BiddableShowConditionChangeType.DYNAMIC, emptyList()).map { it.biddableShowConditionId!! }
        val dynamicConditionsIds = conditionsByType.getOrDefault(BiddableShowConditionChangeType.DYNAMIC_CONDITION, emptyList()).map { it.biddableShowConditionId!! }
        val dynamicIdsByConditionIds = biddableShowConditionsRepository.getBidsDynamicIdsByDynamicConditions(shard, dynamicConditionsIds)
        biddableShowConditionIdsByType[BiddableShowConditionType.DYNAMIC] = (dynamicIds + dynamicIdsByConditionIds).toSet()

        // performance
        biddableShowConditionIdsByType[BiddableShowConditionType.PERFORMANCE] = conditionsByType.getOrDefault(BiddableShowConditionChangeType.PERFORMANCE, emptyList())
            .map { it.biddableShowConditionId!! }
            .toSet()

        // retargeting
        biddableShowConditionIdsByType[BiddableShowConditionType.RETARGETING] = conditionsByType.getOrDefault(BiddableShowConditionChangeType.RETARGETING, emptyList())
            .map { it.biddableShowConditionId!! }
            .toSet()

        return biddableShowConditionsRepository.getBiddableShowConditions(shard, biddableShowConditionIdsByType)
            .map { BiddableShowConditionWriterObject(type = it.type, obj = it) }
    }

    private fun createDeleteObjects(replicationObjects: Collection<Mysql2GrutReplicationObject>): List<BiddableShowConditionWriterObject> {
        // Избавляемся от дубликатов
        val groupedByType = replicationObjects.groupBy { it.biddableShowConditionType!! }
            .mapValues { it.value.map { obj -> obj.biddableShowConditionId!! }.toSet() }
        return groupedByType.flatMap {
            it.value.map { id ->
                BiddableShowConditionWriterObject(
                    type = TYPE_MAPPING[it.key]!!,
                    idToDelete = id,
                )
            }
        }
    }

    override fun getMissingForeignEntitiesObjects(
        shard: Int,
        objects: Collection<BiddableShowConditionWriterObject>
    ): Collection<Mysql2GrutReplicationObject> {
        // Если включена фильтрация объектов с отсутствующими внешними ключами, то создание таких сущностей не делаем
        if (filterForeignEntities.getOrDefault(true)) {
            return listOf()
        }
        val missingRetConds = getMissingInGrutRetargetingConditions(objects)
        logger.info("bsc FK added to replication, ${missingRetConds.size} retargeting conditions : $missingRetConds")
        return missingRetConds.map { Mysql2GrutReplicationObject(retargetingConditionId = it.value) }
    }

    private fun getMissingInGrutRetargetingConditions(
        objects: Collection<BiddableShowConditionWriterObject>
    ) : Map<Long, Long> {
        val bscIdToRetCondId = objects
            .filter { it.type == BiddableShowConditionType.RETARGETING }
            .associate {it.obj!!.id to  it.obj.retargeting!!.retargetingConditionId }
        if (bscIdToRetCondId.isEmpty()) {
            return emptyMap()
        }
        val existingRetCondIds = objectApiService.retargetingConditionGrutApi.getExistingObjects(bscIdToRetCondId.values.toSet()).toSet()
        return bscIdToRetCondId.filterValues { !existingRetCondIds.contains(it) }
    }

    override fun filterObjectsWithParent(
        shard: Int,
        objects: Collection<BiddableShowConditionWriterObject>
    ): Collection<BiddableShowConditionWriterObject> {
        val clientIds = objects.map { it.obj!!.clientId }.distinct()
        val existingInGrutClients = objectApiService.clientGrutDao.getExistingObjects(clientIds).toSet()
        if (existingInGrutClients.size != clientIds.size) {
            logger.info(
                "${clientIds.size - existingInGrutClients.size} biddable show condition parent client don't exist: ${clientIds.minus(existingInGrutClients)}"
            )
        }
        val filteredByParent = objects.filter { existingInGrutClients.contains(it.obj!!.clientId) }

        // Фильтруем bids_retargeting для которых нет retargeting_conditions
        val missingRetConds = getMissingInGrutRetargetingConditions(objects)

        if (missingRetConds.isNotEmpty()) {
            for ((retargetingBid, retCondId) in missingRetConds) {
                logger.info("FK missing: retargetingBid  $retargetingBid, retCondId: $retCondId")
            }
        }
        return filteredByParent.filter {
            it.type != BiddableShowConditionType.RETARGETING || !missingRetConds.keys.contains(it.obj!!.id)
        }
    }

    /**
     * Выкинуть объекты, которые помечены на удаление, но при этом существуют в MySQL
     */
    override fun getNotExistingInMysqlObjects(shard: Int, objects: Collection<BiddableShowConditionWriterObject>): Collection<BiddableShowConditionWriterObject> {
        val exisingBiddableShowConditionsIds = mutableSetOf<Long>()
        // keywords
        val keywordIds = objects.filter { it.type == BiddableShowConditionType.KEYWORD }.map { it.idToDelete!! }
        val existingInMySqlKeywordIds = biddableShowConditionsRepository.getExistingKeywords(shard, keywordIds)
        biddableShowConditionsRepository
        exisingBiddableShowConditionsIds.addAll(existingInMySqlKeywordIds)

        // bids base (offer retargeting + relevance match)
        val bidsBaseIds = objects
            .filter { it.type == BiddableShowConditionType.OFFER_RETARGETING || it.type == BiddableShowConditionType.RELEVANCE_MATCH }
            .map { it.idToDelete!! }
        val existingBidsBaseIds = biddableShowConditionsRepository.getExistingBidsBase(shard, bidsBaseIds)
        exisingBiddableShowConditionsIds.addAll(existingBidsBaseIds)

        // dynamic
        val dynamicIds = objects.filter { it.type == BiddableShowConditionType.DYNAMIC }.map { it.idToDelete!! }
        val existingInMysqlDynamicIds = biddableShowConditionsRepository.getExistingDynamic(shard, dynamicIds)
        exisingBiddableShowConditionsIds.addAll(existingInMysqlDynamicIds)

        // performance
        val performanceIds = objects.filter { it.type == BiddableShowConditionType.PERFORMANCE }.map { it.idToDelete!! }
        val existingInMysqlPerformanceIds = biddableShowConditionsRepository.getExistingPerformance(shard, performanceIds)
        exisingBiddableShowConditionsIds.addAll(existingInMysqlPerformanceIds)

        // retargeting
        val retargetingIds = objects.filter { it.type == BiddableShowConditionType.RETARGETING }.map { it.idToDelete!! }
        val existingInMysqlRetargetingIds = biddableShowConditionsRepository.getExistingRetargeting(shard, retargetingIds)
        exisingBiddableShowConditionsIds.addAll(existingInMysqlRetargetingIds)

        return objects.filterNot { exisingBiddableShowConditionsIds.contains(it.idToDelete) }
    }

    override fun deleteObjectsInGrut(objects: Collection<BiddableShowConditionWriterObject>) {
        val idsByType = objects.groupBy({ it.type }, { it.idToDelete!! })
        objectApiService.biddableShowConditionReplicationGrutDao.deleteBiddableShowConditionsByDirectIdsParallel(idsByType)
    }

    override fun isDisabledInShard(shard: Int): Boolean {
        return skipShardsProperty.get()
            .orEmpty()
            .contains(shard)
    }

    override fun writeObjectsToGrut(shard: Int, objects: Collection<BiddableShowConditionWriterObject>) {
        if (asyncUpdate.getOrDefault(false)) {
            objectApiService.biddableShowConditionReplicationGrutDao
                .createOrUpdateBiddableShowConditionsParallel(objects.map { it.obj!! })
        } else {
            objectApiService.biddableShowConditionReplicationGrutDao
                .createOrUpdateBiddableShowConditions(objects.map { it.obj!! })
        }
    }
}
