package ru.yandex.direct.logicprocessor.processors.mysql2grut.replicationwriter

import java.time.Duration
import java.util.concurrent.CompletableFuture
import java.util.concurrent.ExecutorService
import org.jooq.Condition
import org.slf4j.Logger
import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Value
import org.springframework.context.annotation.Lazy
import org.springframework.stereotype.Component
import ru.yandex.direct.common.db.PpcPropertiesSupport
import ru.yandex.direct.common.db.PpcPropertyNames
import ru.yandex.direct.common.util.DirectThreadPoolExecutor
import ru.yandex.direct.core.entity.additionaltargetings.repository.CampAdditionalTargetingsRepository
import ru.yandex.direct.core.entity.bs.common.service.BsOrderIdCalculator
import ru.yandex.direct.core.entity.campaign.model.CampaignWithBrandSafety
import ru.yandex.direct.core.entity.campaign.model.CampaignWithPackageStrategy
import ru.yandex.direct.core.entity.campaign.model.CampaignWithPricePackage
import ru.yandex.direct.core.entity.campaign.model.CommonCampaign
import ru.yandex.direct.core.entity.campaign.model.WalletTypedCampaign
import ru.yandex.direct.core.entity.campaign.model.WithAbSegmentRetargetingConditionIds
import ru.yandex.direct.core.entity.campaign.service.CampaignAutobudgetRestartService
import ru.yandex.direct.core.entity.campaign.service.WalletHasMoneyChecker
import ru.yandex.direct.core.entity.metrika.repository.MetrikaCampaignRepository
import ru.yandex.direct.core.entity.mobileapp.service.IosSkAdNetworkSlotManager
import ru.yandex.direct.core.grut.api.utils.waitFutures
import ru.yandex.direct.core.grut.replication.GrutApiService
import ru.yandex.direct.dbschema.ppc.Tables
import ru.yandex.direct.dbschema.ppc.enums.CampaignsType
import ru.yandex.direct.ess.logicobjects.mysql2grut.Mysql2GrutReplicationObject
import ru.yandex.direct.logicprocessor.processors.bsexport.utils.SupportedCampaignsService
import ru.yandex.direct.logicprocessor.processors.mysql2grut.repository.CampOrderTypesRepository
import ru.yandex.direct.multitype.repository.filter.ConditionFilter

@Lazy
@Component
class CampaignReplicationWriter(
    private val campaignTypedRepository: SupportedCampaignsService,
    private val objectApiService: GrutApiService,
    @Value("\${grut_replication.get_campaigns_threads:1}") private val getCampaignsThreadPoolSize: Int,
    @Value("\${grut_replication.calc_autobudget_restarts_threads:1}") calcAutobudgetRestartsThreadPoolSize: Int,
    walletHasMoneyChecker: WalletHasMoneyChecker,
    timezoneCache: TimezoneCache,
    campOrderTypesRepository: CampOrderTypesRepository,
    autobudgetRestartService: CampaignAutobudgetRestartService,
    metrikaCampaignRepository: MetrikaCampaignRepository,
    bsOrderIdCalculator: BsOrderIdCalculator,
    campaignAdditionalTargetingsRepository: CampAdditionalTargetingsRepository,
    skAdNetworkSlotManager: IosSkAdNetworkSlotManager,
    ppcPropertiesSupport: PpcPropertiesSupport,
) : BaseCampaignReplicationWriter(
    campaignTypedRepository,
    objectApiService,
    timezoneCache,
    campOrderTypesRepository,
    autobudgetRestartService,
    metrikaCampaignRepository,
    bsOrderIdCalculator,
    walletHasMoneyChecker,
    campaignAdditionalTargetingsRepository,
    calcAutobudgetRestartsThreadPoolSize,
    skAdNetworkSlotManager,
    ppcPropertiesSupport
), AutoCloseable {

    companion object {
        private val campaignLogger = LoggerFactory.getLogger(CampaignReplicationWriter::class.java)
        private const val defaultDbLoadChunkSize = 10_000
        private val campaignsDbLoadTimeout = Duration.ofMinutes(2)
    }

    override val logger: Logger = campaignLogger

    // выбираем количество тредов примерно по числу vCPU в продакшене jobs
    // очередь делаем побольше, при нашем использованииa не видно разницы блокироваться на постановке задачи
    // или на ожидании выполнения
    private val executorService: ExecutorService =
        DirectThreadPoolExecutor(getCampaignsThreadPoolSize, 500, "grut-get-typed-campaigns")
    private val loadCampaignsChunkSize =
        ppcPropertiesSupport.get(PpcPropertyNames.GRUT_REPLICATION_CAMPAIGNS_DB_LOAD_CHUNK_SIZE, Duration.ofSeconds(60))

    private object CampaignNotWalletFilter : ConditionFilter() {
        override fun getCondition(): Condition = Tables.CAMPAIGNS.TYPE.notEqual(CampaignsType.wallet)
        override fun isEmpty(): Boolean = false
    }

    override fun getCampaigns(shard: Int, campaignIds: Collection<Long>): List<CommonCampaign> {
        return getCampaignsAsync(shard, campaignIds)
    }

    private fun getCampaignsSync(shard: Int, campaignIds: Collection<Long>): List<CommonCampaign> {
        return campaignTypedRepository.getTyped(shard, campaignIds, listOf(CampaignNotWalletFilter))
            .map { it as CommonCampaign }
    }

    private fun getCampaignsAsync(shard: Int, campaignIds: Collection<Long>): List<CommonCampaign> {
        val chunkSize = loadCampaignsChunkSize.getOrDefault(defaultDbLoadChunkSize)
        if (campaignIds.size <= 1.05 * chunkSize) {
            // TODO DIRECT-168914: убрать возможность синхронного чтения баннеров из базы
            // когда убедимся, что асинхронно все хорошо работает
            return getCampaignsSync(shard, campaignIds)
        }

        val completableFutures = campaignIds.chunked(chunkSize)
            .map {
                CompletableFuture.supplyAsync({ getCampaignsSync(shard, it) }, executorService)
            }
            .toTypedArray()

        waitFutures(completableFutures, campaignsDbLoadTimeout)
        return completableFutures
            .map { it.get() }
            .flatten()
    }

    override fun getObjectsForDelete(objects: Collection<Mysql2GrutReplicationObject>): List<Mysql2GrutReplicationObject> {
        return objects.filter { it.campaignId != null }.filter { it.isDeleted }
    }

    override fun filterSpecificEntities(
        shard: Int,
        objects: Collection<CampaignWriterData>
    ): Collection<CampaignWriterData> {
        val campaignsWithMissingEntities = getMissingSpecificForeignEntitiesByCampaigns(objects).keys
        return objects.filterNot { campaignsWithMissingEntities.contains(it.campaignId) }
    }

    override fun getNotExistingInMysqlObjects(
        shard: Int,
        objects: Collection<CampaignWriterData>
    ): Collection<CampaignWriterData> {
        val campaignIds = objects.map { it.campaignId }.toSet()
        val existingCampaignsMap = campaignTypedRepository.getSafely(shard, campaignIds, CommonCampaign::class.java)
            .associateBy { it.id }
        return objects.filter { it.campaignId !in existingCampaignsMap }
    }

    override fun deleteObjectsInGrut(objects: Collection<CampaignWriterData>) {
        objectApiService.campaignGrutDao.deleteCampaignsByDirectIds(objects.map { it.campaignId })
    }

    override fun getMissingSpecificForeignEntities(
        shard: Int,
        objects: Collection<CampaignWriterData>
    ): Collection<Mysql2GrutReplicationObject> {
        return getMissingSpecificForeignEntitiesByCampaigns(objects).flatMap { it.value }
    }

    private fun getMissingSpecificForeignEntitiesByCampaigns(
        objects: Collection<CampaignWriterData>
    ): Map<Long, List<Mysql2GrutReplicationObject>> {
        val gotWalletIds = objects.mapNotNull { it.campaign!!.walletId }.filter { it != 0L }.toSet()
        // если кампания и ее кошелек будут в одной пачке, то на данный момент они кампания не будет реплицирована
        // когда все объекты проедут ваншотом, то условия на наличие кошелька не будет, и кампания и кошелек будут реплицированы
        val existingWallets =
            objectApiService.campaignGrutDao.getCampaignsByDirectIds(gotWalletIds, listOf("/meta/direct_id"))
                .map { it.meta.directId }.toSet()
        val allWallets =
            objects.filter { it.campaign is WalletTypedCampaign }.map { it.campaignId }.union(existingWallets)
                .toSet()

        val gotStrategiesIds = objects
            .mapNotNull { it.campaign as? CampaignWithPackageStrategy }
            .map { it.strategyId }
            .filter { it != 0L }
            .toSet()
        val existingStrategies = objectApiService.strategyGrutApi.getExistingObjects(gotStrategiesIds)

        val gotAbSegmentRetCondIds = objects
            .mapNotNull {
                it.campaign as? WithAbSegmentRetargetingConditionIds
            }.flatMap { listOf(it.abSegmentRetargetingConditionId, it.abSegmentStatisticRetargetingConditionId) }
            .filter { it ?: 0L != 0L }
            .toSet()

        val gotBrandSafetyRetCondIds = objects
            .mapNotNull { it.campaign as? CampaignWithBrandSafety }
            .map { it.brandSafetyRetCondId }
            .filter { it ?: 0L != 0L }
            .toSet()

        val existingRetCondIds =
            objectApiService.retargetingConditionGrutApi.getExistingObjects(gotAbSegmentRetCondIds + gotBrandSafetyRetCondIds)

        val walletsToCreate = objects
            .filter {
                it.campaign!!.walletId != 0L && !allWallets.contains(it.campaign.walletId)
            }.map {
                it.campaignId to Mysql2GrutReplicationObject(campaignId = it.campaign!!.walletId)
            }
        logger.info("Found ${walletsToCreate.size} missing wallets")

        val strategiesToCreate = objects
            .mapNotNull { it.campaign as? CampaignWithPackageStrategy }
            .filter {
                it.strategyId != 0L && !existingStrategies.contains(it.strategyId)
            }.map {
                it.id to Mysql2GrutReplicationObject(strategyId = it.strategyId)
            }
        logger.info("Found ${strategiesToCreate.size} missing strategies")

        val objectsWithExperiments = objects
            .map { it.campaign!!.id to it.campaign as? WithAbSegmentRetargetingConditionIds }
            .filter { it.second != null }

        val abSegmentRetCondIdsToCreate = objectsWithExperiments
            .asSequence()
            .filter {
                it.second!!.abSegmentRetargetingConditionId ?: 0L != 0L && !existingRetCondIds.contains(it.second!!.abSegmentRetargetingConditionId)
            }.map {
                it.first to Mysql2GrutReplicationObject(retargetingConditionId = it.second!!.abSegmentRetargetingConditionId)
            }
            .toList()
        logger.info("Found ${abSegmentRetCondIdsToCreate.size} missing abSegmentRetCondIds")

        val abSegmentStatisticRetCondIdsToCreate = objectsWithExperiments
            .asSequence()
            .filter {
                it.second!!.abSegmentStatisticRetargetingConditionId ?: 0L != 0L && !existingRetCondIds.contains(
                    it.second!!.abSegmentStatisticRetargetingConditionId
                )
            }.map {
                it.first to Mysql2GrutReplicationObject(retargetingConditionId = it.second!!.abSegmentStatisticRetargetingConditionId)
            }.distinct()
            .toList()

        logger.info("Found ${abSegmentStatisticRetCondIdsToCreate.size} missing abSegmentStatisticRetCondIds")

        val brandSafetyRetCondIdsToCreate = objects
            .asSequence()
            .mapNotNull { it.campaign as? CampaignWithBrandSafety }
            .filter { it.brandSafetyRetCondId ?: 0 != 0L && !existingRetCondIds.contains(it.brandSafetyRetCondId) }
            .map { it.id to Mysql2GrutReplicationObject(retargetingConditionId = it.brandSafetyRetCondId) }
            .distinct()
            .toList()

        logger.info("Found ${brandSafetyRetCondIdsToCreate.size} missing brandSafetyRetCondIds")

        val pricePackagesToCreate = getPricePackagesToCreate(objects)
        logger.info("Add ${pricePackagesToCreate.size} packages to replication")


        return walletsToCreate
            .union(strategiesToCreate)
            .union(abSegmentRetCondIdsToCreate)
            .union(abSegmentStatisticRetCondIdsToCreate)
            .union(brandSafetyRetCondIdsToCreate)
            .union(pricePackagesToCreate)
            .groupBy({ it.first }, { it.second })
    }

    private fun getPricePackagesToCreate(objects: Collection<CampaignWriterData>): List<Pair<Long, Mysql2GrutReplicationObject>> {
        // Прайсовые пакеты будем всегда добавлять в список для репликации, так как они отдельно не реплицируются и переотправить их нельзя
        // Пусть переотправляются вместе с кампаниями
        // Их в базе совсем мало, так что нагрузку не увеличат
        return objects
            .filter { it.campaign is CampaignWithPricePackage && it.campaign.pricePackageId ?: 0L != 0L }
            .map {
                it.campaignId to
                    Mysql2GrutReplicationObject(pricePackageId = (it.campaign as CampaignWithPricePackage).pricePackageId)
            }
    }

    override fun close() {
        executorService.shutdown()
    }
}
