package ru.yandex.direct.logicprocessor.processors.mysql2grut.replicationwriter

import java.time.Duration
import org.slf4j.Logger
import org.slf4j.LoggerFactory
import org.springframework.stereotype.Component
import ru.yandex.direct.common.db.PpcPropertiesSupport
import ru.yandex.direct.common.db.PpcPropertyNames
import ru.yandex.direct.core.entity.additionaltargetings.repository.ClientAdditionalTargetingsRepository
import ru.yandex.direct.core.entity.campaign.model.BillingAggregateCampaign
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository
import ru.yandex.direct.core.entity.campaign.repository.filter.CampaignFilterFactory
import ru.yandex.direct.core.entity.client.repository.ClientNdsRepository
import ru.yandex.direct.core.entity.client.repository.ClientRepository
import ru.yandex.direct.core.entity.internalads.repository.InternalAdsProductRepository
import ru.yandex.direct.core.entity.product.service.ProductService
import ru.yandex.direct.core.grut.api.BillingAggregateInfo
import ru.yandex.direct.core.grut.api.ClientGrutModel
import ru.yandex.direct.core.grut.replication.GrutApiService
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.ess.logicobjects.mysql2grut.Mysql2GrutReplicationObject

data class ClientWriterObject(
    val clientId: Long
)

@Component
class ClientReplicationWriter(
    private val objectApiService: GrutApiService,
    private val clientRepository: ClientRepository,
    private val clientNdsRepository: ClientNdsRepository,
    private val internalAdsProductRepository: InternalAdsProductRepository,
    private val campaignTypedRepository: CampaignTypedRepository,
    private val productService: ProductService,
    private val clientAdditionalTargetingsRepository: ClientAdditionalTargetingsRepository,
    ppcPropertiesSupport: PpcPropertiesSupport
) : BaseReplicationWriter<ClientWriterObject>() {

    companion object {
        private val clientLogger = LoggerFactory.getLogger(ClientReplicationWriter::class.java)
    }

    override val logger: Logger = clientLogger
    private val asyncUpdate =
        ppcPropertiesSupport.get(PpcPropertyNames.GRUT_CAMP_REPL_ASYNC_UPDATE, Duration.ofSeconds(20))

    override fun isDisabledInShard(shard: Int): Boolean {
        return false
    }

    override fun getLogicObjectsToWrite(
        shard: Int,
        logicObjects: Collection<Mysql2GrutReplicationObject>
    ): ObjectsForUpdateAndDelete<ClientWriterObject> {
        val (logicObjectsForUpdate, logicObjectsForDelete) = logicObjects
            .filter { it.clientId != null }
            .partition { !it.isDeleted }
        val objectForUpdate = logicObjectsForUpdate
            .map { it.clientId!! }
            .distinct()
            .map { ClientWriterObject(it) }

        val objectForDelete = logicObjectsForDelete
            .map { it.clientId!! }
            .distinct()
            .map { ClientWriterObject(it) }

        return ObjectsForUpdateAndDelete(objectForUpdate, objectForDelete)

    }

    override fun filterObjectsWithParent(shard: Int, objects: Collection<ClientWriterObject>) = objects

    override fun writeObjectsToGrut(shard: Int, objects: Collection<ClientWriterObject>) {
        val getClientsArgs = objects.map { ClientId.fromLong(it.clientId) }.toList()
        val clientsForUpsert = clientRepository.get(shard, getClientsArgs)
        val clientIds = clientsForUpsert.map { it.clientId }
        val additionalTargetingsByClientId = clientAdditionalTargetingsRepository.findByClientIds(shard, clientIds)
            .groupBy { it.clientId }


        val clientsNdsHistory = clientNdsRepository.fetchHistoryByClientIds(shard, clientIds)
        val billingAggregates = loadBillingAggregates(shard, clientIds)
        val internalAdProducts = internalAdsProductRepository.getProductsNotStrictly(shard, clientIds)
            .associate { it.clientId.asLong() to it.name }

        val clientGrutModel = clientsForUpsert.map {
            ClientGrutModel(
                client = it,
                ndsHistory = clientsNdsHistory[it.id] ?: listOf(),
                billingAggregates = billingAggregates[it.id] ?: listOf(),
                internalAdProduct = internalAdProducts[it.id],
                additionalTargetings = additionalTargetingsByClientId[it.id]
            )
        }
        if (clientsForUpsert.isNotEmpty()) {
            if (asyncUpdate.getOrDefault(false)) {
                objectApiService.clientGrutDao.createOrUpdateClientsParallel(clientGrutModel)
            } else {
                objectApiService.clientGrutDao.createOrUpdateClients(clientGrutModel)
            }
        }

    }

    override fun deleteObjectsInGrut(objects: Collection<ClientWriterObject>) {
        objectApiService.clientGrutDao.deleteObjects(objects.map { it.clientId })
    }

    override fun getNotExistingInMysqlObjects(
        shard: Int,
        objects: Collection<ClientWriterObject>
    ): Collection<ClientWriterObject> {
        val getClientsArgs = objects.map { ClientId.fromLong(it.clientId) }.toList()
        val existingClients = clientRepository
            .get(shard, getClientsArgs)
            .map { it.clientId }
            .toSet()

        if (existingClients.isNotEmpty()) {
            logger.error("Some clients sent for deletion are still exist in db: $existingClients")
        }
        return objects.filterNot { existingClients.contains(it.clientId) }
    }

    private fun loadBillingAggregates(shard: Int, clientIds: List<Long>): Map<Long, List<BillingAggregateInfo>> {
        val clients = clientIds.map { ClientId.fromLong(it) }.toList()
        val filter = CampaignFilterFactory.campaignClientIdsFilter(clients)

        val aggregates = campaignTypedRepository.getSafely(shard, filter, BillingAggregateCampaign::class.java)
        val walletIds = aggregates.map { it.walletId }.filter { it > 0 }.toSet()
        val directToGrutIds = objectApiService.campaignGrutDao.getCampaignIdsByDirectIds(walletIds)

        return aggregates.filter {
            val walletHasGrutId = it.walletId in directToGrutIds
            if (!walletHasGrutId) {
                logger.warn("Filtered billing aggregate without wallet campaign in GrUT: $it")
            }
            walletHasGrutId
        }.groupBy({ it.clientId }, {
            BillingAggregateInfo(
                id = it.id,
                walletOrderId = directToGrutIds[it.walletId]!!,
                productId = it.productId,
                productType = productService.getProductById(it.productId).type
            )
        })
    }
}
