package ru.yandex.direct.logicprocessor.processors.mysql2grut.replicationwriter

import org.slf4j.Logger
import org.slf4j.LoggerFactory
import org.springframework.stereotype.Component
import ru.yandex.direct.common.db.PpcPropertiesSupport
import ru.yandex.direct.common.db.PpcProperty
import ru.yandex.direct.common.db.PpcPropertyNames
import ru.yandex.direct.core.entity.creative.model.Creative
import ru.yandex.direct.core.entity.creative.repository.CreativeRepository
import ru.yandex.direct.core.grut.api.CreativeGrutModel
import ru.yandex.direct.core.grut.replication.GrutApiService
import ru.yandex.direct.ess.logicobjects.mysql2grut.Mysql2GrutReplicationObject
import java.time.Duration

data class CreativeWriterObject(
    val creativeId: Long,

    // null в операции delete
    val creative: Creative? = null
)

@Component
class CreativeReplicationWriter(
    private val objectApiService: GrutApiService,
    private val creativeRepository: CreativeRepository,
    val ppcPropertiesSupport: PpcPropertiesSupport,
) : BaseReplicationWriter<CreativeWriterObject>() {
    companion object {
        private val creativeLogger = LoggerFactory.getLogger(CreativeReplicationWriter::class.java)
    }

    override val logger: Logger = creativeLogger

    private val property: PpcProperty<Set<Int>> =
        ppcPropertiesSupport.get(PpcPropertyNames.GRUT_SKIP_CREATIVES_REPLICATION_SHARDS, Duration.ofSeconds(20))

    private val asyncUpdate = ppcPropertiesSupport.get(PpcPropertyNames.GRUT_CAMP_REPL_ASYNC_UPDATE, Duration.ofSeconds(20))

    override fun getLogicObjectsToWrite(
        shard: Int,
        logicObjects: Collection<Mysql2GrutReplicationObject>
    ): ObjectsForUpdateAndDelete<CreativeWriterObject> {
        val (logicObjectsForUpdate, logicObjectsForDelete) = logicObjects
            .filter { it.creativeId != null }
            .partition { !it.isDeleted }

        if (logicObjectsForUpdate.isEmpty() && logicObjectsForDelete.isEmpty()) {
            return ObjectsForUpdateAndDelete(emptyList(), emptyList())
        }

        val creativeIds = logicObjectsForUpdate.map { it.creativeId!! }.distinct()
        val creatives = creativeRepository.getCreatives(shard, creativeIds)

        val objectsForUpdate = creatives.map { CreativeWriterObject(it.id, it) }
        val objectsForDelete = logicObjectsForDelete.map { CreativeWriterObject(it.creativeId!!) }

        return ObjectsForUpdateAndDelete(objectsForUpdate, objectsForDelete)
    }

    override fun filterObjectsWithParent(
        shard: Int,
        objects: Collection<CreativeWriterObject>
    ): Collection<CreativeWriterObject> {
        val clientIds = objects.map { it.creative!!.clientId }.distinct()
        val existingClients = objectApiService.clientGrutDao.getExistingObjects(clientIds).toSet()
        if (existingClients.size != clientIds.size) {
            logger.info("${clientIds.size - existingClients.size} clients don't exist")
        }
        return objects.filter { existingClients.contains(it.creative!!.clientId) }
    }

    override fun getNotExistingInMysqlObjects(
        shard: Int,
        objects: Collection<CreativeWriterObject>
    ): Collection<CreativeWriterObject> {
        val creativeIds = objects.map { it.creativeId }.toSet()
        val existingCreativeIds = creativeRepository.getCreatives(shard, creativeIds).map { it.id }
        return objects.filter { it.creativeId !in existingCreativeIds }
    }

    override fun writeObjectsToGrut(shard: Int, objects: Collection<CreativeWriterObject>) {
        val objectsToWrite = objects.map {
            CreativeGrutModel(
                it.creative!!,
                it.creative.clientId
            )
        }
        if(asyncUpdate.getOrDefault(false)) {
            objectApiService.creativeGrutDao.createOrUpdateCreativesParallel(objectsToWrite)
        } else {
            objectApiService.creativeGrutDao.createOrUpdateCreatives(objectsToWrite)
        }
    }

    override fun deleteObjectsInGrut(objects: Collection<CreativeWriterObject>) {
        objectApiService.creativeGrutDao.deleteObjects(objects.map { it.creativeId })
    }

    override fun isDisabledInShard(shard: Int): Boolean {
        return property.get()
            .orEmpty()
            .contains(shard)
    }
}
