package ru.yandex.direct.logicprocessor.processors.mysql2grut.replicationwriter

import org.slf4j.Logger
import org.slf4j.LoggerFactory
import org.springframework.stereotype.Component
import ru.yandex.direct.common.db.PpcPropertiesSupport
import ru.yandex.direct.common.db.PpcProperty
import ru.yandex.direct.common.db.PpcPropertyNames
import ru.yandex.direct.core.grut.replication.GrutApiService
import ru.yandex.direct.core.mysql2grut.repository.MinusPhrase
import ru.yandex.direct.core.mysql2grut.repository.MinusPhraseRepository
import ru.yandex.direct.ess.logicobjects.mysql2grut.Mysql2GrutReplicationObject
import java.time.Duration

data class MinusPhraseWriterObject(
    val minusPhrase: MinusPhrase?, //null on delete requests only
    val minusPhraseId: Long
)

@Component
class MinusPhraseReplicationWriter(
    private val objectApiService: GrutApiService,
    private val minusPhraseRepository: MinusPhraseRepository,
    ppcPropertiesSupport: PpcPropertiesSupport,
) : BaseReplicationWriter<MinusPhraseWriterObject>() {

    companion object {
        private val minusPhraseLogger = LoggerFactory.getLogger(MinusPhraseReplicationWriter::class.java)
    }

    override val logger: Logger = minusPhraseLogger
    private val property: PpcProperty<Set<Int>> =
        ppcPropertiesSupport.get(PpcPropertyNames.GRUT_SKIP_MINUS_PHRASES_REPLICATION_SHARDS, Duration.ofSeconds(20))
    private val asyncUpdate =
        ppcPropertiesSupport.get(PpcPropertyNames.GRUT_CAMP_REPL_ASYNC_UPDATE, Duration.ofSeconds(20))

    override fun getLogicObjectsToWrite(
        shard: Int,
        logicObjects: Collection<Mysql2GrutReplicationObject>
    ): ObjectsForUpdateAndDelete<MinusPhraseWriterObject> {
        val (upsertObjects, deleteObjects) = logicObjects
            .filter { it.minusPhraseId != null }
            .partition { !it.isDeleted }

        val upsertMinusPhrasesIds = upsertObjects
            .map { it.minusPhraseId!! }
            .distinct()
        val deleteMinusPhraseIds = deleteObjects
            .map { it.minusPhraseId!! }
            .distinct()

        val minusPhrases = minusPhraseRepository.getMinusPhrases(shard, upsertMinusPhrasesIds)

        val objectsForUpdate = minusPhrases.map { MinusPhraseWriterObject(it, it.id!!) }
        val objectsForDelete = deleteMinusPhraseIds.map { MinusPhraseWriterObject(null, it) }

        return ObjectsForUpdateAndDelete(objectsForUpdate, objectsForDelete)
    }

    override fun filterObjectsWithParent(
        shard: Int,
        objects: Collection<MinusPhraseWriterObject>
    ): Collection<MinusPhraseWriterObject> {
        val clientsIds = objects.map { it.minusPhrase!!.clientId }.distinct()
        val existingInGrutClients = objectApiService.clientGrutDao.getExistingObjects(clientsIds).toSet()
        if (existingInGrutClients.size != clientsIds.size) {
            logger.info(
                "${clientsIds.size - existingInGrutClients.size} minus phrases clients don't exist: ${
                    clientsIds.minus(
                        existingInGrutClients
                    )
                }"
            )
        }
        return objects.filter { existingInGrutClients.contains(it.minusPhrase!!.clientId) }
    }

    override fun getNotExistingInMysqlObjects(
        shard: Int,
        objects: Collection<MinusPhraseWriterObject>
    ): Collection<MinusPhraseWriterObject> {
        val minusPhraseIds = objects.map { it.minusPhraseId }.toSet()
        val existingMinusPhrasesIds = minusPhraseRepository.getMinusPhrases(shard, minusPhraseIds)
            .map { it.id }
            .toSet()
        return objects.filterNot { existingMinusPhrasesIds.contains(it.minusPhraseId) }
    }

    override fun writeObjectsToGrut(shard: Int, objects: Collection<MinusPhraseWriterObject>) {
        val minusPhrases = objects.map { it.minusPhrase!! }
        if (asyncUpdate.getOrDefault(false)) {
            objectApiService.minusPhrasesGrutDao.createOrUpdateMinusPhrasesParallel(minusPhrases)
        } else {
            objectApiService.minusPhrasesGrutDao.createOrUpdateMinusPhrases(minusPhrases)
        }
    }

    override fun deleteObjectsInGrut(objects: Collection<MinusPhraseWriterObject>) {
        val ids = objects.map { it.minusPhraseId }
        logger.info("Delete requests for minus phrases: $ids")
        objectApiService.minusPhrasesGrutDao.deleteObjects(ids)
    }

    override fun isDisabledInShard(shard: Int): Boolean {
        return property.get().orEmpty().contains(shard)
    }
}
