package ru.yandex.direct.logicprocessor.processors.mysql2grut.replicationwriter

import java.time.Duration
import org.slf4j.Logger
import org.slf4j.LoggerFactory
import org.springframework.stereotype.Component
import ru.yandex.direct.common.db.PpcPropertiesSupport
import ru.yandex.direct.common.db.PpcProperty
import ru.yandex.direct.common.db.PpcPropertyNames
import ru.yandex.direct.core.entity.mobilecontent.model.MobileContent
import ru.yandex.direct.core.entity.mobilecontent.repository.MobileContentRepository
import ru.yandex.direct.core.grut.replication.GrutApiService
import ru.yandex.direct.ess.logicobjects.mysql2grut.Mysql2GrutReplicationObject

data class MobileContentWriterObject(
    val mobileContent: MobileContent?, // null on delete requests only
    val mobileContentId: Long,
    val isDeleted: Boolean = false
)

@Component
class MobileContentReplicationWriter(
    ppcPropertiesSupport: PpcPropertiesSupport,
    private val objectApiService: GrutApiService,
    private val mobileContentRepository: MobileContentRepository,
) : BaseReplicationWriter<MobileContentWriterObject>() {
    companion object {
        private val mobileContentLogger = LoggerFactory.getLogger(MobileContentReplicationWriter::class.java)
    }

    override val logger: Logger = mobileContentLogger
    private val property: PpcProperty<Set<Int>> =
        ppcPropertiesSupport.get(PpcPropertyNames.GRUT_SKIP_MOBILE_CONTENT_REPLICATION_SHARDS, Duration.ofSeconds(20))
    private val asyncUpdate = ppcPropertiesSupport.get(PpcPropertyNames.GRUT_CAMP_REPL_ASYNC_UPDATE, Duration.ofSeconds(20))

    override fun getLogicObjectsToWrite(shard: Int, logicObjects: Collection<Mysql2GrutReplicationObject>): ObjectsForUpdateAndDelete<MobileContentWriterObject> {
        val (upsertObjects, deleteObjects) = logicObjects
            .filter { it.mobileContentId != null }
            .partition { !it.isDeleted }

        val upsertMobileContentIds = upsertObjects
            .map { it.mobileContentId!! }
            .distinct()

        val deleteMobileContentIds = deleteObjects
            .map { it.mobileContentId!! }
            .distinct()

        val mobileContent = mobileContentRepository.getMobileContent(shard, upsertMobileContentIds)

        val objectsForUpdate = mobileContent.map { MobileContentWriterObject(it!!, it.id!!) }
        val objectsForDelete = deleteMobileContentIds.map { MobileContentWriterObject(null, it) }

        return ObjectsForUpdateAndDelete(objectsForUpdate, objectsForDelete)
    }

    override fun filterObjectsWithParent(shard: Int, objects: Collection<MobileContentWriterObject>): Collection<MobileContentWriterObject> {
        val clientsIds = objects.map { it.mobileContent!!.clientId }.distinct()
        val existingInGrutClients = objectApiService.clientGrutDao.getExistingObjects(clientsIds).toSet()
        if (existingInGrutClients.size != clientsIds.size) {
            logger.info("${clientsIds.size - existingInGrutClients.size} mobile_content clients don't exist: ${clientsIds.minus(existingInGrutClients)}")
        }
        return objects.filter { existingInGrutClients.contains(it.mobileContent!!.clientId) }
    }

    override fun getNotExistingInMysqlObjects(shard: Int, objects: Collection<MobileContentWriterObject>): Collection<MobileContentWriterObject> {
        val mobileContentIds = objects.map { it.mobileContentId }.toSet()
        val existingMobileContentIds = mobileContentRepository.getMobileContent(shard, mobileContentIds).map { it.id }
        return objects.filterNot { existingMobileContentIds.contains(it.mobileContentId) }
    }

    override fun writeObjectsToGrut(shard: Int, objects: Collection<MobileContentWriterObject>) {
        val mobileContent = objects.map { it.mobileContent!! }
        if (asyncUpdate.getOrDefault(false)) {
            objectApiService.mobileContentGrutDao.createOrUpdateMobileContentParallel(mobileContent)
        } else {
            objectApiService.mobileContentGrutDao.createOrUpdateMobileContent(mobileContent)
        }
    }

    override fun deleteObjectsInGrut(objects: Collection<MobileContentWriterObject>) {
        val ids = objects.map { it.mobileContentId }
        logger.info("Delete requests for mobile_content with ids: $ids")
        objectApiService.mobileContentGrutDao.deleteObjects(ids)
    }

    override fun isDisabledInShard(shard: Int): Boolean {
        return property.get().orEmpty().contains(shard)
    }


}
