package ru.yandex.direct.logicprocessor.processors.mysql2grut.replicationwriter

import org.slf4j.Logger
import org.slf4j.LoggerFactory
import org.springframework.stereotype.Component
import ru.yandex.direct.common.db.PpcPropertiesSupport
import ru.yandex.direct.common.db.PpcPropertyNames
import ru.yandex.direct.core.entity.pricepackage.repository.PricePackageRepository
import ru.yandex.direct.core.grut.api.PricePackageGrut
import ru.yandex.direct.core.grut.replication.GrutApiService
import ru.yandex.direct.ess.logicobjects.mysql2grut.Mysql2GrutReplicationObject
import java.time.Duration

data class PricePackageWriterObject(
    val pricePackage: PricePackageGrut
)

/**
 * Прайсовые пакеты не реплицируются по настоящему, этот класс нужен для того, чтобы если пакета для кампании еще нет в груте,
 * то можно было бы создать его из репликации кампаний, добавив в список объектов для репликации пакет
 */
@Component
class PricePackageReplicationWriter(
    private val objectApiService: GrutApiService,
    private val pricePackageRepository: PricePackageRepository,
    val ppcPropertiesSupport: PpcPropertiesSupport
) : BaseReplicationWriter<PricePackageWriterObject>() {
    companion object {
        private val logger = LoggerFactory.getLogger(PricePackageReplicationWriter::class.java)
    }

    override val logger: Logger = PricePackageReplicationWriter.logger

    private val asyncUpdate =
        ppcPropertiesSupport.get(PpcPropertyNames.GRUT_CAMP_REPL_ASYNC_UPDATE, Duration.ofSeconds(20))

    override fun getLogicObjectsToWrite(
        shard: Int,
        logicObjects: Collection<Mysql2GrutReplicationObject>
    ): ObjectsForUpdateAndDelete<PricePackageWriterObject> {
        val pricePackageIds = logicObjects
            .mapNotNull { it.pricePackageId }
            .distinct()

        if (pricePackageIds.isEmpty()) {
            return ObjectsForUpdateAndDelete(emptyList(), emptyList())
        }
        val packages = pricePackageRepository.getPricePackages(pricePackageIds)
            .map {
                PricePackageWriterObject(
                    PricePackageGrut(
                        id = it.key,
                        auctionPriority = it.value.auctionPriority,
                        allowedDomains = it.value.allowedDomains ?: listOf(),
                        adGroupTypes = it.value.availableAdGroupTypes ?: setOf()
                    )
                )
            }

        return ObjectsForUpdateAndDelete(packages, listOf())
    }

    override fun filterObjectsWithParent(
        shard: Int,
        objects: Collection<PricePackageWriterObject>
    ): Collection<PricePackageWriterObject> {
        return objects
    }

    override fun getNotExistingInMysqlObjects(
        shard: Int,
        objects: Collection<PricePackageWriterObject>
    ): Collection<PricePackageWriterObject> = listOf() // Прайсовые пакеты не удаляются

    override fun writeObjectsToGrut(shard: Int, objects: Collection<PricePackageWriterObject>) {
        val objectsToWrite = objects.map { it.pricePackage }
        logger.info("Going to write ${objectsToWrite.size} price packages")
        if (asyncUpdate.getOrDefault(false)) {
            objectApiService.pricePackageGrutApi.createOrUpdatePackagesParallel(objectsToWrite)
        } else {
            objectApiService.pricePackageGrutApi.createOrUpdatePackages(objectsToWrite)
        }
    }

    override fun deleteObjectsInGrut(objects: Collection<PricePackageWriterObject>) {
        logger.info("No deletion for price packages")
    }

    override fun isDisabledInShard(shard: Int): Boolean {
        return false
    }
}
