package ru.yandex.direct.logicprocessor.processors.mysql2grut.replicationwriter

import org.springframework.context.annotation.Lazy
import org.springframework.stereotype.Service
import ru.yandex.direct.ess.logicobjects.mysql2grut.Mysql2GrutReplicationObject

@Lazy
@Service
class ReplicationWriterService(
    clientReplicationWriter: ClientReplicationWriter,
    minusPhraseReplicationWriter: MinusPhraseReplicationWriter,
    mobileContentReplicationWriter: MobileContentReplicationWriter,
    campaignReplicationWriter: CampaignReplicationWriter,
    adGroupReplicationWriter: AdGroupReplicationWriter,
    vcardReplicationWriter: VCardReplicationWriter,
    creativeReplicationWriter: CreativeReplicationWriter,
    biddableShowConditionReplicationWriter: BiddableShowConditionReplicationWriter,
    bidModifierReplicationWriter: BidModifierReplicationWriter,
    bannerReplicationWriter: BannerReplicationWriter,
    walletReplicationWriter: WalletReplicationWriter,
    retargetingConditionReplicationWriter: RetargetingConditionReplicationWriter,
    strategyReplicationWriter: StrategyReplicationWriter,
    pricePackageReplicationWriter: PricePackageReplicationWriter,
) {

    /** Порядок в этом списке важно, если объект зависит от других, то в списке должен быть после них */
    private val replicationWritersOrder = listOf(
        clientReplicationWriter,
        pricePackageReplicationWriter,
        minusPhraseReplicationWriter,
        mobileContentReplicationWriter,
        strategyReplicationWriter,
        retargetingConditionReplicationWriter,
        bidModifierReplicationWriter,
        biddableShowConditionReplicationWriter,
        walletReplicationWriter,
        campaignReplicationWriter,
        adGroupReplicationWriter,
        vcardReplicationWriter,
        creativeReplicationWriter,
        bannerReplicationWriter,
    )

    /**
     * На первом проходе writer'ы собирают объекты для записи и удаления, а так же объекты, которых не хватает для корректного
     * обновления(внешние связи). На этом проходе список replicationWritersOrder обходится в обратном порядке,
     * так как нужные для создания сущности находятся перед теми, для кого их нужно создать. Такие объекты добавляются
     * в общий список logicObjects и подхватятся следующими Writer'ами
     *
     * На втором обходе список replicationWritersOrder обходится в прямом порядке и уже происходит запись и удаление объектов
     *
     */
    fun writeObjects(shard: Int, logicObjects: Collection<Mysql2GrutReplicationObject>) {
        var logicObjectsWithMissingForeignEntities = logicObjects.toMutableList()
        val replicationWriterResultWithCallbacksList: MutableList<ReplicationWriterResultWithCallbacks> =
            mutableListOf()

        for (replicationWriter in replicationWritersOrder.reversed()) {
            if (replicationWriter.isDisabledInShard(shard)) {
                replicationWriter.logger.info("Writer disabled, ${logicObjects.size} objects skipped")
                continue
            }
            val replicationWriterResultWithCallbacks = replicationWriter.getReplicationWriterResultWithCallbacks(
                shard,
                logicObjectsWithMissingForeignEntities.toList()
            )
            if (replicationWriterResultWithCallbacks.missingForeignEntities.isNotEmpty()) {
                replicationWriter.logger.info("Found ${replicationWriterResultWithCallbacks.missingForeignEntities.size} missing foreign entities, they will be added to objects list")
                logicObjectsWithMissingForeignEntities.addAll(replicationWriterResultWithCallbacks.missingForeignEntities)
                logicObjectsWithMissingForeignEntities =
                    logicObjectsWithMissingForeignEntities.distinct().toMutableList()
            }
            replicationWriterResultWithCallbacksList.add(replicationWriterResultWithCallbacks)
        }

        for (replicationWriterCallbacks in replicationWriterResultWithCallbacksList.reversed()) {
            replicationWriterCallbacks.writeObjectsCallback.run()
            replicationWriterCallbacks.deleteObjectsCallback.run()
        }
    }
}
