package ru.yandex.direct.logicprocessor.processors.mysql2grut.replicationwriter

import org.slf4j.Logger
import org.slf4j.LoggerFactory
import org.springframework.stereotype.Component
import ru.yandex.direct.common.db.PpcPropertiesSupport
import ru.yandex.direct.common.db.PpcPropertyNames
import ru.yandex.direct.core.entity.retargeting.repository.RetargetingConditionRepository
import ru.yandex.direct.core.grut.api.RetargetingConditionGrut
import ru.yandex.direct.core.grut.replication.GrutApiService
import ru.yandex.direct.core.mysql2grut.repository.RetargetingGoalRepository
import ru.yandex.direct.ess.logicobjects.mysql2grut.Mysql2GrutReplicationObject
import java.time.Duration

data class RetargetingConditionWriterObject(
    val obj: RetargetingConditionGrut? = null,
    val retargetingConditionId: Long,
)

@Component
class RetargetingConditionReplicationWriter(
    ppcPropertiesSupport: PpcPropertiesSupport,
    private val retargetingConditionRepository: RetargetingConditionRepository,
    private val retargetingGoalRepository: RetargetingGoalRepository,
    private val objectApiService: GrutApiService,
) : BaseReplicationWriter<RetargetingConditionWriterObject>() {
    companion object {
        private val retargetingConditionLogger =
            LoggerFactory.getLogger(RetargetingConditionReplicationWriter::class.java)
    }

    override val logger: Logger = retargetingConditionLogger
    private val property = ppcPropertiesSupport.get(
        PpcPropertyNames.GRUT_SKIP_RETARGETING_CONDITION_REPLICATION_SHARDS,
        Duration.ofSeconds(20)
    )

    private val skipWrongParentKeyErrorProperty = ppcPropertiesSupport.get(
        PpcPropertyNames.GRUT_SKIP_WRONG_PARENT_KEY_ERRORS,
        Duration.ofSeconds(20)
    )

    private val asyncUpdate = ppcPropertiesSupport.get(
        PpcPropertyNames.GRUT_CAMP_REPL_ASYNC_UPDATE,
        Duration.ofSeconds(20)
    )

    override fun getLogicObjectsToWrite(
        shard: Int,
        logicObjects: Collection<Mysql2GrutReplicationObject>
    ): ObjectsForUpdateAndDelete<RetargetingConditionWriterObject> {
        val (upsertObjects, deleteObjects) = logicObjects
            .filter { it.retargetingConditionId != null }
            .partition { !it.isDeleted }

        val upsertIds = upsertObjects
            .map { it.retargetingConditionId!! }
            .distinct()
        val deleteIds = deleteObjects
            .map { it.retargetingConditionId!! }
            .distinct()

        val objectsForUpdate = createObjectsForUpdate(shard, upsertIds)
        val objectsForDelete = deleteIds.map { RetargetingConditionWriterObject(retargetingConditionId = it) }

        return ObjectsForUpdateAndDelete(objectsForUpdate, objectsForDelete)
    }

    private fun createObjectsForUpdate(shard: Int, upsertIds: List<Long>): List<RetargetingConditionWriterObject> {
        if (upsertIds.isEmpty()) return emptyList()

        val retargetingConditions = retargetingConditionRepository
            .getConditions(shard, upsertIds)
        val retargetingGoalsById = retargetingGoalRepository.getRetargetingGoals(shard, upsertIds)
            .associateBy { it.goalId }
        return retargetingConditions
            .map {
                RetargetingConditionWriterObject(
                    RetargetingConditionGrut(it, retargetingGoalsById),
                    it.id,
                )
            }
    }

    override fun filterObjectsWithParent(
        shard: Int,
        objects: Collection<RetargetingConditionWriterObject>
    ): Collection<RetargetingConditionWriterObject> {
        val clientIds = objects.map { it.obj!!.retCond.clientId }.distinct()
        val existingInGrutClients = objectApiService.clientGrutDao.getExistingObjects(clientIds).toSet()
        if (existingInGrutClients.size != clientIds.size) {
            logger.info(
                "${clientIds.size - existingInGrutClients.size} " +
                    "retargeting condition parent client don't exist: ${clientIds.minus(existingInGrutClients)}"
            )
        }
        val objectsWithParent = objects
            .filter { existingInGrutClients.contains(it.obj!!.retCond.clientId) }

        // Фильтруем объекты, которые уже есть в GrUT с другим родителем, если флаг выставлен
        if (skipWrongParentKeyErrorProperty.getOrDefault(false)) {
            val retCondIds = objectsWithParent.map { it.retargetingConditionId }
            val retCondIdToClientId = objectApiService.retargetingConditionGrutApi
                .getRetargetingConditions(retCondIds, attrs = listOf("/meta"))
                .associate { it.meta.id to it.meta.clientId }

            val (wrongParent, validParent) = objectsWithParent.partition {
                retCondIdToClientId.containsKey(it.retargetingConditionId) &&
                    retCondIdToClientId[it.retargetingConditionId] != it.obj!!.retCond.clientId
            }
            if (wrongParent.isNotEmpty()) {
                for (obj in wrongParent) {
                    val retCondId = obj.retargetingConditionId
                    logger.error(
                        "retCond $retCondId skipped due wrong parent, " +
                            "existing clientId: ${retCondIdToClientId[retCondId]} " +
                            "passed clientId: ${obj.obj!!.retCond.clientId}"
                    )
                }
            }
            return validParent
        } else {
            return objectsWithParent
        }
    }

    override fun getNotExistingInMysqlObjects(
        shard: Int,
        objects: Collection<RetargetingConditionWriterObject>
    ): Collection<RetargetingConditionWriterObject> {
        val retCondIds = objects.map { it.retargetingConditionId }.distinct()
        val existingRetargetingConditions = retargetingConditionRepository.getConditions(shard, retCondIds)
            .map { it.id }
            .toSet()
        return objects.filterNot { existingRetargetingConditions.contains(it.retargetingConditionId) }
    }

    override fun writeObjectsToGrut(shard: Int, objects: Collection<RetargetingConditionWriterObject>) {
        val retConds = objects.map { it.obj!! }
        if (asyncUpdate.getOrDefault(false)) {
            objectApiService.retargetingConditionGrutApi.createOrUpdateRetargetingConditionsParallel(retConds)
        } else {
            objectApiService.retargetingConditionGrutApi.createOrUpdateRetargetingConditions(retConds)
        }
    }

    /**
     * retargeting_conditions никогда не удаляются, но реализацию на всякий случай написал.
     */
    override fun deleteObjectsInGrut(objects: Collection<RetargetingConditionWriterObject>) {
        val ids = objects.map { it.retargetingConditionId }.distinct()
        logger.info("Deleting retargeting conditions: $ids")
        objectApiService.retargetingConditionGrutApi.deleteObjects(ids)
    }

    override fun isDisabledInShard(shard: Int): Boolean {
        return property.get().orEmpty().contains(shard)
    }
}
