package ru.yandex.direct.logicprocessor.processors.mysql2grut.replicationwriter

import org.slf4j.Logger
import org.slf4j.LoggerFactory
import org.springframework.stereotype.Component
import ru.yandex.direct.common.db.PpcPropertiesSupport
import ru.yandex.direct.common.db.PpcProperty
import ru.yandex.direct.common.db.PpcPropertyNames
import ru.yandex.direct.core.entity.strategy.model.BaseStrategy
import ru.yandex.direct.core.entity.strategy.model.CommonStrategy
import ru.yandex.direct.core.entity.strategy.repository.StrategyTypedRepository
import ru.yandex.direct.core.entity.strategy.utils.StrategyModelUtils.clientId
import ru.yandex.direct.core.grut.api.StrategyGrut
import ru.yandex.direct.core.grut.replication.GrutApiService
import ru.yandex.direct.ess.logicobjects.mysql2grut.Mysql2GrutReplicationObject
import java.time.Duration

data class StrategyWriterObject(
    val strategy: BaseStrategy
)

@Component
class StrategyReplicationWriter(
    private val objectApiService: GrutApiService,
    private val strategyRepository: StrategyTypedRepository,
    val ppcPropertiesSupport: PpcPropertiesSupport
) : BaseReplicationWriter<StrategyWriterObject>() {
    companion object {
        private val logger = LoggerFactory.getLogger(StrategyReplicationWriter::class.java)
    }

    override val logger: Logger = StrategyReplicationWriter.logger

    private val property: PpcProperty<Set<Int>> =
        ppcPropertiesSupport.get(PpcPropertyNames.GRUT_SKIP_STRATEGIES_REPLICATION_SHARDS, Duration.ofSeconds(20))

    private val asyncUpdate =
        ppcPropertiesSupport.get(PpcPropertyNames.GRUT_CAMP_REPL_ASYNC_UPDATE, Duration.ofSeconds(20))

    override fun getLogicObjectsToWrite(
        shard: Int,
        logicObjects: Collection<Mysql2GrutReplicationObject>
    ): ObjectsForUpdateAndDelete<StrategyWriterObject> {
        val logicObjectsForUpdate = logicObjects
            .filter { it.strategyId != null }
            // стратегии не могут удаляться
            .filter { !it.isDeleted }

        if (logicObjectsForUpdate.isEmpty()) {
            return ObjectsForUpdateAndDelete(emptyList(), emptyList())
        }
        val strategyIds = logicObjectsForUpdate.map { it.strategyId!! }.distinct()
        val strategies = strategyRepository.getTyped(shard, strategyIds)
            .map { StrategyWriterObject(it) }

        return ObjectsForUpdateAndDelete(strategies, listOf())
    }

    override fun filterObjectsWithParent(
        shard: Int,
        objects: Collection<StrategyWriterObject>
    ): Collection<StrategyWriterObject> {
        val clientIds = objects.map { it.strategy.clientId()!! }.distinct()
        val existingClients = objectApiService.clientGrutDao.getExistingObjects(clientIds).toSet()
        if (existingClients.size != clientIds.size) {
            logger.info("${clientIds.size - existingClients.size} clients don't exist")
        }
        return objects.filter { existingClients.contains(it.strategy.clientId()!!) }
    }

    override fun getNotExistingInMysqlObjects(
        shard: Int,
        objects: Collection<StrategyWriterObject>
    ) = listOf<StrategyWriterObject>() // Стратегии не удаляются

    override fun isDisabledInShard(shard: Int): Boolean {
        return property.get()
            .orEmpty()
            .contains(shard)
    }

    override fun writeObjectsToGrut(shard: Int, objects: Collection<StrategyWriterObject>) {
        val objectsToWrite = objects.map {
            StrategyGrut(
                id = it.strategy.id,
                clientId = it.strategy.clientId()!!,
                name = (it.strategy as CommonStrategy).name
            )
        }
        if (asyncUpdate.getOrDefault(false)) {
            objectApiService.strategyGrutApi.createOrUpdateStrategiesParallel(objectsToWrite)
        } else {
            objectApiService.strategyGrutApi.createOrUpdateStrategies(objectsToWrite)
        }
    }

    override fun deleteObjectsInGrut(objects: Collection<StrategyWriterObject>) {
        logger.info("No deletion for strategies")
    }
}
