package ru.yandex.direct.logicprocessor.processors.mysql2grut.replicationwriter

import java.time.Duration
import org.slf4j.Logger
import org.slf4j.LoggerFactory
import org.springframework.stereotype.Component
import ru.yandex.direct.common.db.PpcPropertiesSupport
import ru.yandex.direct.common.db.PpcProperty
import ru.yandex.direct.common.db.PpcPropertyNames
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository
import ru.yandex.direct.core.entity.vcard.model.PointOnMap
import ru.yandex.direct.core.entity.vcard.model.Vcard
import ru.yandex.direct.core.entity.vcard.repository.VcardRepository
import ru.yandex.direct.core.entity.vcard.repository.internal.AddressesRepository
import ru.yandex.direct.core.entity.vcard.repository.internal.DbAddress
import ru.yandex.direct.core.entity.vcard.repository.internal.MapsRepository
import ru.yandex.direct.core.grut.api.VCardGrutModel
import ru.yandex.direct.core.grut.replication.GrutApiService
import ru.yandex.direct.ess.logicobjects.mysql2grut.Mysql2GrutReplicationObject

data class VCardDetails(
    val clientId: Long,
    val vcard: Vcard,
    val address: DbAddress? = null,
    val mapPoint: PointOnMap? = null,
    val mapPointAuto: PointOnMap? = null,
)

data class VCardWriterObject(
    val vcardId: Long,

    // null в операции delete
    val vcardDetails: VCardDetails? = null,
)

@Component
class VCardReplicationWriter(
    private val objectApiService: GrutApiService,
    private val vcardRepository: VcardRepository,
    private val campaignRepository: CampaignRepository,
    private val addressesRepository: AddressesRepository,
    private val mapsRepository: MapsRepository,
    ppcPropertiesSupport: PpcPropertiesSupport,
) : BaseReplicationWriter<VCardWriterObject>() {
    companion object {
        private val vcardLogger = LoggerFactory.getLogger(VCardReplicationWriter::class.java)
    }

    override val logger: Logger = vcardLogger

    private val property: PpcProperty<Set<Int>> =
        ppcPropertiesSupport.get(PpcPropertyNames.GRUT_SKIP_VCARDS_REPLICATION_SHARDS, Duration.ofSeconds(20))

    private val asyncUpdate = ppcPropertiesSupport.get(PpcPropertyNames.GRUT_CAMP_REPL_ASYNC_UPDATE, Duration.ofSeconds(20))

    override fun getLogicObjectsToWrite(
        shard: Int,
        logicObjects: Collection<Mysql2GrutReplicationObject>
    ): ObjectsForUpdateAndDelete<VCardWriterObject> {
        val (logicObjectsForUpdate, logicObjectsForDelete) = logicObjects
            .filter { it.vcardId != null || it.vcardAddressId != null }
            .partition { !it.isDeleted }

        if (logicObjectsForUpdate.isEmpty() && logicObjectsForDelete.isEmpty()) {
            return ObjectsForUpdateAndDelete(emptyList(), emptyList())
        }

        // id визиток, обновлённых через апдейт адресов
        val vcardAddressIds = logicObjectsForUpdate.mapNotNull { it.vcardAddressId }.distinct()
        val vcardIdsByAddressIds = if (vcardAddressIds.isNotEmpty()) {
            vcardRepository.getVcardsByAddressIds(shard, vcardAddressIds).map { it.id }
        } else {
            emptyList()
        }
        // id визиток, обновлённых явно по vcardId
        val vcardIds: List<Long> = logicObjectsForUpdate.mapNotNull { it.vcardId }.distinct()

        // Общий список id визиток для обработки
        val allVcardIds = (vcardIds + vcardIdsByAddressIds).distinct()
        val vcards = if (allVcardIds.isNotEmpty()) {
            vcardRepository.getVcards(shard, allVcardIds)
        } else {
            emptyList()
        }

        // кампании нужны для получения ClientID, к которым привязаны визитки
        val campaigns = campaignRepository.getCampaigns(shard, vcards.map { it.campaignId }.distinct())
        val campaignByIdMap = campaigns.associateBy { it.id }

        val addressesMap = addressesRepository.getAddresses(shard, vcards.mapNotNull { it.addressId }.distinct())
        val allPointIds = (addressesMap.values.map { it.mapId } + addressesMap.values.map { it.mapIdAuto })
            .filterNotNull()
        val pointsMap = mapsRepository.getPoints(shard, allPointIds)

        val (vcardsWithCampaign, vcardsWithoutCampaign) = vcards.partition { campaignByIdMap.containsKey(it.campaignId) }

        val objectsForUpdate = vcardsWithCampaign
            .map {
                val address: DbAddress? = it.addressId?.let { addressId -> addressesMap[addressId] }
                VCardWriterObject(
                    it.id,
                    VCardDetails(
                        campaignByIdMap[it.campaignId]!!.clientId,
                        it,
                        address,
                        address?.mapId?.let { mapId -> pointsMap[mapId] },
                        address?.mapIdAuto?.let { mapId -> pointsMap[mapId] }
                    )
                )
            }
        if (vcardsWithoutCampaign.isNotEmpty()) {
            logger.info("Skipped ${vcardsWithoutCampaign.size} vcards without campaign: $vcardsWithoutCampaign")
        }
        val objectsForDelete = logicObjectsForDelete.map { VCardWriterObject(it.vcardId!!) }

        return ObjectsForUpdateAndDelete(objectsForUpdate, objectsForDelete)
    }

    override fun filterObjectsWithParent(
        shard: Int,
        objects: Collection<VCardWriterObject>
    ): Collection<VCardWriterObject> {
        val gotClientIds = objects.map { it.vcardDetails!!.clientId }.distinct()
        val existingClients = objectApiService.clientGrutDao.getExistingObjects(gotClientIds).toSet()
        if (existingClients.size != gotClientIds.size) {
            logger.info("${gotClientIds.size - existingClients.size} clients don't exist")
        }
        return objects.filter { existingClients.contains(it.vcardDetails!!.clientId) }
    }

    override fun getNotExistingInMysqlObjects(
        shard: Int,
        objects: Collection<VCardWriterObject>
    ): Collection<VCardWriterObject> {
        val vcardIds = objects.map { it.vcardId }.toSet()
        val existingVcardIds = vcardRepository.getVcards(shard, vcardIds).map { it.id }
        return objects.filter { it.vcardId !in existingVcardIds }
    }

    override fun writeObjectsToGrut(shard: Int, objects: Collection<VCardWriterObject>) {
        val objectsToWrite = objects.map {
            VCardGrutModel(
                it.vcardDetails!!.vcard,
                it.vcardDetails.clientId,
                it.vcardDetails.address,
                it.vcardDetails.mapPoint,
                it.vcardDetails.mapPointAuto
            )
        }
        if (asyncUpdate.getOrDefault(false)) {
            objectApiService.vcardGrutDao.createOrUpdateVCardsParallel(objectsToWrite)
        } else {
            objectApiService.vcardGrutDao.createOrUpdateVCards(objectsToWrite)
        }
    }

    override fun deleteObjectsInGrut(objects: Collection<VCardWriterObject>) {
        objectApiService.vcardGrutDao.deleteObjects(objects.map { it.vcardId })
    }

    override fun isDisabledInShard(shard: Int): Boolean {
        return property.get()
            .orEmpty()
            .contains(shard)
    }
}
