package ru.yandex.direct.logicprocessor.processors.mysql2grut.service

import com.google.common.collect.ArrayListMultimap
import com.google.common.collect.Multimap
import org.slf4j.LoggerFactory
import org.springframework.stereotype.Component
import ru.yandex.direct.core.entity.bidmodifier.AgeType
import ru.yandex.direct.core.entity.bidmodifier.BidModifier
import ru.yandex.direct.core.entity.bidmodifier.BidModifierABSegment
import ru.yandex.direct.core.entity.bidmodifier.BidModifierAdjustment
import ru.yandex.direct.core.entity.bidmodifier.BidModifierBannerType
import ru.yandex.direct.core.entity.bidmodifier.BidModifierContentDuration
import ru.yandex.direct.core.entity.bidmodifier.BidModifierDemographics
import ru.yandex.direct.core.entity.bidmodifier.BidModifierDemographicsAdjustment
import ru.yandex.direct.core.entity.bidmodifier.BidModifierDesktop
import ru.yandex.direct.core.entity.bidmodifier.BidModifierDesktopOnly
import ru.yandex.direct.core.entity.bidmodifier.BidModifierGeo
import ru.yandex.direct.core.entity.bidmodifier.BidModifierInventory
import ru.yandex.direct.core.entity.bidmodifier.BidModifierMobile
import ru.yandex.direct.core.entity.bidmodifier.BidModifierPerformanceTgo
import ru.yandex.direct.core.entity.bidmodifier.BidModifierPrismaIncomeGrade
import ru.yandex.direct.core.entity.bidmodifier.BidModifierRetargeting
import ru.yandex.direct.core.entity.bidmodifier.BidModifierRetargetingFilter
import ru.yandex.direct.core.entity.bidmodifier.BidModifierSmartTV
import ru.yandex.direct.core.entity.bidmodifier.BidModifierTablet
import ru.yandex.direct.core.entity.bidmodifier.BidModifierTrafaretPosition
import ru.yandex.direct.core.entity.bidmodifier.BidModifierTraffic
import ru.yandex.direct.core.entity.bidmodifier.BidModifierType
import ru.yandex.direct.core.entity.bidmodifier.BidModifierVideo
import ru.yandex.direct.core.entity.bidmodifier.BidModifierWeather
import ru.yandex.direct.core.entity.bidmodifiers.repository.BidModifierLevel
import ru.yandex.direct.core.entity.bidmodifiers.repository.BidModifierRepository
import ru.yandex.direct.core.entity.bidmodifiers.service.BidModifierService
import ru.yandex.direct.core.grut.api.BidModifierGrut
import ru.yandex.direct.core.mysql2grut.repository.BidModifierReplicationRepository
import ru.yandex.direct.ess.logicobjects.mysql2grut.BidModifierTableType
import ru.yandex.direct.logicprocessor.processors.mysql2grut.replicationwriter.BidModifierWriterObject

@Component
class BidModifierReplicationService(
    private val bidModifierRepository: BidModifierRepository,
    private val bidModifierService: BidModifierService,
    private val bidModifierReplicationRepository: BidModifierReplicationRepository,
) {

    companion object {
        private val logger = LoggerFactory.getLogger(BidModifierReplicationService::class.java)

        // Фильтр на типы корректировок, которые на данный момент поддержаны в репликации
        val supportedBidModifierTypes = setOf(
            // multi-valued bid-modifiers
            BidModifierType.MOBILE_MULTIPLIER,
            BidModifierType.TABLET_MULTIPLIER,
            BidModifierType.DEMOGRAPHY_MULTIPLIER,
            BidModifierType.RETARGETING_MULTIPLIER,
            BidModifierType.RETARGETING_FILTER,
            BidModifierType.GEO_MULTIPLIER,
            BidModifierType.AB_SEGMENT_MULTIPLIER,
            BidModifierType.BANNER_TYPE_MULTIPLIER,
            BidModifierType.INVENTORY_MULTIPLIER,
            BidModifierType.WEATHER_MULTIPLIER,
            BidModifierType.PRISMA_INCOME_GRADE_MULTIPLIER,
            BidModifierType.TRAFARET_POSITION_MULTIPLIER,
            // single-valued bid-modifiers
            BidModifierType.DESKTOP_ONLY_MULTIPLIER,
            BidModifierType.DESKTOP_MULTIPLIER,
            BidModifierType.PERFORMANCE_TGO_MULTIPLIER,
            BidModifierType.SMARTTV_MULTIPLIER,
            BidModifierType.VIDEO_MULTIPLIER,
        )

        /**
         * Таблицы уточнений корректировок, по которым однозначно определяется тип коректировки
         */
        private val TABLE_TO_BID_MODIFIER_TYPE = mapOf(
            BidModifierTableType.GEO to BidModifierType.GEO_MULTIPLIER,
            BidModifierTableType.TABLET to BidModifierType.TABLET_MULTIPLIER,
            BidModifierTableType.MOBILE to BidModifierType.MOBILE_MULTIPLIER,
            BidModifierTableType.WEATHER to BidModifierType.WEATHER_MULTIPLIER,
            BidModifierTableType.AB_SEGMENT to BidModifierType.AB_SEGMENT_MULTIPLIER,
            BidModifierTableType.BANNER_TYPE to BidModifierType.BANNER_TYPE_MULTIPLIER,
            BidModifierTableType.DEMOGRAPHY to BidModifierType.DEMOGRAPHY_MULTIPLIER,
            BidModifierTableType.INVENTORY to BidModifierType.INVENTORY_MULTIPLIER,
            BidModifierTableType.TRAFARET_POSITION to BidModifierType.TRAFARET_POSITION_MULTIPLIER,
        )

        fun toMultiMap(parentBidModifiers: Collection<BidModifier>): Multimap<BidModifierType, Long> {
            val result = ArrayListMultimap.create<BidModifierType, Long>()
            parentBidModifiers
                .groupBy({ it.type!! }, { it.id!! })
                .forEach { (k, v) -> result.putAll(k, v) }
            return result
        }

        /**
         * По корректировке, для которой нет записей в -values таблицах, вытащить adjustment
         */
        fun extractSingleAdjustment(bidModifier: BidModifier): BidModifierAdjustment {
            return when (bidModifier) {
                is BidModifierDesktopOnly -> bidModifier.desktopOnlyAdjustment
                is BidModifierDesktop -> bidModifier.desktopAdjustment
                is BidModifierPerformanceTgo -> bidModifier.performanceTgoAdjustment
                is BidModifierSmartTV -> bidModifier.smartTVAdjustment
                is BidModifierVideo -> bidModifier.videoAdjustment
                is BidModifierMobile -> bidModifier.mobileAdjustment
                is BidModifierTablet -> bidModifier.tabletAdjustment
                else -> throw RuntimeException("unsupported single-valued bid modifier: type ${bidModifier.type}, id ${bidModifier.id}")
            }
        }


        /**
         * По корректировке, реализующей BidModifierAdjustmentMultiple вытащить записи из -values таблиц
         */
        fun extractMultipleAdjustments(bidModifier: BidModifier): Collection<BidModifierAdjustment> {
            return when (bidModifier) {
                is BidModifierDemographics -> bidModifier.demographicsAdjustments
                is BidModifierMobile -> listOf(bidModifier.mobileAdjustment)
                is BidModifierTablet -> listOf(bidModifier.tabletAdjustment)
                is BidModifierRetargeting -> bidModifier.retargetingAdjustments
                is BidModifierRetargetingFilter -> bidModifier.retargetingAdjustments
                is BidModifierGeo -> bidModifier.regionalAdjustments
                is BidModifierABSegment -> bidModifier.abSegmentAdjustments
                is BidModifierBannerType -> bidModifier.bannerTypeAdjustments
                is BidModifierInventory -> bidModifier.inventoryAdjustments
                is BidModifierWeather -> bidModifier.weatherAdjustments
                is BidModifierTraffic -> bidModifier.expressionAdjustments
                is BidModifierContentDuration -> bidModifier.expressionAdjustments
                is BidModifierPrismaIncomeGrade -> bidModifier.expressionAdjustments
                is BidModifierTrafaretPosition -> bidModifier.trafaretPositionAdjustments
                else -> throw RuntimeException("unsupported multi-valued bid modifier: type ${bidModifier.type}, id ${bidModifier.id}")
            }
        }
    }

    /**
     * По объектам роутера получить прогруженные корректировки
     */
    fun getBidModifiers(shard: Int, bidModifiersIdsByTableType: Map<BidModifierTableType, Collection<Long>>): List<BidModifier> {
        if(bidModifiersIdsByTableType.isEmpty()) return emptyList()
        // Группируем таблица - идентификаторы
        val emptyBidModifiers = mutableSetOf<BidModifier>()

        // Получить тип корректировки по ее идентификатору в таблице hierarchical_multipliers
        if (bidModifiersIdsByTableType.containsKey(BidModifierTableType.PARENT)) {
            val parentIds = bidModifiersIdsByTableType[BidModifierTableType.PARENT]!!.distinct()
            // Берем объекты корректировок без уточнений по ключу hierarchical_multipliers
            val parentBidModifiers: List<BidModifier> = bidModifierRepository
                .getEmptyBidModifiersByIds(shard, parentIds)
            emptyBidModifiers.addAll(parentBidModifiers)
        }
        // Получить тип корректировки из двух возможных в таблице retargeting_multiplier_values (retargeting и retargeting filter)
        if (bidModifiersIdsByTableType.containsKey(BidModifierTableType.RETARGETING)) {
            val retargetingValuesIds = bidModifiersIdsByTableType[BidModifierTableType.RETARGETING]!!.distinct()
            // Берем корректировки без уточнений по retargeting_id чтобы узнать тип (retargeting и retargeting filter)
            val parentBidModifiers = bidModifierReplicationRepository.getEmptyAmbiguousBidModifiers(shard,
                BidModifierReplicationRepository.AmbiguousBidModifierTable.RETARGETING_OR_RETARGEING_FILTER,
                retargetingValuesIds)
            emptyBidModifiers.addAll(parentBidModifiers)
        }
        // Получить тип корректировки из возможных (express_traffic_multiplier, prisma_income_grade_multiplier)
        // из таблицы expression_multiplier_values
        if (bidModifiersIdsByTableType.containsKey(BidModifierTableType.EXPRESSION)) {
            val expressionValuesIds = bidModifiersIdsByTableType[BidModifierTableType.EXPRESSION]!!.distinct()
            val parentBidModifiers = bidModifierReplicationRepository.getEmptyAmbiguousBidModifiers(shard,
                BidModifierReplicationRepository.AmbiguousBidModifierTable.EXPRESSION,
                expressionValuesIds)
            emptyBidModifiers.addAll(parentBidModifiers)
        }
        val multimap = toMultiMap(emptyBidModifiers)
        // Для всех оставшихся записей мы однозначно определяем тип по таблице c уточнением
        bidModifiersIdsByTableType
            .filterKeys { TABLE_TO_BID_MODIFIER_TYPE.containsKey(it) }
            .forEach{multimap.putAll(TABLE_TO_BID_MODIFIER_TYPE[it.key], it.value)}

        val unsupportedBidModifiers = multimap.keys().filter { !supportedBidModifierTypes.contains(it) }
        if (unsupportedBidModifiers.isNotEmpty()) {
            logger.warn("Unsupported bid modifier types filtered: $unsupportedBidModifiers")
            unsupportedBidModifiers.forEach { multimap.removeAll(it) }
        }

        if (multimap.isEmpty) {
            return emptyList()
        }

        // Получаем все уточнения для всех корректировок
        return bidModifierRepository.getAllByIds(shard, multimap, supportedBidModifierTypes)
    }

    fun getGrutIdsForAdGroups(shard: Int, adGroupsIds: Set<Long>): Map<Long, Set<BidModifierId>> {

        val bidModifiers = bidModifierService.getByAdGroupIds(
            shard,
            adGroupsIds,
            supportedBidModifierTypes,
            setOf(BidModifierLevel.ADGROUP),
        ).filter { it.enabled } // если корректировка стала enabled = false, убираем ее из связи
        return getGrutIds(bidModifiers)
    }


    /**
     * По корректировкам в Директе получить их идентификаторый в Груте
     */
    private fun getGrutIds(bidModifiers: Collection<BidModifier>): Map<Long, Set<BidModifierId>> {
        val result = mutableMapOf<Long, MutableSet<BidModifierId>>()
        val (singleValued, multiValued) = bidModifiers.partition { isSingleValued(it) }
        singleValued.forEach {
            val adjustmentId = extractSingleAdjustment(it).id
            result.getOrPut(it.adGroupId) { mutableSetOf() }.add(BidModifierId(adjustmentId))
        }
        multiValued.forEach {
            val adjustmentsIds = extractMultipleAdjustments(it)
                .map { a -> BidModifierId(a.id, it.id, true) }
            result.getOrPut(it.adGroupId) { mutableSetOf() }.addAll(adjustmentsIds)
        }
        return result
    }

    /**
     * Является ли корректировка "простой" (без записей в values-таблицах)
     */
    fun isSingleValued(bidModifier: BidModifier): Boolean = !bidModifierReplicationRepository.typeSupportDispatcher
        .getTypeSupport<BidModifier, BidModifierAdjustment>(bidModifier.type).isMultiValues


    fun getExisingInMysqlBidModifiers(shard: Int, objects: Collection<BidModifierWriterObject>): Set<Long> {
        val existingBidModifiersIds = mutableSetOf<Long>()
        val bidModifierIdsByType = objects.groupBy({ it.tableType }, { it.id })
        for ((tableType, ids) in bidModifierIdsByType) {
            when {
                // hierarchical_multiplier - возвращаем hierarchical_multiplier_id, которые еще существуют в Директе
                tableType == BidModifierTableType.PARENT ->
                    bidModifierRepository.getBidModifierKeysByIds(shard, ids.distinct())
                        .map { it.key!! }
                        .toCollection(existingBidModifiersIds)
                // Идентификаторы expression_multiplier_value_id
                // Остальные типы с типом expression (content_duration, traffic_jam) сейчас не поддерживаются
                tableType == BidModifierTableType.EXPRESSION ->
                    bidModifierRepository
                        .getExistingMultiValuedBidModifiersIds(shard, BidModifierType.PRISMA_INCOME_GRADE_MULTIPLIER, ids)
                        .toCollection(existingBidModifiersIds)
                // Идентификаторы retargeting_mulitplier_value_id
                // В таблице RETARGETING_MULTIPLIER_VALUES лежат корректировки типа retargeting и retargeting_filter
                // метод проверяет существование объектов обоих типов так как вызывает метод из абстрактного класса
                // getAdjustmentsByIds AbstractBidModifierRetargetingTypeSupport, покрыто тестом
                tableType == BidModifierTableType.RETARGETING -> bidModifierRepository
                    .getExistingMultiValuedBidModifiersIds(shard, BidModifierType.RETARGETING_MULTIPLIER, ids)
                    .toCollection(existingBidModifiersIds)
                // таблицы, по которым однозначно определяется тип
                TABLE_TO_BID_MODIFIER_TYPE.containsKey(tableType) -> bidModifierRepository
                    .getExistingMultiValuedBidModifiersIds(shard, TABLE_TO_BID_MODIFIER_TYPE[tableType]!!, ids)
                    .toCollection(existingBidModifiersIds)
                else -> throw RuntimeException("Unsupported bid modifier table: $tableType")
            }
        }
        return existingBidModifiersIds
    }
}

data class BidModifierId(val grutId: Long, val hierarchicalMultiplierId: Long = grutId, val isMultiValued: Boolean = false)
