package ru.yandex.direct.logicprocessor.processors.promocodescheckcampaignchanges;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nullable;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import ru.yandex.direct.balance.client.exception.BalanceClientException;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.promocodes.model.CampPromocodes;
import ru.yandex.direct.core.entity.promocodes.model.PromocodeInfo;
import ru.yandex.direct.core.entity.promocodes.model.TearOffReason;
import ru.yandex.direct.core.entity.promocodes.repository.CampPromocodesRepository;
import ru.yandex.direct.core.entity.promocodes.service.PromocodesAntiFraudService;
import ru.yandex.direct.ess.logicobjects.promocodescheckcampaignchanges.IdTypeEnum;
import ru.yandex.direct.ess.logicobjects.promocodescheckcampaignchanges.PromocodesCheckCampaignChangesObject;


@Service
public class PromocodesCheckCampaignChangesService {

    private static final Logger logger = LoggerFactory.getLogger(PromocodesCheckCampaignChangesService.class);

    private final CampPromocodesRepository campPromocodesRepository;
    private final CampaignRepository campaignRepository;
    private final AdGroupRepository adGroupRepository;
    private final PromocodesAntiFraudService antiFraudService;
    private final int directServiceId;

    @Autowired
    public PromocodesCheckCampaignChangesService(CampPromocodesRepository campPromocodesRepository,
                                                 CampaignRepository campaignRepository,
                                                 AdGroupRepository adGroupRepository,
                                                 PromocodesAntiFraudService antiFraudService,
                                                 @Value("${balance.directServiceId}") int directServiceId) {

        this.campPromocodesRepository = campPromocodesRepository;
        this.campaignRepository = campaignRepository;
        this.adGroupRepository = adGroupRepository;
        this.antiFraudService = antiFraudService;
        this.directServiceId = directServiceId;
    }

    public List<Long> processObjects(int shard, List<PromocodesCheckCampaignChangesObject> objects) {
        Set<Long> cids = new HashSet<>();
        Set<Long> pids = new HashSet<>();
        for (var object : objects) {
            if (object.getIdType() == IdTypeEnum.CID) {
                cids.add(object.getId());
            } else if (object.getIdType() == IdTypeEnum.PID) {
                pids.add(object.getId());
            } else {
                throw new IllegalStateException("Unsupported id type, " + object.getIdType());
            }
        }

        var additionalCids = adGroupRepository.getCampaignIdsByAdGroupIds(shard, pids).values();
        cids.addAll(additionalCids);

        var mapOfCidToWalletIds = campaignRepository.getWalletIdsByCampaingIds(shard, cids);
        var cidsOrWalletIds = cids.stream()
                .filter(cid -> {
                    var hasKey = mapOfCidToWalletIds.containsKey(cid);
                    if (!hasKey) {
                        logger.warn("cid={} not found in result of CampaignRepository.getWalletIdsByCampaingIds", cid);
                    }
                    return hasKey;
                })
                .map(cid -> {
                    var walletId = mapOfCidToWalletIds.get(cid);
                    return walletId == 0L ? cid : walletId;
                })
                .collect(Collectors.toSet());

        List<Long> badCids = new ArrayList<>();
        for (Long cid : cidsOrWalletIds) {
            if (!checkCampaign(shard, cid)) {
                badCids.add(cid);
            }
        }
        return badCids;
    }

    /**
     * Проводит проверку промокодов кампании, в случае обнаружения нарушения правил использования делает
     * отрыв промокода.
     *
     * @param shard      номер шарда
     * @param campaignId идентификатор кампании
     * @return {@code true} если обработка прошла без ошибок,
     * иначе (если была неудачная попытка отрыва промокода) {@code false}
     */
    private boolean checkCampaign(int shard, Long campaignId) {
        logger.debug("got campaign {}", campaignId);

        boolean success = true;
        CampPromocodes campPromocodes = campPromocodesRepository.getCampaignPromocodes(shard, campaignId);
        if (needToProcess(campPromocodes) && isRestrictionsViolated(campPromocodes)) {
            List<PromocodeInfo> promocodesToTearOff = campPromocodes.getPromocodes()
                    .stream()
                    .filter(this::isApplicablePromocode)
                    .collect(Collectors.toList());
            success = tearOffPromocodes(campaignId, promocodesToTearOff);
        }
        return success;
    }

    private boolean tearOffPromocodes(Long campaignId, List<PromocodeInfo> promocodesToTearOff) {
        try {
            antiFraudService.tearOffPromocodes(directServiceId, campaignId, promocodesToTearOff,
                    TearOffReason.CAMPAIGN_CHANGES);
        } catch (BalanceClientException e) {
            logger.error("Failed to tear off promocodes from campaign " + campaignId, e);
            return false;
        }
        return true;
    }

    /**
     * Определить по сведениям о промокодах - требуется ли обработка кампании
     *
     * @param campaignPromocodes сведения о промокодах кампании
     * @return {@code true} если есть хотя бы один применимый промокод, иначе {@code false}
     */
    private boolean needToProcess(@Nullable CampPromocodes campaignPromocodes) {
        if (campaignPromocodes == null) {
            return false;
        }

        return campaignPromocodes.getPromocodes()
                .stream()
                .anyMatch(this::isApplicablePromocode);
    }

    /**
     * Нужно ли применять антифрод к данному промокоду. Условие - промокод активирован после граничной даты.
     * <p>
     * Проверка даты нужна для случая, когда между сохранением сведений о промокоде из нотификации и их обработкой здесь
     * граничную дату передвинули в будущее.
     *
     * @param promocodeInfo директовые сведения о промокоде
     * @return {@code true} если нужно, {@code false} иначе
     */
    private boolean isApplicablePromocode(PromocodeInfo promocodeInfo) {
        return promocodeInfo.getInvoiceEnabledAt().isAfter(antiFraudService.getAntiFraudStartDate());
    }

    /**
     * Проверить, нарушены ли правила использования промокода (сменился рекламируемый домен)
     *
     * @param campaignPromocodes сведения о промокоде
     * @return {@code true} если домены не совпадает, {@code false} иначе
     */
    private boolean isRestrictionsViolated(CampPromocodes campaignPromocodes) {
        String newRestrictedDomain = antiFraudService.determineRestrictedDomain(campaignPromocodes.getCampaignId());
        return !campaignPromocodes.getRestrictedDomain().equals(newRestrictedDomain);
    }
}
