package ru.yandex.direct.logicprocessor.processors.recomtracer;

import java.time.Duration;
import java.util.List;
import java.util.Set;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.core.entity.recommendation.model.RecommendationKey;
import ru.yandex.direct.env.NonProductionEnvironment;
import ru.yandex.direct.env.ProductionOnly;
import ru.yandex.direct.ess.config.recomtracer.RecomTracerConfig;
import ru.yandex.direct.ess.logicobjects.recomtracer.RecomTracerLogicObject;
import ru.yandex.direct.juggler.check.annotation.JugglerCheck;
import ru.yandex.direct.logicprocessor.common.BaseLogicProcessor;
import ru.yandex.direct.logicprocessor.common.EssLogicProcessor;
import ru.yandex.direct.logicprocessor.common.EssLogicProcessorContext;
import ru.yandex.direct.utils.JsonUtils;
import ru.yandex.direct.ytcore.entity.recommendation.service.RecommendationService;

import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.common.db.PpcPropertyNames.RECOMMENDATIONS_TRACER_ENABLED_TYPES;
import static ru.yandex.direct.juggler.check.model.CheckTag.DIRECT_PRIORITY_1;

@JugglerCheck(ttl = @JugglerCheck.Duration(minutes = 5), needCheck = ProductionOnly.class, tags = {DIRECT_PRIORITY_1})
@JugglerCheck(ttl = @JugglerCheck.Duration(minutes = 10), needCheck = NonProductionEnvironment.class, tags = {DIRECT_PRIORITY_1})
@EssLogicProcessor(RecomTracerConfig.class)
public class RecomTracerProcessor extends BaseLogicProcessor<RecomTracerLogicObject> {

    private static final Logger logger = LoggerFactory.getLogger(RecomTracerProcessor.class);

    private final PpcProperty<String> propEnabledTypes;
    private final RecomTracerService recomTracerService;

    private RecommendationService recommendationService;

    private Set<Long> enabledTypesCache;

    @Autowired
    public RecomTracerProcessor(EssLogicProcessorContext context,
                                PpcPropertiesSupport ppcPropertiesSupport,
                                RecomTracerService recomTracerService,
                                RecommendationService recommendationService) {
        super(context);
        propEnabledTypes = ppcPropertiesSupport.get(RECOMMENDATIONS_TRACER_ENABLED_TYPES, Duration.ofMinutes(1));
        this.recomTracerService = recomTracerService;
        this.recommendationService = recommendationService;
    }

    @Override
    public void process(List<RecomTracerLogicObject> recomTracerLogicObjects) {
        Set<Long> typesEnabled = loadTypesEnabled();

        Set<RecomTracerLogicObject> recomTracerObjectsWithEnabledTypes = recomTracerLogicObjects.stream()
                .filter(recomTracerObject -> typesEnabled.contains(recomTracerObject.getRecommendationTypeId()))
                .collect(toSet());

        Set<RecommendationKey> recommendationKeysToCancel =
                recomTracerService.getRecommendationKeysToCancel(getShard(), recomTracerObjectsWithEnabledTypes);

        if (!recommendationKeysToCancel.isEmpty()) {
            int cancelledCnt = recommendationService.cancel(getShard(), recommendationKeysToCancel);
            logger.info("Cancelled {} recommendations of {} for shard {}", cancelledCnt,
                    recommendationKeysToCancel.size(), getShard());
        }
    }

    private Set<Long> loadTypesEnabled() {
        // Дневной бюджет по умолчанию включен
        var newTypes = Set.of(JsonUtils.fromJson(propEnabledTypes.getOrDefault("[1]"), Long[].class));

        if (enabledTypesCache != null && !newTypes.equals(enabledTypesCache)) {
            logger.info("Enabled types changed. Was {}, now {}", enabledTypesCache, newTypes);
        }
        enabledTypesCache = newTypes;
        return newTypes;
    }
}
