package ru.yandex.direct.logicprocessor.processors.recomtracer;

import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nonnull;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.recommendation.model.RecommendationKey;
import ru.yandex.direct.core.entity.recomtracer.container.BannersLoadedObject;
import ru.yandex.direct.core.entity.recomtracer.container.CampaignsLoadedObject;
import ru.yandex.direct.core.entity.recomtracer.container.LoadedObject;
import ru.yandex.direct.core.entity.recomtracer.container.PhrasesLoadedObject;
import ru.yandex.direct.core.entity.recomtracer.repository.RecomTracerRepository;
import ru.yandex.direct.ess.common.utils.TablesEnum;
import ru.yandex.direct.ess.logicobjects.recomtracer.RecomTracerLogicObject;
import ru.yandex.direct.ess.logicobjects.recomtracer.RecommendationKeyIdentifier;
import ru.yandex.direct.grid.core.entity.recommendation.model.GdiRecommendation;
import ru.yandex.direct.grid.core.entity.recommendation.repository.GridRecommendationYtRepository;
import ru.yandex.direct.logicprocessor.processors.recomtracer.cancellers.RecommendationCanceller;

import static java.util.stream.Collectors.groupingBy;
import static java.util.stream.Collectors.mapping;
import static java.util.stream.Collectors.toMap;
import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.logicprocessor.processors.recomtracer.utils.RecommendationKeyUtils.getAllPossiblePrefixesFromGdiKeys;

@Service
public class RecomTracerService {

    private final RecomTracerRepository recomTracerRepository;
    private final GridRecommendationYtRepository recommendationYtRepository;
    private final Map<Long, RecommendationCanceller> supportedTypeToRecommendationCancellers;

    @Autowired
    public RecomTracerService(RecomTracerRepository recomTracerRepository,
                              GridRecommendationYtRepository recommendationYtRepository,
                              List<RecommendationCanceller> recommendationCancellers) {
        this.recomTracerRepository = recomTracerRepository;
        this.recommendationYtRepository = recommendationYtRepository;
        this.supportedTypeToRecommendationCancellers = recommendationCancellers.stream()
                .collect(toMap(
                        recommendationCanceller -> recommendationCanceller.supportedType().getId(),
                        recommendationCanceller -> recommendationCanceller
                ));

    }

    Set<RecommendationKey> getRecommendationKeysToCancel(int shard,
                                                         @Nonnull Set<RecomTracerLogicObject> recomTracerLogicObjects) {
        Map<RecomTracerLogicObject, RecommendationKey> recomTracerObjectsToKeyPrefixes =
                getRecommendationKeysPrefixForObjects(shard, recomTracerLogicObjects);

        Set<RecommendationKey> allRecommendationKeyPrefixes = new HashSet<>(recomTracerObjectsToKeyPrefixes.values());

        /* Получаем активные рекомендации для всех префиксов */
        Set<GdiRecommendation> activeGdiRecommendations =
                recommendationYtRepository.getRecommendationsByKeyPrefixes(allRecommendationKeyPrefixes, shard);

        /* Для каждого префикса ищем все полные ключи рекомендаций */
        Map<RecommendationKey, List<RecommendationKey>> activePrefixToFullKeys =
                getAllPossiblePrefixesFromGdiKeys(activeGdiRecommendations);

        return getRecommendationKeysToCancelFromPrefixes(activePrefixToFullKeys,
                recomTracerObjectsToKeyPrefixes);
    }

    /**
     * Для тех объектов, которые не требуют загрузки из mysql, префиксы ключей рекомендации формируются на основе
     * самого объекта
     * Для объектов, которым для формирования префикса рекомендации нужно получить свойства из mysql,
     * сначала выполнялется запрос в БД, согласно таблице для загрузки, после чего формуруеетя префикс рекомендации
     * на основе самого объекта и {@link LoadedObject}. Если из mysql для объекта ничего не вернулось - он не будет
     * дальше обрабатываться
     *
     * @return сопосталение объект -> префикс рекомендации
     */
    Map<RecomTracerLogicObject, RecommendationKey> getRecommendationKeysPrefixForObjects(int shard,
                                                                                         Set<RecomTracerLogicObject> recomTracerLogicObjects) {
        Map<RecomTracerLogicObject, RecommendationKey> objectToKeyPrefix =
                getRecommendationKeysPrefixForObjectsWithoutLoad(recomTracerLogicObjects);

        Map<TablesEnum, Map<Long, LoadedObject>> loadedObjectForTableAndPrimaryKey =
                loadEntitiesForRecomTracerObjects(shard, recomTracerLogicObjects);

        recomTracerLogicObjects.stream()
                .filter(RecomTracerLogicObject::isNeedLoad)
                .filter(recomTracerObject -> loadedObjectForTableAndPrimaryKey.get(recomTracerObject.getTableToLoad()).containsKey(recomTracerObject.getPrimaryKey()))
                .forEach(recomTracerObject -> {
                    LoadedObject loadedObject =
                            loadedObjectForTableAndPrimaryKey.get(recomTracerObject.getTableToLoad()).get(recomTracerObject.getPrimaryKey());
                    RecommendationKey recommendationKey = getRecommendationKeyPrefixFromRecomTracerObject(loadedObject,
                            recomTracerObject);
                    objectToKeyPrefix.put(recomTracerObject, recommendationKey);
                });
        return objectToKeyPrefix;
    }

    private Map<RecomTracerLogicObject, RecommendationKey> getRecommendationKeysPrefixForObjectsWithoutLoad(Set<RecomTracerLogicObject> recomTracerLogicObjects) {
        return recomTracerLogicObjects.stream()
                .filter(recomTracerObject -> !recomTracerObject.isNeedLoad())
                .collect(toMap(
                        recomTracerObject -> recomTracerObject,
                        RecomTracerService::getRecommendationKeyPrefixFromRecomTracerObject
                ));
    }

    private Map<TablesEnum, Map<Long, LoadedObject>> loadEntitiesForRecomTracerObjects(int shard,
                                                                                       Set<RecomTracerLogicObject> recomTracerLogicObjects) {
        Map<TablesEnum, Set<Long>> tableToPrimaryKeysToLoad = recomTracerLogicObjects.stream()
                .filter(RecomTracerLogicObject::isNeedLoad)
                .collect(groupingBy(RecomTracerLogicObject::getTableToLoad,
                        mapping(RecomTracerLogicObject::getPrimaryKey, toSet())));
        return tableToPrimaryKeysToLoad.entrySet().stream()
                .collect(toMap(
                        Map.Entry::getKey,
                        tableToPrimaryKeysToLoadEntry ->
                                loadMysqlEntitiesForTable(shard, tableToPrimaryKeysToLoadEntry.getKey(),
                                        tableToPrimaryKeysToLoadEntry.getValue())
                ));
    }

    /**
     * 1) Из всех полученных RecomTracerLogicObject выбираются те, префикс ключа рекомендации которых активен
     * 2) Для каждого RecomTracerLogicObject берется нужный для его типа рекомендации обработчик
     * {@link RecommendationCanceller}
     * Каждый полный ключ рекомендации передается обработчику
     * 3) Отбираются только те полные ключи рекомендации, для которых обраотчик вернул - отменить
     *
     * @param activePrefixToFullKeys       - маппинг активных префиксов ключей к списку полных ключей
     * @param recomTracerObjectToKeyPrefix - маппинг объектов для обработки к ключам рекомендаций
     * @return все рекомендации, которые необходитмо отменить
     */
    private Set<RecommendationKey> getRecommendationKeysToCancelFromPrefixes(Map<RecommendationKey,
            List<RecommendationKey>> activePrefixToFullKeys, Map<RecomTracerLogicObject, RecommendationKey> recomTracerObjectToKeyPrefix) {

        return recomTracerObjectToKeyPrefix.entrySet().stream()
                .filter(recomTracerObjectToKey -> activePrefixToFullKeys.containsKey(recomTracerObjectToKey.getValue()))
                .flatMap(recomTracerObjectToKey -> {
                    List<RecommendationKey> recommendationKeysForObject =
                            activePrefixToFullKeys.get(recomTracerObjectToKey.getValue());
                    return recommendationKeysForObject.stream()
                            .map(recommendationKeyForObject ->
                                    new RecommendationKeyWithAdditionalColumns(recommendationKeyForObject,
                                            recomTracerObjectToKey.getKey().getAdditionalColumns()));
                })
                .filter(keyWithAdditionalColumns -> supportedTypeToRecommendationCancellers.get(keyWithAdditionalColumns.getRecommendationKey().getType()).recommendationsToCancel(keyWithAdditionalColumns))
                .map(RecommendationKeyWithAdditionalColumns::getRecommendationKey)
                .collect(toSet());
    }

    /**
     * Загружает недостающие сущности для построения ключа из MYSQL
     *
     * @return маппинг первичный ключ таблицы -> объект с недостающими сущностями
     */
    private Map<Long, LoadedObject> loadMysqlEntitiesForTable(int shard, TablesEnum table, Set<Long> primaryKeys) {
        switch (table) {
            case CAMPAIGNS:
                return recomTracerRepository.loadCampaignsEntities(shard, primaryKeys);
            case PHRASES:
                return recomTracerRepository.loadPhrasesEntities(shard, primaryKeys);
            case BANNERS:
                return recomTracerRepository.loadBannerEntities(shard, primaryKeys);
            default:
                throw new IllegalStateException("Unsupported table for method loadMysqlEntitiesForTable" + table);
        }
    }

    /**
     * Если все сущности, необходимы для ключа, уже есть в {@link RecomTracerLogicObject}
     * , то запрос в MYSQL не делается, все данные для префикса ключа рекомендации берутся из объекта
     */
    static RecommendationKey getRecommendationKeyPrefixFromRecomTracerObject(RecomTracerLogicObject recomTracerLogicObject) {
        return new RecommendationKey()
                .withType(recomTracerLogicObject.getRecommendationTypeId())
                .withClientId(recomTracerLogicObject.getRecommendationKeyIdentifier(RecommendationKeyIdentifier.CLIENT_ID))
                .withCampaignId(recomTracerLogicObject.getRecommendationKeyIdentifier(RecommendationKeyIdentifier.CID))
                .withAdGroupId(recomTracerLogicObject.getRecommendationKeyIdentifier(RecommendationKeyIdentifier.PID))
                .withBannerId(recomTracerLogicObject.getRecommendationKeyIdentifier(RecommendationKeyIdentifier.BID));

    }

    /**
     * Создает ключ рекомендации из свойств полученного объекта и объекта, загруженного из MYSQL
     *
     * @param loadedObject           объект из mysql
     * @param recomTracerLogicObject объект из топика
     */
    static RecommendationKey getRecommendationKeyPrefixFromRecomTracerObject(LoadedObject loadedObject,
                                                                             RecomTracerLogicObject recomTracerLogicObject) {
        switch (recomTracerLogicObject.getTableToLoad()) {
            case CAMPAIGNS:
                return getRecommendationKeyForCampaigns(loadedObject, recomTracerLogicObject);
            case PHRASES:
                return getRecommendationKeyForPhrases(loadedObject, recomTracerLogicObject);
            case BANNERS:
                return getRecommendationKeyForBanners(loadedObject, recomTracerLogicObject);
            default:
                throw new IllegalStateException("Unsupported table for method " +
                        "getRecommendationKeyPrefixFromRecomTracerObject" + recomTracerLogicObject.getTableToLoad());
        }
    }

    /**
     * Создает ключ рекомендации, если таблица для загрузки из Mysql была CAMPAIGNS
     */
    private static RecommendationKey getRecommendationKeyForCampaigns(LoadedObject loadedObject,
                                                                      RecomTracerLogicObject recomTracerLogicObject) {
        CampaignsLoadedObject campaignsLoadedObject = (CampaignsLoadedObject) loadedObject;
        RecommendationKey recommendationKey = new RecommendationKey()
                .withType(recomTracerLogicObject.getRecommendationTypeId())
                .withCampaignId(campaignsLoadedObject.getCid())
                .withClientId(campaignsLoadedObject.getClientId());
        if (recomTracerLogicObject.isRecommendationKeyIdentifierPresent(RecommendationKeyIdentifier.PID)) {
            Long pid = recomTracerLogicObject.getRecommendationKeyIdentifier(RecommendationKeyIdentifier.PID);
            recommendationKey.withAdGroupId(pid);
        }
        if (recomTracerLogicObject.isRecommendationKeyIdentifierPresent(RecommendationKeyIdentifier.BID)) {
            Long bid = recomTracerLogicObject.getRecommendationKeyIdentifier(RecommendationKeyIdentifier.BID);
            recommendationKey.withBannerId(bid);
        }
        return recommendationKey;
    }

    /**
     * Создает ключ рекомендации, если таблица для загрузки из Mysql была PHRASES
     */
    private static RecommendationKey getRecommendationKeyForPhrases(LoadedObject loadedObject,
                                                                    RecomTracerLogicObject recomTracerLogicObject) {
        PhrasesLoadedObject phrasesLoadedObject = (PhrasesLoadedObject) loadedObject;
        RecommendationKey recommendationKey = new RecommendationKey()
                .withType(recomTracerLogicObject.getRecommendationTypeId())
                .withClientId(phrasesLoadedObject.getClientId())
                .withCampaignId(phrasesLoadedObject.getCid())
                .withAdGroupId(phrasesLoadedObject.getPid());

        /* Если все баннеры для группы архивные - надо отменить рекомендацию на всю группу */
        if (recomTracerLogicObject.isRecommendationKeyIdentifierPresent(RecommendationKeyIdentifier.BID)
                && !phrasesLoadedObject.getAllBannersArchived()) {
            Long bid = recomTracerLogicObject.getRecommendationKeyIdentifier(RecommendationKeyIdentifier.BID);
            recommendationKey.withBannerId(bid);
        }
        return recommendationKey;
    }

    /**
     * Создает ключ рекомендации, если таблица для загрузки из Mysql была BANNERS
     */
    private static RecommendationKey getRecommendationKeyForBanners(LoadedObject loadedObject,
                                                                    RecomTracerLogicObject recomTracerLogicObject) {
        BannersLoadedObject bannersLoadedObject = (BannersLoadedObject) loadedObject;
        return new RecommendationKey()
                .withType(recomTracerLogicObject.getRecommendationTypeId())
                .withClientId(bannersLoadedObject.getClientId())
                .withCampaignId(bannersLoadedObject.getCid())
                .withAdGroupId(bannersLoadedObject.getPid())
                .withBannerId(bannersLoadedObject.getBid());
    }
}

