package ru.yandex.direct.logicprocessor.processors.recomtracer.utils;

import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.IntStream;
import java.util.stream.Stream;

import org.apache.commons.lang3.tuple.Pair;

import ru.yandex.direct.core.entity.recommendation.model.RecommendationKey;
import ru.yandex.direct.grid.core.entity.recommendation.model.GdiRecommendation;

import static com.google.common.base.Preconditions.checkArgument;
import static java.util.stream.Collectors.groupingBy;
import static java.util.stream.Collectors.mapping;
import static java.util.stream.Collectors.toList;

public class RecommendationKeyUtils {

    private RecommendationKeyUtils() {
    }

    /**
     * Для всех активных рекомендаций считает всех их возможные префиксы
     * [clientId], [clientId, cid], [clientId, cid, pid], [clientId, cid, pid, bid]
     *
     * @return маппинг префикса к списку всех полных ключей
     */
    public static Map<RecommendationKey, List<RecommendationKey>> getAllPossiblePrefixesFromGdiKeys(Set<GdiRecommendation> gdiRecommendations) {
        return gdiRecommendations.stream()
                .map(RecommendationKeyUtils::getRecommendationKeyFromGdiRecommendation)
                .flatMap(RecommendationKeyUtils::getAllPrefixesFromFullKey)
                .collect(groupingBy(
                        Pair::getLeft,
                        mapping(Pair::getRight, toList())
                ));
    }

    /**
     * Возвращает поток пар префикс ключа -> полный ключ
     *
     * @param fullRecommendationKey полный ключ, из которых будут получены префиксы
     */
    private static Stream<Pair<RecommendationKey, RecommendationKey>> getAllPrefixesFromFullKey(RecommendationKey fullRecommendationKey) {
        return IntStream.rangeClosed(2, 4)
                .boxed()
                .map(prefixLen -> getKeyPrefixWithLengthFromFullKey(fullRecommendationKey, prefixLen))
                .map(recommendationKeyPrefix -> Pair.of(recommendationKeyPrefix, fullRecommendationKey));
    }

    /**
     * Пребразует {@link GdiRecommendation} в ключ рекомендации с указанной длиной
     */
    static RecommendationKey getKeyPrefixWithLengthFromFullKey(RecommendationKey fullRecommendationKey,
                                                               int prefixLength) {
        checkArgument(prefixLength > 0, "Recommendation key prefix must be greater then 0");
        RecommendationKey recommendationKey = new RecommendationKey()
                .withClientId(fullRecommendationKey.getClientId())
                .withType(fullRecommendationKey.getType());

        if (prefixLength > 1) {
            recommendationKey.withCampaignId(fullRecommendationKey.getCampaignId());
        }
        if (prefixLength > 2) {
            recommendationKey.withAdGroupId(fullRecommendationKey.getAdGroupId());
        }
        if (prefixLength > 3) {
            recommendationKey.withBannerId(fullRecommendationKey.getBannerId());
        }
        return recommendationKey;

    }

    /**
     * Преобразовывает {@link GdiRecommendation} в полный ключ рекомендации
     */
    public static RecommendationKey getRecommendationKeyFromGdiRecommendation(GdiRecommendation gdiRecommendation) {
        return new RecommendationKey()
                .withType(gdiRecommendation.getType().getId())
                .withClientId(gdiRecommendation.getClientId())
                .withCampaignId(gdiRecommendation.getCid())
                .withAdGroupId(gdiRecommendation.getPid())
                .withBannerId(gdiRecommendation.getBid())
                .withUserKey1(gdiRecommendation.getUserKey1())
                .withUserKey2(gdiRecommendation.getUserKey2())
                .withUserKey3(gdiRecommendation.getUserKey3())
                .withTimestamp(gdiRecommendation.getTimestamp());
    }
}
