package ru.yandex.direct.logicprocessor.processors.uac.updatestatuses

import java.time.Duration
import org.slf4j.Logger
import org.slf4j.LoggerFactory
import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod
import ru.yandex.direct.common.db.PpcPropertiesSupport
import ru.yandex.direct.common.db.PpcPropertyNames.GRUT_CAMPAIGNS_BATCH_SIZE
import ru.yandex.direct.common.db.PpcPropertyNames.PROCESS_WATCHLOG_IN_UAC_PROCESSOR
import ru.yandex.direct.core.entity.banner.repository.BannerRelationsRepository
import ru.yandex.direct.core.entity.campaign.AvailableCampaignSources
import ru.yandex.direct.core.entity.campaign.model.CommonCampaign
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository
import ru.yandex.direct.core.entity.uac.service.GrutCampaignContentUpdateService
import ru.yandex.direct.env.ProductionOnly
import ru.yandex.direct.ess.config.uac.updatestatuses.UacUpdateStatusesConfig
import ru.yandex.direct.ess.logicobjects.uac.updatestatuses.UacUpdateStatusesObject
import ru.yandex.direct.juggler.JugglerStatus
import ru.yandex.direct.juggler.check.annotation.JugglerCheck
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification
import ru.yandex.direct.juggler.check.model.CheckTag
import ru.yandex.direct.juggler.check.model.NotificationRecipient
import ru.yandex.direct.logicprocessor.common.BaseLogicProcessor
import ru.yandex.direct.logicprocessor.common.EssLogicProcessor
import ru.yandex.direct.logicprocessor.common.EssLogicProcessorContext

/**
 * Джоба обновления контента и заявки UAC при изменении статусов модерации, причин отклонения и пр. в mysql
 * Обновляет статус и причины отклонения контента а так же поле content_flags на заявке
 */
@JugglerCheck(
    ttl = JugglerCheck.Duration(minutes = 10),
    needCheck = ProductionOnly::class,
    notifications = [OnChangeNotification(
        recipient = [NotificationRecipient.LOGIN_MSPIRIT],
        method = [NotificationMethod.TELEGRAM],
        status = [JugglerStatus.OK, JugglerStatus.CRIT],
    )],
    tags = [CheckTag.DIRECT_PRIORITY_0],
)
@EssLogicProcessor(UacUpdateStatusesConfig::class)
class UacUpdateStatusesProcessor(
    essLogicProcessorContext: EssLogicProcessorContext,
    ppcPropertiesSupport: PpcPropertiesSupport,
    private val bannerRelationsRepository: BannerRelationsRepository,
    private val grutCampaignContentUpdateService: GrutCampaignContentUpdateService,
    private val campaignTypedRepository: CampaignTypedRepository,
) : BaseLogicProcessor<UacUpdateStatusesObject>(essLogicProcessorContext) {

    companion object {
        private val logger: Logger = LoggerFactory.getLogger(UacUpdateStatusesProcessor::class.java)
    }

    private val campaignsChunkSizeProperty = ppcPropertiesSupport.get(GRUT_CAMPAIGNS_BATCH_SIZE, Duration.ofMinutes(5))
    private val processWatchlogEventsProperty = ppcPropertiesSupport.get(PROCESS_WATCHLOG_IN_UAC_PROCESSOR, Duration.ofMinutes(5))

    override fun process(logicObjects: List<UacUpdateStatusesObject>) {
        val processWatchlog = processWatchlogEventsProperty.getOrDefault(false)
        val watchlogFilter = { logicObject: UacUpdateStatusesObject -> processWatchlog || logicObject.fromGrut != true }

        val bannerIds = logicObjects.asSequence().filter(watchlogFilter).mapNotNull { it.bannerId }.distinct().toList()
        val campaignIds = logicObjects.asSequence().filter(watchlogFilter).mapNotNull { it.campaignId }.distinct().toList()

        val campaignIdsFromBannerIds = bannerRelationsRepository.getCampaignIdsByBannerIds(shard, bannerIds).values.distinct()

        val allCampaignIds = (campaignIdsFromBannerIds + campaignIds).distinct()
        val campaignIdsToRecalc = campaignTypedRepository
            .getSafely(shard, allCampaignIds, CommonCampaign::class.java)
            .asSequence()
            .filter { AvailableCampaignSources.isUC(it.source) }
            .map { it.id }
            .toList()

        logger.info("Going to handle ${campaignIdsToRecalc.size} campaigns")
        campaignIdsToRecalc.chunked(campaignsChunkSizeProperty.getOrDefault(50))
            .forEach {
                grutCampaignContentUpdateService.updateCampaignContentsAndCampaignFlags(shard, it)
            }
    }
}
