package ru.yandex.direct.logicprocessor.processors.userblock

import org.slf4j.LoggerFactory
import ru.yandex.direct.ansiblejuggler.model.notifications.NotificationMethod
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository
import ru.yandex.direct.core.entity.user.repository.UserRepository
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.dbutil.sharding.ShardHelper
import ru.yandex.direct.env.ProductionOnly
import ru.yandex.direct.ess.config.userblock.UserBlockConfig
import ru.yandex.direct.ess.logicobjects.userblock.UserBlockObject
import ru.yandex.direct.juggler.JugglerStatus
import ru.yandex.direct.juggler.check.annotation.JugglerCheck
import ru.yandex.direct.juggler.check.annotation.OnChangeNotification
import ru.yandex.direct.juggler.check.model.CheckTag
import ru.yandex.direct.juggler.check.model.NotificationRecipient
import ru.yandex.direct.logicprocessor.common.BaseLogicProcessor
import ru.yandex.direct.logicprocessor.common.EssLogicProcessor
import ru.yandex.direct.logicprocessor.common.EssLogicProcessorContext


/**
 * Процессор останавливает кампании у заблокированных пользователей и блокирует всех пользователей с таким же clientId
 */
@JugglerCheck(
    ttl = JugglerCheck.Duration(minutes = 5),
    needCheck = ProductionOnly::class,
    notifications = [OnChangeNotification(
        recipient = [NotificationRecipient.LOGIN_MSPIRIT],
        method = [NotificationMethod.TELEGRAM],
        status = [JugglerStatus.OK, JugglerStatus.CRIT],
    )],
    tags = [CheckTag.DIRECT_PRIORITY_1],
)
@EssLogicProcessor(UserBlockConfig::class)
class UserBlockProcessor(
    private val shardHelper: ShardHelper,
    private val campaignRepository: CampaignRepository,
    private val userRepository: UserRepository,
    val essLogicProcessorContext: EssLogicProcessorContext)
    : BaseLogicProcessor<UserBlockObject>(essLogicProcessorContext) {

    companion object {
        const val CAMPAIGNS_STOP_CHUNK = 1000
        const val USERS_BLOCKED_CHUNK = 1000
        private val logger = LoggerFactory.getLogger(UserBlockProcessor::class.java)
    }

    override fun process(logicObjects: List<UserBlockObject>) {
        val userIds = logicObjects.map { it.uid }.distinct()
        logger.info("Got ${userIds.size} blocked users")
        val clientIdsByUserIds = shardHelper.getClientIdsByUids(userIds)
            .mapValues { ClientId.fromLong(it.value) }

        val clientIds = clientIdsByUserIds.values.distinct().map { it.asLong() }
        val campaignIdsToStop = campaignRepository.getCampaignIdsForStopping(shard, clientIds).values.flatten()

        if (campaignIdsToStop.isNotEmpty()) {
            logger.info("Going to stop ${campaignIdsToStop.size} campaigns")

            campaignIdsToStop.chunked(CAMPAIGNS_STOP_CHUNK)
                .forEach { campaignRepository.stopCampaignsForBlockedUsers(shard, it) }
        }

        val blockedUsers = userIds.chunked(USERS_BLOCKED_CHUNK)
            .map { userRepository.blockAllClientsForUser(shard, it) }
            .sum()
        logger.info("Blocked ${blockedUsers} users")
    }
}
