package ru.yandex.direct.ess.router.components;

import java.time.Duration;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Component;

import ru.yandex.direct.binlogbroker.logbroker_utils.models.BinlogEventWithOffset;
import ru.yandex.direct.binlogbroker.logbroker_utils.reader.LogbrokerBatchReader;
import ru.yandex.direct.binlogbroker.logbroker_utils.reader.LogbrokerCommitState;
import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.ess.router.models.WatchlogEvent;
import ru.yandex.direct.utils.Interrupts;

import static ru.yandex.direct.common.db.PpcPropertyNames.GRUT_READ_WATCHLOG_IN_ESS;

@Component
@Lazy
@ParametersAreNonnullByDefault
public class EventSourcingSystemRouterAppRunnable implements Runnable {

    private final Interrupts.InterruptibleFunction<List<BinlogEventWithOffset>, LogbrokerCommitState>
            routerBinlogConsumer;
    private final LogbrokerBatchReader<BinlogEventWithOffset> binlogLogbrokerReader;

    private final Interrupts.InterruptibleFunction<List<WatchlogEvent>, LogbrokerCommitState> routerWatchlogConsumer;
    private final LogbrokerBatchReader<WatchlogEvent> watchlogLogbrokerReader;

    private final PpcProperty<Boolean> readWatchlogProperty;

    @Autowired
    public EventSourcingSystemRouterAppRunnable(
            Interrupts.InterruptibleFunction<List<BinlogEventWithOffset>, LogbrokerCommitState> routerBinlogConsumer,
            LogbrokerBatchReader<BinlogEventWithOffset> binlogLogbrokerReader,
            Interrupts.InterruptibleFunction<List<WatchlogEvent>, LogbrokerCommitState> routerWatchlogConsumer,
            LogbrokerBatchReader<WatchlogEvent> watchlogLogbrokerReader,
            PpcPropertiesSupport ppcPropertiesSupport) {
        this.routerBinlogConsumer = routerBinlogConsumer;
        this.binlogLogbrokerReader = binlogLogbrokerReader;
        this.routerWatchlogConsumer = routerWatchlogConsumer;
        this.watchlogLogbrokerReader = watchlogLogbrokerReader;

        readWatchlogProperty = ppcPropertiesSupport.get(GRUT_READ_WATCHLOG_IN_ESS, Duration.ofMinutes(5));
    }

    @Override
    public void run() {
        while (!Thread.currentThread().isInterrupted()) {
            try {
                binlogLogbrokerReader.fetchEvents(routerBinlogConsumer);
                if (readWatchlogProperty.getOrDefault(false)) {
                    watchlogLogbrokerReader.fetchEvents(routerWatchlogConsumer);
                }
            } catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
        }
    }
}
