package ru.yandex.direct.ess.router.components;

import java.time.Duration;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Supplier;

import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.PreDestroy;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Component;

import ru.yandex.direct.binlogbroker.logbroker_utils.writer.LogbrokerWriter;
import ru.yandex.direct.ess.common.logbroker.LogbrokerClientFactoryFacade;
import ru.yandex.direct.ess.common.logbroker.LogbrokerProducerProperties;
import ru.yandex.direct.ess.common.logbroker.LogbrokerProducerPropertiesImpl;
import ru.yandex.direct.ess.common.utils.EssCommonUtils;
import ru.yandex.direct.ess.router.config.LogbrokerWriterAdditionalConfig;
import ru.yandex.direct.ess.router.config.LogbrokerWriterCommonConfig;
import ru.yandex.direct.ess.router.models.rule.ProcessedObject;
import ru.yandex.kikimr.persqueue.compression.CompressionCodec;
import ru.yandex.kikimr.persqueue.producer.AsyncProducer;

import static ru.yandex.direct.ess.common.configuration.EssCommonConfiguration.ESS_LOGBROKER_CLIENT_FACTORY_BEAN_NAME;


@Component
@ParametersAreNonnullByDefault
public class LogbrokerWriterFactory implements AutoCloseable {

    public static final String GRUT_WATCHLOG_SOURCE_ID_PREFIX = "grut_watchlog_";

    private final LogbrokerClientFactoryFacade logbrokerClientFactory;
    private final Map<LogbrokerWriterAdditionalConfig, LogbrokerWriter<ProcessedObject>> logbrokerConfigToWriter =
            new ConcurrentHashMap<>();
    private final LogbrokerWriterCommonConfig logbrokerWriterCommonConfig;
    private final EssCommonUtils essCommonUtils;

    private static final CompressionCodec COMPRESSION_CODEC = CompressionCodec.GZIP;

    @Autowired
    public LogbrokerWriterFactory(
            @Qualifier(ESS_LOGBROKER_CLIENT_FACTORY_BEAN_NAME) LogbrokerClientFactoryFacade logbrokerClientFactory,
            LogbrokerWriterCommonConfig logbrokerWriterCommonConfig, EssCommonUtils essCommonUtils) {
        this.logbrokerClientFactory = logbrokerClientFactory;
        this.logbrokerWriterCommonConfig = logbrokerWriterCommonConfig;
        this.essCommonUtils = essCommonUtils;
    }

    LogbrokerWriter<ProcessedObject> getLogbrokerWriter(
            LogbrokerWriterAdditionalConfig logbrokerWriterAdditionalConfig) {
        if (logbrokerConfigToWriter.containsKey(logbrokerWriterAdditionalConfig)) {
            return logbrokerConfigToWriter.get(logbrokerWriterAdditionalConfig);
        }
        String logbrokerTopic = essCommonUtils.getAbsoluteTopicPath(logbrokerWriterAdditionalConfig.getTopic());
        LogbrokerProducerProperties routerLogbrokerProducerProperties =
                LogbrokerProducerPropertiesImpl.newBuilder()
                        .setHost(logbrokerWriterCommonConfig.getLogbrokerHost())
                        .setTimeoutSec(logbrokerWriterCommonConfig.getLogbrokerTimeout())
                        .setGroup(logbrokerWriterAdditionalConfig.getGroup())
                        .setRetries(logbrokerWriterCommonConfig.getLogbrokerRetries())
                        .setWriteTopic(logbrokerTopic)
                        .setCompressionCodec(COMPRESSION_CODEC)
                        .build();

        String sourceId = getSourceId(logbrokerWriterAdditionalConfig);
        LogbrokerWriter<ProcessedObject> logbrokerWriter = createLogbrokerWriter(routerLogbrokerProducerProperties,
                sourceId, logbrokerWriterCommonConfig.getLogbrokerRetries());

        logbrokerConfigToWriter.put(logbrokerWriterAdditionalConfig, logbrokerWriter);
        return logbrokerWriter;
    }

    private String getSourceId(LogbrokerWriterAdditionalConfig logbrokerWriterAdditionalConfig) {
        if (logbrokerWriterAdditionalConfig.isForWatchlog()) {
            return GRUT_WATCHLOG_SOURCE_ID_PREFIX + logbrokerWriterAdditionalConfig.getGroup();
        } else {
            return String.valueOf(logbrokerWriterAdditionalConfig.getGroup());
        }
    }

    private LogbrokerWriter<ProcessedObject> createLogbrokerWriter(
            LogbrokerProducerProperties routerLogbrokerProducerProperties,
            String sourceId, int retries) {

        Supplier<CompletableFuture<AsyncProducer>> asyncProducerSupplier =
                logbrokerClientFactory.createProducerSupplier(routerLogbrokerProducerProperties, sourceId);

        return new LogicObjectLogbrokerWriter(asyncProducerSupplier,
                Duration.ofSeconds(routerLogbrokerProducerProperties.getTimeoutSec()), retries);
    }

    @Override
    @PreDestroy
    public void close() {
        closeAllProducers();
    }

    private void closeAllProducers() {
        logbrokerConfigToWriter.forEach((config, logbrokerWriter) -> logbrokerWriter.close());
    }
}
