package ru.yandex.direct.ess.router.components;

import java.time.ZoneOffset;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicLong;

import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

import ru.yandex.direct.binlogbroker.logbroker_utils.models.BinlogEventWithOffset;
import ru.yandex.direct.binlogbroker.logbroker_utils.reader.LogbrokerCommitState;
import ru.yandex.direct.binlogbroker.logbroker_utils.writer.LogBrokerWriterException;
import ru.yandex.direct.binlogbroker.logbroker_utils.writer.LogbrokerWriter;
import ru.yandex.direct.ess.common.models.LogicObjectListWithInfo;
import ru.yandex.direct.ess.router.config.LogbrokerWriterAdditionalConfig;
import ru.yandex.direct.ess.router.models.rule.ProcessedObject;
import ru.yandex.direct.ess.router.models.rule.RuleProcessingResult;
import ru.yandex.direct.utils.InterruptedRuntimeException;
import ru.yandex.direct.utils.Interrupts;

import static java.lang.Long.sum;
import static java.lang.Math.max;
import static java.lang.Math.min;
import static java.util.stream.Collectors.groupingBy;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;
import static one.util.streamex.MoreCollectors.mapping;
import static ru.yandex.direct.binlogbroker.logbroker_utils.reader.LogbrokerCommitState.DONT_COMMIT;
import static ru.yandex.direct.binlogbroker.logbroker_utils.reader.LogbrokerCommitState.NEED_COMMIT;
import static ru.yandex.direct.ess.router.utils.RouterUtilsKt.pingProcessedObjectsCreator;


/**
 * This class describes consumer which:
 * 1) filters all BinlogEventWithOffset according to specified rules
 * 2) writes them to LB
 */
@Component
public class RouterBinlogConsumer implements Interrupts.InterruptibleFunction<List<BinlogEventWithOffset>,
        LogbrokerCommitState> {

    private static final Logger logger = LoggerFactory.getLogger(RouterBinlogConsumer.class);

    private final RulesProcessingService rulesProcessingService;
    private final LogbrokerWriterFactory logbrokerWriterFactory;
    private final RouterBinlogMonitoring routerMonitoring;
    private final int iterationsBeforeCommit;
    private final int routerTimeoutSec;

    private int currentIteration = 0;

    @Autowired
    public RouterBinlogConsumer(
            RulesProcessingService rulesProcessingService,
            LogbrokerWriterFactory logbrokerWriterFactory,
            RouterBinlogMonitoring routerMonitoring,
            @Value("${ess.router.logbroker.iterations_before_commit}") int iterationsBeforeCommit,
            @Value("${ess.router.router_timeout}") int routerTimeoutSec
    ) {
        this.rulesProcessingService = rulesProcessingService;
        this.logbrokerWriterFactory = logbrokerWriterFactory;
        this.routerMonitoring = routerMonitoring;
        this.iterationsBeforeCommit = iterationsBeforeCommit;
        this.routerTimeoutSec = routerTimeoutSec;
    }


    /**
     * Все бинлоги передаются обработчику бинлогов {@link RulesProcessingService}, в результате которого возвращается
     * поток {@link RuleProcessingResult}, содержащие название топика, список партиций и данные для записи
     * Для кажой пары топик-партиция выбирается подходящий {@link LogbrokerWriter}, после чего происходит запись
     * <p>
     * Если для какой-либо пары топик-партиция не оказывается данных для записи, то в топик будет
     * записан ping-объект, у которого utcTimestamp = максимальный utcTimestamp для всей партиции, seqNo = минимальный
     * seqNo для партиции. У такого объекта будет проставлен флаг {@link LogicObjectListWithInfo#getIsPingObject()}
     */
    @Override
    public LogbrokerCommitState apply(List<BinlogEventWithOffset> binlogEvents) {
        // todo удалить после того, как станет понятно, почему позвникает ошибка https://st.yandex-team.ru/DIRECT-97393
        if (logger.isDebugEnabled()) {
            Map<Integer, List<Long>> partitionToSeqNo =
                    binlogEvents.stream()
                            .collect(groupingBy(BinlogEventWithOffset::getPartition,
                                    mapping(BinlogEventWithOffset::getSeqNo, toList())));
            StringBuilder logBuilder = new StringBuilder();
            partitionToSeqNo.forEach((partition, seqNos) -> {
                List<Long> duplicatedSeqNos = StreamEx.of(seqNos).distinct(2).toList();
                if (!duplicatedSeqNos.isEmpty()) {
                    logBuilder.append(" found duplicate seqNo in partition ").append(partition).append(": ")
                            .append(duplicatedSeqNos).append(";");
                } else {
                    logBuilder.append(" no duplicate seqNo in partition ").append(partition).append(";");
                }
            });
            logger.debug(logBuilder.toString());
        }

        handleAndWrite(binlogEvents);

        if (currentIteration >= iterationsBeforeCommit) {
            currentIteration = 0;
            return NEED_COMMIT;
        }
        currentIteration++;
        return DONT_COMMIT;
    }

    private void handleAndWrite(List<BinlogEventWithOffset> binlogEventWithOffsets) {
        Map<Integer, PartitionMetrics> partitionToMetrics = getBinlogEventsPartitionMetrics(binlogEventWithOffsets);
        AtomicLong startWritingTimestamp = new AtomicLong();
        CompletableFuture<Void> startFuture = new CompletableFuture<>();
        CompletableFuture<Void> writeCompletableFuture = CompletableFuture.allOf(
                rulesProcessingService.processEvents(binlogEventWithOffsets, startFuture)
                        .map(completableFuture -> completableFuture
                                .thenApply(
                                        handledEvents -> {
                                            routerMonitoring.addLogicObjectsMetrics(handledEvents);
                                            return handledEvents;
                                        })
                                .thenCompose(handledEvents -> {
                                    startWritingTimestamp.set(System.currentTimeMillis() / 1000);
                                    return writeResultToLogbroker(handledEvents, partitionToMetrics);
                                }))
                        .toArray(CompletableFuture[]::new)
        ).thenAcceptAsync(unused -> routerMonitoring.updateMetrics(partitionToMetrics, startWritingTimestamp.get()));

        startFuture.complete(null);
        try {
            writeCompletableFuture.get(routerTimeoutSec, TimeUnit.SECONDS);
        } catch (InterruptedException ex) {
            Thread.currentThread().interrupt();
            throw new InterruptedRuntimeException(ex);
        } catch (ExecutionException | TimeoutException ex) {
            throw new LogBrokerWriterException(ex);
        }
    }

    /**
     * Для каждой партиции, для который были бинлоги,  пробует получить обработанные данные для записи.
     * Если данных нет - пишет пинг-объект, инчае пишет данные
     *
     * @param ruleProcessingResult - топик для записи и обработанные данные для партиций
     * @param partitionsToMetrics  партиция -> максимальный таймстемп, минимальный seqNo, колчество строк
     * @return future, которая завершается, когда запишутся данные во все партиции
     */
    private CompletableFuture<Void> writeResultToLogbroker(RuleProcessingResult ruleProcessingResult, Map<Integer,
            PartitionMetrics> partitionsToMetrics) {

        return CompletableFuture.allOf(partitionsToMetrics.entrySet().stream()
                .map(partitionToMetrics -> {
                            int partition = partitionToMetrics.getKey();
                            PartitionMetrics partitionMetrics = partitionToMetrics.getValue();

                            // Если для партиции нет данных - будем передавать пинг-объект
                            List<ProcessedObject> processedObjects =
                                    ruleProcessingResult.getProcessedObjectsOrCompute(partition,
                                            pingProcessedObjectsCreator(partitionMetrics));

                            LogbrokerWriterAdditionalConfig logbrokerWriterAdditionalConfig =
                                    LogbrokerWriterAdditionalConfig
                                            .fromTopicAndPartition(ruleProcessingResult.getTopic(), partition);

                            LogbrokerWriter<ProcessedObject> logbrokerWriter =
                                    logbrokerWriterFactory.getLogbrokerWriter(logbrokerWriterAdditionalConfig);
                            return logbrokerWriter.write(processedObjects)
                                    .thenAccept(writingMessages -> routerMonitoring.addWritingMessages(logbrokerWriterAdditionalConfig, writingMessages));
                        }
                ).toArray(CompletableFuture[]::new));
    }

    /**
     * Группирует список событий по номеру партиции. Для каждой партиции считает агрегированные метрики: максимальное
     * время, суммарное количество строк, минимальный seqNo
     */
    Map<Integer, PartitionMetrics> getBinlogEventsPartitionMetrics(List<BinlogEventWithOffset> binlogEventsWithOffset) {
        return binlogEventsWithOffset.stream()
                .collect(toMap(
                        BinlogEventWithOffset::getPartition,
                        binlogEvent -> new PartitionMetrics(
                                binlogEvent.getEvent().getUtcTimestamp().toEpochSecond(ZoneOffset.UTC),
                                binlogEvent.getSeqNo(),
                                binlogEvent.getEvent().getRows().size()),
                        (metrics1, metrics2) -> {
                            metrics1.maxTimestamp = max(metrics1.maxTimestamp, metrics2.maxTimestamp);
                            metrics1.minSeqNo = min(metrics1.minSeqNo, metrics2.minSeqNo);
                            metrics1.rowsCount = sum(metrics1.rowsCount, metrics2.rowsCount);
                            return metrics1;
                        })
                );
    }
}
