package ru.yandex.direct.ess.router.components;

import java.time.Duration;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;

import javax.annotation.PostConstruct;

import com.google.common.util.concurrent.ThreadFactoryBuilder;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

import ru.yandex.direct.juggler.JugglerAsyncMultiSender;
import ru.yandex.direct.juggler.JugglerEvent;
import ru.yandex.direct.juggler.JugglerSubject;
import ru.yandex.monlib.metrics.registry.MetricRegistry;

@Component
public class RouterBinlogMonitoring extends RouterMonitoring implements AutoCloseable {

    private final Map<String, JugglerSubject> jugglerSubjects = new HashMap<>();

    private static final String JUGGLER_SUBJECT_PREFIX = "app.ess_router.lag.ppc_";
    private static final String JUGGLER_CRIT_MESSAGE_TEMPLATE = "Lag for shard %s: %ds > %ds";

    private final JugglerAsyncMultiSender jugglerClient;
    private final ScheduledExecutorService executor;
    private final Duration jugglerMonitoringInterval;
    private final long criticalBinlogDelay;

    public RouterBinlogMonitoring(JugglerAsyncMultiSender jugglerClient,
                                  MetricRegistry metricRegistry,
                                  @Value("${ess.router.juggler_monitoring_interval_sec}") int jugglerMonitoringInterval,
                                  @Value("${ess.router.critical_binlog_delay_sec}") int criticalBinlogDelay) {
        super(metricRegistry);
        this.jugglerClient = jugglerClient;
        this.executor = createDefaultExecutor();
        this.jugglerMonitoringInterval = Duration.ofSeconds(jugglerMonitoringInterval);
        this.criticalBinlogDelay = criticalBinlogDelay;
    }

    @PostConstruct
    public void afterPropertiesSet() {
        executor.scheduleAtFixedRate(
                this::setJugglerStatus,
                jugglerMonitoringInterval.getSeconds(),
                jugglerMonitoringInterval.getSeconds(),
                TimeUnit.SECONDS
        );
    }

    private long calculateBinlogDelay(long timestamp) {
        long now = System.currentTimeMillis() / 1000;
        return now - timestamp;
    }

    private void setJugglerStatus() {
        maxTimestamps.forEach(this::setJugglerStatusForShard);
    }

    private void setJugglerStatusForShard(String shard, long timestamp) {
        JugglerSubject jugglerSubject =
                jugglerSubjects.computeIfAbsent(shard, s -> new JugglerSubject(JUGGLER_SUBJECT_PREFIX + shard));
        long delay = calculateBinlogDelay(timestamp);
        JugglerEvent jugglerEvent;
        if (delay > criticalBinlogDelay) {
            jugglerEvent = jugglerSubject
                    .makeCrit(String.format(JUGGLER_CRIT_MESSAGE_TEMPLATE, shard, delay, criticalBinlogDelay));
        } else {
            jugglerEvent = jugglerSubject.makeOk();
        }
        jugglerClient.sendEvent(jugglerEvent, jugglerMonitoringInterval);
    }

    private ScheduledExecutorService createDefaultExecutor() {
        ThreadFactory threadFactory = new ThreadFactoryBuilder().setNameFormat("ess-router-monitoring-%02d").build();
        return Executors.newScheduledThreadPool(1, threadFactory);
    }

    @Override
    String getDelayMetricName() {
        return "binlog_delay";
    }

    @Override
    String getProcessedLogicObjectsMetricName() {
        return "processed_logic_objects";
    }

    @Override
    String getAdditionalLogicObjectsMetricName() {
        return "additional_logic_objects";
    }

    @Override
    String getSkippedAdditionalLogicObjectsMetricName() {
        return "skipped_additional_logic_objects";
    }

    @Override
    String getRowCountMetricName() {
        return "handled_binlog_rows_count";
    }

    @Override
    String getWritingMessagesMetricName() {
        return "writing_messages";
    }

    @Override
    String getWritingDurationMetricName() {
        return "writing_duration";
    }

    @Override
    public void close() {
        if (!executor.isShutdown()) {
            executor.shutdown();
        }
    }
}
