package ru.yandex.direct.ess.router.components

import java.time.Duration
import java.util.concurrent.CompletableFuture
import java.util.concurrent.ExecutionException
import java.util.concurrent.TimeUnit
import java.util.concurrent.TimeoutException
import java.util.concurrent.atomic.AtomicLong
import java.util.stream.Collectors
import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.beans.factory.annotation.Value
import org.springframework.stereotype.Component
import ru.yandex.direct.binlogbroker.logbroker_utils.reader.LogbrokerCommitState
import ru.yandex.direct.binlogbroker.logbroker_utils.writer.LogBrokerWriterException
import ru.yandex.direct.common.db.PpcPropertiesSupport
import ru.yandex.direct.common.db.PpcPropertyNames.ESS_WATCHLOG_PROCESSING_TIMEOUT_SEC
import ru.yandex.direct.common.db.PpcPropertyNames.GRUT_WATCHLOG_ITERATIONS_BEFORE_COMMIT
import ru.yandex.direct.common.db.PpcPropertyNames.GRUT_WRITE_WATCHLOG_IN_ESS
import ru.yandex.direct.ess.router.config.LogbrokerWriterAdditionalConfig
import ru.yandex.direct.ess.router.models.WatchlogEvent
import ru.yandex.direct.ess.router.models.getUtcTimestamp
import ru.yandex.direct.ess.router.models.rule.RuleProcessingResult
import ru.yandex.direct.ess.router.utils.pingProcessedObjectsCreator
import ru.yandex.direct.utils.InterruptedRuntimeException
import ru.yandex.direct.utils.Interrupts
import kotlin.math.max
import kotlin.math.min
import kotlin.streams.toList

@Component
class RouterWatchlogConsumer @Autowired constructor(
    ppcPropertiesSupport: PpcPropertiesSupport,
    private val rulesProcessingService: RulesProcessingService,
    private val logbrokerWriterFactory: LogbrokerWriterFactory,
    private val routerMonitoring: RouterWatchlogMonitoring,
    @Value("\${ess.router.logbroker.iterations_before_commit}") private val defaultIterationsBeforeCommit: Int,
    @Value("\${ess.router.router_timeout}") private val routerTimeoutSec: Int,
) : Interrupts.InterruptibleFunction<MutableList<WatchlogEvent>, LogbrokerCommitState> { // MutableList для совместимости с java.util.List

    private val watchlogProcessingTimeoutSecProp =
        ppcPropertiesSupport.get(ESS_WATCHLOG_PROCESSING_TIMEOUT_SEC, Duration.ofMinutes(5))
    private val writeWatchlogEventsProp = ppcPropertiesSupport.get(GRUT_WRITE_WATCHLOG_IN_ESS, Duration.ofMinutes(5))
    private val watchlogIterationsBeforeCommit =
        ppcPropertiesSupport.get(GRUT_WATCHLOG_ITERATIONS_BEFORE_COMMIT, Duration.ofMinutes(5))

    companion object {
        private val logger = LoggerFactory.getLogger(RouterWatchlogConsumer::class.java)
    }

    private var currentIteration = 0

    override fun apply(watchlogEvents: MutableList<WatchlogEvent>): LogbrokerCommitState {
        logger.info("Got ${watchlogEvents.size} watchlog events")
        if (writeWatchlogEventsProp.getOrDefault(false)) {
            handleAndWrite(watchlogEvents)
        }
        logger.info("Finish processing watchlog, iteration $currentIteration")

        val iterationsBeforeCommit = watchlogIterationsBeforeCommit.getOrDefault(defaultIterationsBeforeCommit)
        if (currentIteration >= iterationsBeforeCommit) {
            currentIteration = 0
            return LogbrokerCommitState.NEED_COMMIT
        }
        currentIteration++
        // всегда возвращаем need_commit так как сейчас топик ватчлогов отключен, но там остались данные,
        // которых не хватает на нужное кол-во итераций, поэтому коммитим каждый раз
        return LogbrokerCommitState.NEED_COMMIT
    }

    private fun handleAndWrite(watchlogEvents: List<WatchlogEvent>) {
        val partitionToMetrics: Map<Int, PartitionMetrics> = getWatchlogEventsPartitionMetrics(watchlogEvents)
        val startFuture = CompletableFuture<Void>()
        val startWritingTimestamp = AtomicLong()
        val processRuleFutures = rulesProcessingService.processWatchlogEvents(watchlogEvents, startFuture)
            .map { future ->
                future
                    .thenApply { rulesProcessingResult ->
                        routerMonitoring.addLogicObjectsMetrics(rulesProcessingResult)
                        rulesProcessingResult
                    }.thenCompose { rulesProcessingResult ->
                        startWritingTimestamp.set(System.currentTimeMillis() / 1000)
                        writeResultToLogbroker(partitionToMetrics, rulesProcessingResult)
                    }
            }.toList()
        val resultFuture = CompletableFuture.allOf(*processRuleFutures.toTypedArray()).thenRunAsync {
            routerMonitoring.updateMetrics(partitionToMetrics, startWritingTimestamp.get())
        }

        startFuture.complete(null)
        logger.info("Start processing watchlog")

        try {
            val timeout = watchlogProcessingTimeoutSecProp.getOrDefault(routerTimeoutSec.toLong())
            resultFuture[timeout, TimeUnit.SECONDS]
        } catch (ex: InterruptedException) {
            Thread.currentThread().interrupt()
            throw InterruptedRuntimeException(ex)
        } catch (ex: ExecutionException) {
            throw LogBrokerWriterException(ex)
        } catch (ex: TimeoutException) {
            throw LogBrokerWriterException(ex)
        }
    }

    private fun getWatchlogEventsPartitionMetrics(watchlogEvents: List<WatchlogEvent>): Map<Int, PartitionMetrics> {
        return watchlogEvents.stream()
            .collect(Collectors.toMap(
                { event -> event.partition },
                { event -> PartitionMetrics(event.getUtcTimestamp(), event.seqNo, 1) }
            ) { metrics1, metrics2 ->
                metrics1.maxTimestamp = max(metrics1.maxTimestamp, metrics2.maxTimestamp)
                metrics1.minSeqNo = min(metrics1.minSeqNo, metrics2.minSeqNo)
                metrics1.rowsCount = metrics1.rowsCount + metrics2.rowsCount
                metrics1
            })
    }

    private fun writeResultToLogbroker(
        partitionToMetrics: Map<Int, PartitionMetrics>,
        ruleProcessingResult: RuleProcessingResult,
    ): CompletableFuture<Void> {
        val writeToPartitionFutures = partitionToMetrics.map { (partition, metrics) ->
            val objectToWrite =
                ruleProcessingResult.getProcessedObjectsOrCompute(partition, pingProcessedObjectsCreator(metrics))
            logger.info(
                "Trying to write ${objectToWrite.size} objects to partition $partition " +
                    "for topic ${ruleProcessingResult.topic}"
            )
            val logbrokerWriterAdditionalConfig = LogbrokerWriterAdditionalConfig
                .fromTopicAndPartition(ruleProcessingResult.topic, partition, true)
            val logbrokerWriter =
                logbrokerWriterFactory.getLogbrokerWriter(logbrokerWriterAdditionalConfig)
            logbrokerWriter.write(objectToWrite)
                .thenAccept { writingMessages -> routerMonitoring.addWritingMessages(logbrokerWriterAdditionalConfig, writingMessages) }
        }

        return CompletableFuture.allOf(*writeToPartitionFutures.toTypedArray())
    }
}
