package ru.yandex.direct.ess.router.components;

import java.time.Duration;
import java.time.LocalDateTime;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Stream;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.binlogbroker.logbroker_utils.models.BinlogEventWithOffset;
import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.common.db.PpcPropertyNames;
import ru.yandex.direct.ess.router.models.BinlogEventsFilter;
import ru.yandex.direct.ess.router.models.WatchlogEvent;
import ru.yandex.direct.ess.router.models.rule.AbstractRule;
import ru.yandex.direct.ess.router.models.rule.RuleProcessingResult;
import ru.yandex.direct.utils.Condition;

import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Component
@ParametersAreNonnullByDefault
public class RulesProcessingService {

    private static final Logger logger = LoggerFactory.getLogger(RulesProcessingService.class);

    private static final Duration PPC_PROPERTY_EXPIRE_SEC = Duration.ofSeconds(30);

    private final List<? extends AbstractRule> rules;
    private final BinlogEventsFilter binlogEventsFilter;
    private final PpcProperty<Set<String>> disabledRulesProperty;

    @Autowired
    public RulesProcessingService(List<? extends AbstractRule> rules,
                                  BinlogEventsFilter binlogEventsFilter,
                                  PpcPropertiesSupport ppcPropertiesSupport,
                                  List<? extends Condition> conditions) {
        this(rules, binlogEventsFilter, ppcPropertiesSupport, getRuleToConditionMap(rules, conditions));
    }

    private static Map<? extends Class<? extends AbstractRule>, ? extends Condition> getRuleToConditionMap(
            List<? extends AbstractRule> rules,
            List<? extends Condition> conditions) {
        var classToCondition = listToMap(conditions, Condition::getClass);
        return StreamEx.of(rules)
                .mapToEntry(r -> r.getClass(), AbstractRule::getRunConditionClass)
                .mapValues(classToCondition::get)
                .peek(entry -> {
                    if (entry.getValue() == null) {
                        throw new NullPointerException(String.format("condition for rule %s is not a component",
                                entry.getKey()));
                    }
                })
                .toMap();
    }

    public RulesProcessingService(List<? extends AbstractRule> rules,
                                  BinlogEventsFilter binlogEventsFilter,
                                  PpcPropertiesSupport ppcPropertiesSupport,
                                  Map<? extends Class<? extends AbstractRule>, ? extends Condition> ruleToConditionMap) {
        this.rules = filterList(rules, r -> ruleToConditionMap.get(r.getClass()).evaluate());
        this.binlogEventsFilter = binlogEventsFilter;
        this.disabledRulesProperty = ppcPropertiesSupport.get(PpcPropertyNames.DISABLED_ESS_RULES,
                PPC_PROPERTY_EXPIRE_SEC);
    }

    public Stream<CompletableFuture<RuleProcessingResult>> processEvents(List<BinlogEventWithOffset> binlogEvents,
                                                                         CompletableFuture<Void> startFuture) {
        List<BinlogEventWithOffset> filteredBinlogEvents = binlogEventsFilter.filter(binlogEvents);

        if (filteredBinlogEvents.isEmpty()) {
            return Stream.empty();
        }

        logger.info("processing {} events", filteredBinlogEvents.size());
        if (logger.isDebugEnabled()) {
            var eventsTimeRange = getEventsTimeRange(filteredBinlogEvents);
            logger.debug("processed events occurred from {} to {})",
                    eventsTimeRange.from.plusHours(3), eventsTimeRange.to.plusHours(3));
        }

        return getActiveRules()
                .map(rule -> startFuture.thenApplyAsync(unused -> rule.processEvents(filteredBinlogEvents)));
    }

    public Stream<CompletableFuture<RuleProcessingResult>> processWatchlogEvents(List<WatchlogEvent> watchlogEvents,
                                                                                 CompletableFuture<Void> startFuture) {
        if (watchlogEvents.isEmpty()) {
            return Stream.empty();
        }

        logger.info("processing {} watchlog events", watchlogEvents.size());

        return getActiveRules()
                .map(rule -> startFuture.thenApplyAsync(unused -> rule.processWatchlogEvents(watchlogEvents)));
    }

    private Stream<? extends AbstractRule> getActiveRules() {
        Set<String> disabledRules = disabledRulesProperty.getOrDefault(Collections.emptySet());

        return rules.stream()
                .filter(rule -> !disabledRules.contains(rule.getLogicProcessName()));
    }

    private TimeRange getEventsTimeRange(List<BinlogEventWithOffset> binlogEvents) {
        var timestamps = mapList(binlogEvents, event -> event.getEvent().getUtcTimestamp());
        var minTime = timestamps.stream().min(LocalDateTime::compareTo).get();
        var maxTime = timestamps.stream().max(LocalDateTime::compareTo).get();
        return new TimeRange(minTime, maxTime);
    }

    private class TimeRange {
        private final LocalDateTime from;
        private final LocalDateTime to;

        public TimeRange(LocalDateTime from, LocalDateTime to) {
            this.from = from;
            this.to = to;
        }

        public LocalDateTime getFrom() {
            return from;
        }

        public LocalDateTime getTo() {
            return to;
        }

    }
}
