package ru.yandex.direct.ess.router.configuration;

import java.time.Duration;
import java.util.Arrays;
import java.util.List;
import java.util.concurrent.ForkJoinPool;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.ComponentScan;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.FilterType;
import org.springframework.context.annotation.Import;
import org.springframework.context.annotation.Primary;

import ru.yandex.direct.binlogbroker.logbroker_utils.models.BinlogEventWithOffset;
import ru.yandex.direct.binlogbroker.logbroker_utils.reader.BinlogLogbrokerReader;
import ru.yandex.direct.binlogbroker.logbroker_utils.reader.LogbrokerBatchReader;
import ru.yandex.direct.common.configuration.MetricsConfiguration;
import ru.yandex.direct.common.configuration.MonitoringHttpServerConfiguration;
import ru.yandex.direct.common.configuration.TracingConfiguration;
import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.config.EssentialConfiguration;
import ru.yandex.direct.dbutil.configuration.DbUtilConfiguration;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.env.EnvironmentType;
import ru.yandex.direct.ess.common.configuration.EssCommonConfiguration;
import ru.yandex.direct.ess.common.logbroker.LogbrokerClientFactoryFacade;
import ru.yandex.direct.ess.router.components.EssAppDestroyer;
import ru.yandex.direct.ess.router.components.WatchlogLogbrokerReader;
import ru.yandex.direct.ess.router.config.LogbrokerWriterCommonConfig;
import ru.yandex.direct.ess.router.config.RouterLogbrokerConsumerProperties;
import ru.yandex.direct.ess.router.configuration.commandline.AppParams;
import ru.yandex.direct.ess.router.configuration.commandline.LogbrokerParams;
import ru.yandex.direct.ess.router.models.BinlogEventsFilter;
import ru.yandex.direct.ess.router.models.EssWorkMode;
import ru.yandex.direct.ess.router.models.WatchlogEvent;
import ru.yandex.direct.juggler.check.configuration.JugglerCheckConfiguration;
import ru.yandex.direct.logging.GlobalCustomMdc;
import ru.yandex.direct.tracing.TraceMdcAdapter;
import ru.yandex.direct.ytwrapper.chooser.WorkModeChooser;
import ru.yandex.direct.ytwrapper.client.YtClusterConfig;
import ru.yandex.direct.ytwrapper.client.YtClusterConfigProvider;
import ru.yandex.direct.ytwrapper.client.YtClusterTypesafeConfigProvider;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.kikimr.persqueue.consumer.SyncConsumer;
import ru.yandex.monlib.metrics.registry.MetricRegistry;

import static java.lang.String.format;
import static ru.yandex.direct.ess.common.configuration.EssCommonConfiguration.ESS_LOGBROKER_CLIENT_FACTORY_BEAN_NAME;
import static ru.yandex.direct.ess.common.utils.EssCommonUtils.getAbsoluteConsumerPath;
import static ru.yandex.direct.solomon.SolomonUtils.SOLOMON_REGISTRY;

@Configuration
@Import({
        EssentialConfiguration.class,
        AppJcommanderConfiguration.class,
        DbUtilConfiguration.class,
        TracingConfiguration.class,
        EssCommonConfiguration.class,
        MonitoringHttpServerConfiguration.class,
        MetricsConfiguration.class,
        JugglerCheckConfiguration.class,
})
@ComponentScan(
        basePackages = {
                "ru.yandex.direct.ess.router.components",
                "ru.yandex.direct.ess.router.rules",
                "ru.yandex.direct.ess.router.models"
        },
        excludeFilters = @ComponentScan.Filter(value = Configuration.class, type = FilterType.ANNOTATION)
)
public class RouterApplicationConfiguration {

    public static final String CONSUMER_NAME_BEAN = "consumerName";
    public static final String LOCKING_YT_CONFIG = "essLockingYtConfig";
    public static final String ESS_SHARDS = "essShards";
    public static final String ROUTER_CHUNK_ID = "essRouterChunkId";
    public static final String BINLOG_LOGBROKER_READER_CONFIG_BEAN = "binlogLogbrokerReaderConfig";
    public static final String WATCHLOG_LOGBROKER_READER_CONFIG_BEAN = "watchlogLogbrokerReaderConfig";

    public static final String METHOD_TEMPL = "ess-router-%d";

    @Bean(CONSUMER_NAME_BEAN)
    public String consumerName(@Value("${ess.logbroker.consumer_name}") String consumerName) {
        return consumerName;
    }

    /**
     * Конфигурация YT-клиента для хранения небольших атрибутов и локов.
     * На кластере locke нет grpc-прокси, он поддерживает только HTTP-интерфейс.
     */
    @Bean(name = LOCKING_YT_CONFIG)
    public YtClusterConfig lockingYtConfig(
            DirectConfig directConfig,
            @Value("${ess.router.yt_lock.cluster_name}") String clusterName) {
        YtClusterConfigProvider provider =
                new YtClusterTypesafeConfigProvider(directConfig.getConfig().getConfig("yt"));
        YtCluster cluster = YtCluster.clusterFromNameToUpperCase(clusterName);
        return provider.get(cluster);
    }

    @Bean
    public EssWorkMode workMode(WorkModeChooser<EssWorkMode> workModeChooser) {
        EssWorkMode workMode = workModeChooser.getOrRequestWorkMode();
        GlobalCustomMdc.put(TraceMdcAdapter.METHOD_KEY, format(METHOD_TEMPL, workMode.getChunk()));
        return workMode;
    }

    @Bean(name = ESS_SHARDS)
    public List<Integer> essShards(
            EssWorkMode workMode,
            LogbrokerParams logbrokerParams,
            ShardHelper shardHelper
    ) {
        var totalShards = shardHelper.dbShards();
        // если в параметрах явно задан список шардов - берём его
        if (logbrokerParams.partitions != null && !logbrokerParams.partitions.isEmpty()) {
            return logbrokerParams.partitions.stream()
                    .filter(totalShards::contains)
                    .collect(Collectors.toList());
        } else {
            return workMode.getShards(totalShards);
        }
    }

    @Bean
    public WorkModeChooser<EssWorkMode> workModeChooser(
            @Qualifier(LOCKING_YT_CONFIG) YtClusterConfig lockingYtConfig,
            @Value("${ess.router.yt_lock.work_mode_locks_subpath}") String workModeLocksSubpath,
            EnvironmentType environmentType,
            EssAppDestroyer appDestroyer
    ) {
        return new WorkModeChooser<>(
                lockingYtConfig,
                workModeLocksSubpath,
                environmentType,
                appDestroyer,
                Arrays.asList(EssWorkMode.values())
        );
    }

    @Bean(BINLOG_LOGBROKER_READER_CONFIG_BEAN)
    public RouterLogbrokerConsumerProperties binlogLogbrokerReaderConfig(
            @Value("${ess.logbroker.logbroker_prefix}") String logbrokerPrefix,
            @Value("${ess.logbroker.host}") String logbrokerHost,
            @Qualifier(CONSUMER_NAME_BEAN) String consumerName,
            @Value("${ess.logbroker.read_timeout}") int logbrokerReadTimeoutSec,
            @Value("${ess.logbroker.init_timeout}") int logbrokerInitTimeoutSec,
            @Value("${ess.logbroker.retries}") int retries,
            @Value("${ess.router.logbroker.binlog_topic}") String logbrokerTopic,
            @Value("${ess.router.logbroker.max_count}") int maxCount,
            @Value("${ess.router.logbroker.read_buffer_size}") int readBufferSize,
            LogbrokerParams logbrokerParams,
            @Qualifier(ESS_SHARDS) List<Integer> shards) {
        return getRouterLogbrokerConsumerProperties(logbrokerPrefix, logbrokerHost, consumerName,
                logbrokerReadTimeoutSec, logbrokerInitTimeoutSec, retries, logbrokerTopic, maxCount, readBufferSize,
                logbrokerParams, shards);
    }

    @Bean(WATCHLOG_LOGBROKER_READER_CONFIG_BEAN)
    public RouterLogbrokerConsumerProperties watchlogLogbrokerReaderConfig(
            @Value("${ess.logbroker.logbroker_prefix}") String logbrokerPrefix,
            @Value("${ess.logbroker.host}") String logbrokerHost,
            @Qualifier(CONSUMER_NAME_BEAN) String consumerName,
            @Value("${ess.logbroker.read_timeout}") int logbrokerReadTimeoutSec,
            @Value("${ess.logbroker.init_timeout}") int logbrokerInitTimeoutSec,
            @Value("${ess.logbroker.retries}") int retries,
            @Value("${ess.router.logbroker.watchlog_topic}") String logbrokerTopic,
            @Value("${ess.router.logbroker.max_count}") int maxCount,
            @Value("${ess.router.logbroker.read_buffer_size}") int readBufferSize,
            LogbrokerParams logbrokerParams,
            @Qualifier(ESS_SHARDS) List<Integer> shards) {
        return getRouterLogbrokerConsumerProperties(logbrokerPrefix, logbrokerHost, consumerName,
                logbrokerReadTimeoutSec, logbrokerInitTimeoutSec, retries, logbrokerTopic, maxCount, readBufferSize,
                logbrokerParams, shards);
    }

    private RouterLogbrokerConsumerProperties getRouterLogbrokerConsumerProperties(
            String logbrokerPrefix,
            String logbrokerHost,
            String consumerName,
            int logbrokerReadTimeoutSec,
            int logbrokerInitTimeoutSec,
            int retries,
            String logbrokerTopic,
            int maxCount,
            int readBufferSize,
            LogbrokerParams logbrokerParams,
            List<Integer> shards
    ) {
        if (shards.isEmpty()) {
            throw new IllegalStateException(
                    "No logbroker partitions to read. Maybe you need to specify --sources argument");
        }

        String absoluteConsumerPath = getAbsoluteConsumerPath(logbrokerPrefix, consumerName);
        return new RouterLogbrokerConsumerProperties.Builder()
                .setConsumerName(absoluteConsumerPath)
                .setGroups(shards)
                .setHost(logbrokerHost)
                .setTopic(logbrokerTopic)
                .setReadDataTimeoutSec(logbrokerReadTimeoutSec)
                .setInitTimeoutSec(logbrokerInitTimeoutSec)
                .setRetries(retries)
                .setReadOnlyNewData(logbrokerParams.readNewData)
                .readDataOnlyAfterTimestampMs(logbrokerParams.getReadOnlyDataCreatedAfterTimestampMs())
                .setMaxCount(maxCount)
                .setReadBufferSize(readBufferSize)
                .build();
    }

    @Bean
    public LogbrokerWriterCommonConfig getLogbrokerWriterCommonConfig(
            @Value("${ess.logbroker.host}") String logbrokerHost,
            @Value("${ess.logbroker.writer_timeout}") long logbrokerwriterTimeoutSec,
            @Value("${ess.logbroker.retries}") int retries) {
        return new LogbrokerWriterCommonConfig(logbrokerHost, logbrokerwriterTimeoutSec, retries);
    }

    @Bean(name = ROUTER_CHUNK_ID)
    public String routerChunkId(EssWorkMode workMode) {
        return String.valueOf(workMode.getChunk());
    }

    @Primary
    @Bean
    public MetricRegistry metricRegistry(@Qualifier(ROUTER_CHUNK_ID) String routerChunkId) {
        return SOLOMON_REGISTRY.subRegistry("router_chunk_id", routerChunkId);
    }

    @Bean
    public LogbrokerBatchReader<BinlogEventWithOffset> binlogEventReader(
            @Qualifier(BINLOG_LOGBROKER_READER_CONFIG_BEAN) RouterLogbrokerConsumerProperties routerLogbrokerConsumerProperties,
            LogbrokerParams logbrokerParams,
            MetricRegistry metricRegistry,
            @Qualifier(ESS_LOGBROKER_CLIENT_FACTORY_BEAN_NAME) LogbrokerClientFactoryFacade logbrokerClientFactory,
            @Value("${ess.router.logbroker.rows_threshold}") long rowsThreshold,
            @Value("${ess.router.logbroker.time_threshold_sec}") int timeThresholdSec,
            @Value("${ess.router.logbroker.bytes_threshold}") int bytesThreshold) {
        Supplier<SyncConsumer> syncConsumerSupplier =
                logbrokerClientFactory.createConsumerSupplier(routerLogbrokerConsumerProperties);

        ForkJoinPool deserializeForkJoinPool = ForkJoinPool.commonPool();
        Duration timeThreshold = Duration.ofSeconds(timeThresholdSec);
        // Для ess-router сейчас нельзя использовать механизм retry т.к. retry выполняется только в случае неудачного
        // коммита, а в случае router'а он выполняется не на кажой итерации
        // https://st.yandex-team.ru/DIRECT-100449 - если будет решение, то надо будет перевести на retry
        return new BinlogLogbrokerReader.Builder()
                .withLogbrokerConsumerSupplier(syncConsumerSupplier)
                .withLogbrokerNoCommit(logbrokerParams.logbrokerNoCommit)
                .withNeedReadingOptimization(true)
                .withMetricRegistry(metricRegistry)
                .withForkJoinPool(deserializeForkJoinPool)
                .withRowsThreshold(rowsThreshold)
                .withTimeThreshold(timeThreshold)
                .withBytesThreshold(bytesThreshold)
                .build();
    }

    @Bean
    public LogbrokerBatchReader<WatchlogEvent> watchlogEventReader(
            @Qualifier(WATCHLOG_LOGBROKER_READER_CONFIG_BEAN) RouterLogbrokerConsumerProperties routerLogbrokerConsumerProperties,
            LogbrokerParams logbrokerParams,
            @Qualifier(ESS_LOGBROKER_CLIENT_FACTORY_BEAN_NAME) LogbrokerClientFactoryFacade logbrokerClientFactory) {
        Supplier<SyncConsumer> syncConsumerSupplier =
                logbrokerClientFactory.createConsumerSupplier(routerLogbrokerConsumerProperties);

        return new WatchlogLogbrokerReader(syncConsumerSupplier, logbrokerParams.logbrokerNoCommit);
    }

    @Bean
    public BinlogEventsFilter binlogEventsFilter(
            EnvironmentType environmentType,
            AppParams appParams
    ) {
        if (!environmentType.isBeta()) {
            return BinlogEventsFilter.doNothing();
        }
        return new BinlogEventsFilter(
                event -> appParams.essTag.equals(event.getEvent().getEssTag()));
    }
}
