package ru.yandex.direct.ess.router.configuration.commandline;

import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import com.beust.jcommander.IStringConverter;
import com.beust.jcommander.Parameter;
import com.beust.jcommander.ParameterException;

import static ru.yandex.direct.utils.DateTimeUtils.moscowDateTimeToInstant;

@ParametersAreNonnullByDefault
public class LogbrokerParams {

    @Parameter(
            names = {"--partitions"},
            description = "Comma separated list of partitions, e.g.: 1,2,3,4"
    )
    public List<Integer> partitions;


    @Parameter(
            names = {"--read-new-data", "--read-only-new-data"},
            description = "Read data from logbroker created only after start time."
    )
    public boolean readNewData = false;

    @Parameter(
            names = {"--read-only-data-created-after", "--read-data-after"},
            description = "Read data from logbroker created only after this time, e.g. '2020-07-16 15:00:00'",
            converter = DateTimeToTimestampMsConverter.class
    )
    private Long readOnlyDataCreatedAfterTimestampMs = null;

    @Parameter(
            names = {"--logbroker-no-commit"},
            description = "Mode for debugging. Do not commit any fetched message from logbroker."
    )
    public boolean logbrokerNoCommit = false;


    public void validate() {

        if (readNewData) {
            if (readOnlyDataCreatedAfterTimestampMs != null) {
                throw new ParameterException("Parameters 'read-new-data-only' and 'read-data-only-created-after' " +
                        "are mutually exclusive and can't be set both");
            }

            readOnlyDataCreatedAfterTimestampMs = System.currentTimeMillis();
        }
    }

    public Long getReadOnlyDataCreatedAfterTimestampMs() {
        return readOnlyDataCreatedAfterTimestampMs;
    }

    private static class DateTimeToTimestampMsConverter implements IStringConverter<Long> {
        @Override
        public Long convert(String value) {
            if (value == null || value.isEmpty()) {
                return null;
            }

            var datetime = LocalDateTime.parse(value, DateTimeFormatter.ISO_LOCAL_DATE_TIME);
            return moscowDateTimeToInstant(datetime).toEpochMilli();
        }
    }
}
