package ru.yandex.direct.ess.router.models.rule;

import java.lang.reflect.InvocationTargetException;
import java.time.ZoneOffset;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.binlog.model.BinlogEvent;
import ru.yandex.direct.binlogbroker.logbroker_utils.models.BinlogEventWithOffset;
import ru.yandex.direct.ess.common.converter.LogicObjectWithSystemInfoConverter;
import ru.yandex.direct.ess.common.models.BaseEssConfig;
import ru.yandex.direct.ess.common.models.BaseLogicObject;
import ru.yandex.direct.ess.common.models.LogicObjectListWithInfo;
import ru.yandex.direct.ess.router.models.TEssEvent;
import ru.yandex.direct.ess.router.models.WatchlogEvent;
import ru.yandex.direct.utils.Condition;

import static java.util.Collections.emptyList;
import static java.util.stream.Collectors.groupingBy;
import static java.util.stream.Collectors.mapping;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;
import static ru.yandex.direct.ess.router.models.WatchlogEventKt.getUtcTimestamp;

public abstract class AbstractRule<T extends BaseLogicObject> {
    private static final Logger logger = LoggerFactory.getLogger(AbstractRule.class);

    private final String topic;
    private final String logicProcessName;
    private final Class<? extends Condition> runConditionClass;
    private final LogicObjectWithSystemInfoConverter<T> logicObjectWithSystemInfoConverter;
    private final AdditionalObjectsHandler<T> additionalObjectsHandler;
    protected final boolean processReshardingEvents;

    protected AbstractRule() {
        EssRule essConfigClass = getClass().getAnnotation(EssRule.class);
        BaseEssConfig essConfig;
        try {
            essConfig = essConfigClass.value().getDeclaredConstructor().newInstance();
        } catch (
                NoSuchMethodException
                        | IllegalAccessException
                        | InvocationTargetException
                        | InstantiationException
                        | IllegalStateException
                        ex
        ) {
            throw new IllegalStateException("Can't create new instance of ess config class", ex);
        }
        this.runConditionClass = essConfigClass.runCondition();

        this.topic = essConfig.getTopic();
        this.logicProcessName = essConfig.getLogicProcessName();
        this.additionalObjectsHandler = new AdditionalObjectsHandler<>(essConfig.getLogicProcessName(),
                essConfig.getLogicObject());
        this.logicObjectWithSystemInfoConverter = new LogicObjectWithSystemInfoConverter<>(essConfig.getLogicObject());
        this.processReshardingEvents = essConfig.processReshardingEvents();
    }

    /**
     * Метод, проверяющий, подходит ли бинлог для дальшейшей обработки, и конвертирующий его в список
     * пользовательских объектов
     * Пользовательские объекты будут в том же порядке, что и rows в BinlogEvent
     */
    public abstract List<T> mapBinlogEvent(BinlogEvent binlogEvent);

    /**
     * Метод, аналогичный {@link #mapBinlogEvent(BinlogEvent)}, для грутового вотчлога.
     * По умолчанию ничего не делает, так как большинство процессоров еще не готовы к объектам в груте
     */
    public List<T> mapWatchlogEvent(TEssEvent watchlogEvent) {
        return emptyList();
    }

    /**
     * Каждое событие обраатывается методом {@link #mapBinlogEvent(BinlogEvent)}, которой превращает бинлог в список
     * пользовательских объектов
     * Все полученные объекты сериализуются в json и группируются по номеру партиции
     *
     * @return топик для записи и сгруппированные по номеру партиции данные для записи.
     * Если никакие данные не отобрались, то поле {@link RuleProcessingResult#getGroupedLogicObjectsByPartitions()}
     * содержит пустую мапу
     */
    public RuleProcessingResult processEvents(
            List<BinlogEventWithOffset> unprocessedBinlogEvents) {
        var partitionToProcessedObjectWithStatList = unprocessedBinlogEvents.stream()
                .filter(this::isEventSuitable)
                .map(this::binlogEventToProcessedObject)
                .collect(toList());

        return getRuleProcessingResult(partitionToProcessedObjectWithStatList);
    }

    /**
     * Метод, аналогичный {@link #processEvents}, для грутового вотчлога.
     */
    public RuleProcessingResult processWatchlogEvents(List<WatchlogEvent> unprocessedWatchlogEvents) {
        var partitionToProcessedObjectWithStatList = unprocessedWatchlogEvents.stream()
                .filter(this::isEventSuitable)
                .map(this::watchlogEventToProcessedObject)
                .collect(toList());

        return getRuleProcessingResult(partitionToProcessedObjectWithStatList);
    }

    private RuleProcessingResult getRuleProcessingResult(List<PartitionToProcessedObjectWithStat> processedObjects) {
        Map<Integer, List<ProcessedObject>> groupedLogicObjectsByPartitions = processedObjects
                .stream()
                .filter(partitionToProcessedObjectWithStat ->
                        Objects.nonNull(partitionToProcessedObjectWithStat.processedObjects))
                .collect(
                        groupingBy(PartitionToProcessedObjectWithStat::getPartition,
                                mapping(PartitionToProcessedObjectWithStat::getProcessedObjects, toList())));

        Map<Integer, RuleProcessingStat> groupedStatByPartition = processedObjects
                .stream()
                .collect(toMap(PartitionToProcessedObjectWithStat::getPartition, this::getRuleProcessingStat,
                        this::mergeStats));

        return new RuleProcessingResult(topic, groupedLogicObjectsByPartitions, groupedStatByPartition);
    }

    /**
     * Фильтрует подходящие события для обработки
     * <p>
     * События, возникающие в процессе решардинга, фильтруются в зависимости от конфигурации
     * {@link BaseEssConfig#processReshardingEvents()}
     *
     * @return true, если события должно обрабатываться правилом
     */
    protected boolean isEventSuitable(BinlogEventWithOffset binlogEventWithOffset) {
        return processReshardingEvents || !binlogEventWithOffset.getEvent().isResharding();
    }

    /**
     * Метод, аналогичный {@link #isEventSuitable}, для грутовых событий.
     */
    protected boolean isEventSuitable(WatchlogEvent watchlogEvent) {
        return true;
    }

    private PartitionToProcessedObjectWithStat binlogEventToProcessedObject(BinlogEventWithOffset unprocessedBinlogEvent) {
        var objectsSuitableByRule = mapBinlogEvent(unprocessedBinlogEvent.getEvent());
        var additionalObjects = additionalObjectsHandler.processAdditionalObjects(unprocessedBinlogEvent.getEvent());

        List<T> logicObjects = new ArrayList<>(objectsSuitableByRule);
        logicObjects.addAll(additionalObjects.getLogicObjects());

        ProcessedObject processedObject;
        if (!logicObjects.isEmpty()) {
            logger.debug("Binlog event with seqNo {}, partition {}, reqId {}, gtid {}, has been proceeded by rule {}",
                    unprocessedBinlogEvent.getSeqNo(), unprocessedBinlogEvent.getPartition(),
                    unprocessedBinlogEvent.getEvent().getTraceInfoReqId(), unprocessedBinlogEvent.getEvent().getGtid(),
                    this.getClass().getSimpleName());
            processedObject = new ProcessedObject(unprocessedBinlogEvent.getSeqNo(),
                    serializeEvent(unprocessedBinlogEvent, logicObjects));


        } else {
            processedObject = null;
        }

        var partitionToProcessedObjectWithStat =
                new PartitionToProcessedObjectWithStat(unprocessedBinlogEvent.getPartition(), processedObject);
        partitionToProcessedObjectWithStat.setAdditionalObjectsCnt(additionalObjects.getLogicObjects().size());
        partitionToProcessedObjectWithStat.setAdditionalSkippedCnt(additionalObjects.getSkippedObjectsCnt());
        partitionToProcessedObjectWithStat.setProcessedObjectsCnt(objectsSuitableByRule.size());
        return partitionToProcessedObjectWithStat;
    }

    private PartitionToProcessedObjectWithStat watchlogEventToProcessedObject(WatchlogEvent unprocessedWatchlogEvent) {
        var logicObjects = mapWatchlogEvent(unprocessedWatchlogEvent.getEvent());

        ProcessedObject processedObject;
        if (!logicObjects.isEmpty()) {
            processedObject = new ProcessedObject(unprocessedWatchlogEvent.getSeqNo(),
                    serializeWatchlogEvent(unprocessedWatchlogEvent, logicObjects));
        } else {
            processedObject = null;
        }

        var partitionToProcessedObjectWithStat =
                new PartitionToProcessedObjectWithStat(unprocessedWatchlogEvent.getPartition(), processedObject);
        partitionToProcessedObjectWithStat.setProcessedObjectsCnt(logicObjects.size());
        return partitionToProcessedObjectWithStat;
    }

    private byte[] serializeEvent(BinlogEventWithOffset binlogEvent, List<T> logicObjects) {

        LogicObjectListWithInfo<T> logicObjectListWithInfo = new LogicObjectListWithInfo.Builder<T>()
                .withReqId(binlogEvent.getEvent().getTraceInfoReqId())
                .withUtcTimestamp(binlogEvent.getEvent().getUtcTimestamp().toEpochSecond(ZoneOffset.UTC))
                .withGtid(binlogEvent.getEvent().getGtid())
                .withSeqNo(binlogEvent.getSeqNo())
                .withSource(binlogEvent.getEvent().getSource())
                .withLogicObjects(logicObjects)
                .build();

        return logicObjectWithSystemInfoConverter.toJson(logicObjectListWithInfo);
    }

    private byte[] serializeWatchlogEvent(WatchlogEvent watchlogEvent, List<T> logicObjects) {
        LogicObjectListWithInfo<T> logicObjectListWithInfo = new LogicObjectListWithInfo.Builder<T>()
                .withUtcTimestamp(getUtcTimestamp(watchlogEvent))
                .withLogicObjects(logicObjects)
                .build();

        return logicObjectWithSystemInfoConverter.toJson(logicObjectListWithInfo);
    }

    private RuleProcessingStat getRuleProcessingStat(PartitionToProcessedObjectWithStat partitionToProcessedObjectWithStat) {
        return new RuleProcessingStat.Builder()
                .setAdditionalObjectsCnt(partitionToProcessedObjectWithStat.additionalObjectsCnt)
                .setAdditionalSkippedCnt(partitionToProcessedObjectWithStat.additionalSkippedCnt)
                .setProcessedObjectsCnt(partitionToProcessedObjectWithStat.processedObjectsCnt)
                .build();
    }

    private RuleProcessingStat mergeStats(RuleProcessingStat ruleProcessingStat1,
                                          RuleProcessingStat ruleProcessingStat2) {
        return new RuleProcessingStat.Builder()
                .setProcessedObjectsCnt(ruleProcessingStat1.getProcessedObjectsCnt() + ruleProcessingStat2.getAdditionalObjectsCnt())
                .setAdditionalSkippedCnt(ruleProcessingStat1.getAdditionalSkippedCnt() + ruleProcessingStat2.getAdditionalSkippedCnt())
                .setAdditionalObjectsCnt(ruleProcessingStat1.getAdditionalObjectsCnt() + ruleProcessingStat2.getAdditionalObjectsCnt())
                .build();
    }

    public String getLogicProcessName() {
        return logicProcessName;
    }

    public Class<? extends Condition> getRunConditionClass() {
        return runConditionClass;
    }

    public String getTopic() {
        return topic;
    }

    private class PartitionToProcessedObjectWithStat {

        int partition;
        ProcessedObject processedObjects;

        long processedObjectsCnt;
        long additionalObjectsCnt;
        long additionalSkippedCnt;

        PartitionToProcessedObjectWithStat(int partition, ProcessedObject processedObjects) {
            this.partition = partition;
            this.processedObjects = processedObjects;
        }

        int getPartition() {
            return partition;
        }

        ProcessedObject getProcessedObjects() {
            return processedObjects;
        }

        public void setProcessedObjectsCnt(long processedObjectsCnt) {
            this.processedObjectsCnt = processedObjectsCnt;
        }

        public void setAdditionalObjectsCnt(long additionalObjectsCnt) {
            this.additionalObjectsCnt = additionalObjectsCnt;
        }

        public void setAdditionalSkippedCnt(long additionalSkippedCnt) {
            this.additionalSkippedCnt = additionalSkippedCnt;
        }
    }
}
